-- WordPress Management System Database Schema
-- Date: 2025-01-18
-- Description: Adds tables for WordPress site management, WP-CLI containers, and plugin recommendations

USE whp;

-- Table to track which container images support WP-CLI
CREATE TABLE IF NOT EXISTS wp_cli_containers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    container_image VARCHAR(255) NOT NULL UNIQUE,
    description VARCHAR(500),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table to track WordPress installations
CREATE TABLE IF NOT EXISTS wordpress_sites (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user VARCHAR(50) NOT NULL,
    domain VARCHAR(255) NOT NULL,
    container_name VARCHAR(255) NOT NULL,
    site_path VARCHAR(500) DEFAULT '/home/{user}/public_html/',
    wp_version VARCHAR(20),
    is_multisite BOOLEAN DEFAULT FALSE,
    is_staging BOOLEAN DEFAULT FALSE,
    parent_site_id INT DEFAULT NULL, -- For staging sites
    auto_update_core BOOLEAN DEFAULT FALSE,
    auto_update_plugins BOOLEAN DEFAULT FALSE,
    auto_update_themes BOOLEAN DEFAULT FALSE,
    auto_backup_before_update BOOLEAN DEFAULT TRUE,
    admin_email VARCHAR(255),
    admin_username VARCHAR(60),
    status ENUM('active', 'inactive', 'maintenance', 'error') DEFAULT 'active',
    last_health_check TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (parent_site_id) REFERENCES wordpress_sites(id) ON DELETE SET NULL,
    UNIQUE KEY unique_domain (domain),
    INDEX idx_user (user),
    INDEX idx_status (status),
    INDEX idx_container (container_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for recommended plugins (managed by root)
CREATE TABLE IF NOT EXISTS wp_recommended_plugins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(200) NOT NULL UNIQUE,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    category VARCHAR(100),
    install_url VARCHAR(500), -- WordPress.org slug or custom URL
    is_premium BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (is_active),
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table to track installed plugins per site
CREATE TABLE IF NOT EXISTS wp_site_plugins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    plugin_slug VARCHAR(200) NOT NULL,
    plugin_name VARCHAR(255),
    version VARCHAR(20),
    is_active BOOLEAN DEFAULT FALSE,
    auto_update BOOLEAN DEFAULT FALSE,
    last_updated TIMESTAMP NULL,
    FOREIGN KEY (site_id) REFERENCES wordpress_sites(id) ON DELETE CASCADE,
    UNIQUE KEY unique_site_plugin (site_id, plugin_slug),
    INDEX idx_site (site_id),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table to track WordPress update history
CREATE TABLE IF NOT EXISTS wp_update_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    update_type ENUM('core', 'plugin', 'theme') NOT NULL,
    item_name VARCHAR(255) NOT NULL,
    old_version VARCHAR(20),
    new_version VARCHAR(20),
    status ENUM('pending', 'in_progress', 'completed', 'failed', 'rolled_back') NOT NULL DEFAULT 'pending',
    backup_id INT DEFAULT NULL, -- Reference to backup_history table
    error_message TEXT,
    started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (site_id) REFERENCES wordpress_sites(id) ON DELETE CASCADE,
    FOREIGN KEY (backup_id) REFERENCES backup_history(id) ON DELETE SET NULL,
    INDEX idx_site (site_id),
    INDEX idx_status (status),
    INDEX idx_started (started_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for WordPress security settings
CREATE TABLE IF NOT EXISTS wp_security_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL UNIQUE,
    disable_file_editor BOOLEAN DEFAULT TRUE,
    disable_xmlrpc BOOLEAN DEFAULT TRUE,
    hide_wp_version BOOLEAN DEFAULT TRUE,
    disable_directory_browsing BOOLEAN DEFAULT TRUE,
    block_php_in_uploads BOOLEAN DEFAULT TRUE,
    limit_login_attempts BOOLEAN DEFAULT TRUE,
    change_login_url VARCHAR(50) DEFAULT NULL,
    database_prefix VARCHAR(20) DEFAULT 'wp_',
    last_security_scan TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES wordpress_sites(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for one-time login tokens
CREATE TABLE IF NOT EXISTS wp_login_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    token VARCHAR(64) NOT NULL UNIQUE,
    admin_username VARCHAR(60) NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    used BOOLEAN DEFAULT FALSE,
    used_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES wordpress_sites(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Note: WordPress permissions will be handled by the permission_checker.php system
-- For now, WordPress management is available to all authenticated users

-- Add some default recommended plugins
INSERT INTO wp_recommended_plugins (slug, name, description, category, install_url, sort_order) VALUES
    ('wordfence', 'Wordfence Security', 'Comprehensive security plugin with firewall and malware scanner', 'Security', 'wordfence', 10),
    ('updraftplus', 'UpdraftPlus Backup', 'Complete backup solution with remote storage options', 'Backup', 'updraftplus', 20),
    ('wp-optimize', 'WP-Optimize', 'Database optimization and caching plugin', 'Performance', 'wp-optimize', 30),
    ('contact-form-7', 'Contact Form 7', 'Simple but flexible contact form plugin', 'Forms', 'contact-form-7', 40),
    ('really-simple-ssl', 'Really Simple SSL', 'Lightweight SSL and security hardening plugin', 'Security', 'really-simple-ssl', 50)
ON DUPLICATE KEY UPDATE description = VALUES(description);

-- Rollback procedure
DELIMITER //
CREATE PROCEDURE IF NOT EXISTS rollback_wordpress_management()
BEGIN
    -- Drop tables in correct order
    DROP TABLE IF EXISTS wp_login_tokens;
    DROP TABLE IF EXISTS wp_security_settings;
    DROP TABLE IF EXISTS wp_update_history;
    DROP TABLE IF EXISTS wp_site_plugins;
    DROP TABLE IF EXISTS wp_recommended_plugins;
    DROP TABLE IF EXISTS wordpress_sites;
    DROP TABLE IF EXISTS wp_cli_containers;
END//
DELIMITER ;