#!/bin/bash
# Test script for PostgreSQL integration

echo "==============================================="
echo "WHP PostgreSQL Integration Test"
echo "==============================================="

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo "Please run as root"
    exit 1
fi

echo "1. Checking PostgreSQL PHP module..."
php -m | grep pgsql >/dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "✓ PostgreSQL PHP module is installed"
    php -m | grep pdo_pgsql >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        echo "✓ PDO PostgreSQL driver is available"
    else
        echo "✗ PDO PostgreSQL driver is not available"
    fi
else
    echo "✗ PostgreSQL PHP module is not installed"
    echo "  Run: dnf install -y php-pgsql && systemctl restart php-fpm"
fi

echo ""
echo "2. Checking PostgreSQL container..."
docker ps | grep postgresql >/dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "✓ PostgreSQL container is running"
    
    # Test connection
    docker exec postgresql pg_isready -U postgres >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        echo "✓ PostgreSQL is accepting connections"
    else
        echo "✗ PostgreSQL is not accepting connections"
    fi
else
    echo "✗ PostgreSQL container is not running"
    echo "  Container should be started during setup or update"
fi

echo ""
echo "3. Checking pgAdmin installation..."
if [ -f /usr/pgadmin4/web/pgAdmin4.wsgi ]; then
    echo "✓ pgAdmin 4 is installed"
    
    if [ -f /etc/httpd/conf.d/pgadmin4.conf ]; then
        echo "✓ pgAdmin Apache configuration exists"
    else
        echo "✗ pgAdmin Apache configuration missing"
    fi
else
    echo "✗ pgAdmin 4 is not installed"
fi

echo ""
echo "4. Checking WHP files..."
if [ -f /docker/whp/web/libs/pgsqlmgmt.php ]; then
    echo "✓ PostgreSQL management library exists"
else
    echo "✗ PostgreSQL management library missing"
fi

if [ -f /docker/whp/web/pages/postgresql-management.php ]; then
    echo "✓ PostgreSQL management page exists"
else
    echo "✗ PostgreSQL management page missing"
fi

if [ -f /docker/whp/web/pgadmin-sso.php ]; then
    echo "✓ pgAdmin SSO page exists"
else
    echo "✗ pgAdmin SSO page missing"
fi

echo ""
echo "5. Checking database tables..."
mysql -e "USE whp; SHOW TABLES LIKE 'postgresql_%';" 2>/dev/null | grep -q postgresql
if [ $? -eq 0 ]; then
    echo "✓ PostgreSQL tracking tables exist"
    mysql -e "USE whp; SELECT COUNT(*) as table_count FROM information_schema.TABLES WHERE TABLE_SCHEMA = 'whp' AND TABLE_NAME LIKE 'postgresql_%';" 2>/dev/null
else
    echo "✗ PostgreSQL tracking tables missing"
    echo "  Run the migration: mysql < /root/whp/sql/migrations/staging/001_postgresql_support.sql"
fi

echo ""
echo "6. Checking configuration..."
if [ -f /root/.pgpass ]; then
    echo "✓ PostgreSQL password file exists"
else
    echo "✗ PostgreSQL password file missing (/root/.pgpass)"
fi

# Test PHP PostgreSQL connection
echo ""
echo "7. Testing PHP PostgreSQL connection..."
if [ -f /root/.pgpass ]; then
    pg_password=$(cut -d: -f5 /root/.pgpass 2>/dev/null)
    if [ ! -z "$pg_password" ]; then
        php -r "
        try {
            \$pdo = new PDO('pgsql:host=127.0.0.1;port=5432;dbname=postgres', 'postgres', '$pg_password');
            echo '✓ PHP can connect to PostgreSQL\n';
        } catch (Exception \$e) {
            echo '✗ PHP cannot connect to PostgreSQL: ' . \$e->getMessage() . '\n';
        }
        " 2>/dev/null
    else
        echo "✗ Cannot read PostgreSQL password from .pgpass"
    fi
else
    echo "✗ No .pgpass file found"
fi

echo ""
echo "==============================================="
echo "PostgreSQL Integration Test Complete"
echo "==============================================="