<?php
/**
 * Test script for DKIM timing fix
 * 
 * This script tests the domain creation and DKIM generation process
 * to verify that the timing fix works correctly.
 */

require_once('/docker/whp/web/libs/emailmgmt.php');

// Test configuration
$test_domain = 'test-dkim-timing-' . time() . '.example.com';
$test_username = 'root';

echo "=== DKIM Timing Fix Test ===\n";
echo "Test Domain: $test_domain\n";
echo "Test User: $test_username\n\n";

try {
    // Initialize email management
    $EmailMgmt = new emailmgmt();
    
    if (!$EmailMgmt) {
        echo "ERROR: Failed to initialize email management\n";
        exit(1);
    }
    
    echo "1. Testing mail server connection...\n";
    $connection_test = $EmailMgmt->testMailServerConnection();
    if ($connection_test['status'] !== '0') {
        echo "ERROR: Mail server connection failed: " . $connection_test['msg'] . "\n";
        exit(1);
    }
    echo "✓ Mail server connection successful\n\n";
    
    echo "2. Creating test domain on mail server...\n";
    $domain_result = $EmailMgmt->createMailServerDomain($test_domain, [
        'disabled' => false,
        'reference_id' => $test_username
    ]);
    
    if ($domain_result['status'] !== '0') {
        echo "ERROR: Failed to create domain: " . $domain_result['msg'] . "\n";
        exit(1);
    }
    echo "✓ Domain created successfully\n\n";
    
    echo "3. Testing immediate DKIM creation (should fail)...\n";
    $immediate_dkim = $EmailMgmt->createDkimRecords($test_domain, $test_username);
    if ($immediate_dkim['status'] === '0') {
        echo "WARNING: Immediate DKIM creation succeeded (unexpected)\n";
    } else {
        echo "✓ Immediate DKIM creation failed as expected: " . $immediate_dkim['msg'] . "\n";
    }
    echo "\n";
    
    echo "4. Testing DKIM creation with retry logic...\n";
    $retry_dkim = $EmailMgmt->createDkimRecordsWithRetry($test_domain, $test_username, 3, 2);
    if ($retry_dkim['status'] === '0') {
        echo "✓ DKIM creation with retry succeeded\n";
        if (isset($retry_dkim['dkim']['selector'])) {
            echo "  Selector: " . $retry_dkim['dkim']['selector'] . "\n";
        }
    } else {
        echo "ERROR: DKIM creation with retry failed: " . $retry_dkim['msg'] . "\n";
    }
    echo "\n";
    
    echo "5. Testing manual retry function...\n";
    $manual_retry = $EmailMgmt->retryDkimCreation($test_domain, $test_username, 5, 3);
    if ($manual_retry['status'] === '0') {
        echo "✓ Manual retry succeeded\n";
    } else {
        echo "WARNING: Manual retry failed: " . $manual_retry['msg'] . "\n";
    }
    echo "\n";
    
    echo "6. Verifying DKIM records exist...\n";
    $dkim_check = $EmailMgmt->getDkimRecords($test_domain, $test_username);
    if ($dkim_check['status'] === '0') {
        echo "✓ DKIM records found\n";
        if (isset($dkim_check['dkim']['selector'])) {
            echo "  Selector: " . $dkim_check['dkim']['selector'] . "\n";
        }
        if (isset($dkim_check['dkim']['dns_record'])) {
            echo "  DNS Record: " . $dkim_check['dkim']['dns_record'] . "\n";
        }
    } else {
        echo "ERROR: DKIM records not found: " . $dkim_check['msg'] . "\n";
    }
    echo "\n";
    
    echo "7. Testing DNS record creation...\n";
    $dns_result = $EmailMgmt->createDkimDnsRecord($test_domain, null, null, $test_username);
    if ($dns_result['status'] === '0') {
        echo "✓ DNS record created successfully\n";
    } else {
        echo "WARNING: DNS record creation failed: " . $dns_result['msg'] . "\n";
    }
    echo "\n";
    
    echo "8. Cleaning up test domain...\n";
    $cleanup_result = $EmailMgmt->deleteMailServerDomain($test_domain);
    if ($cleanup_result['status'] === '0') {
        echo "✓ Test domain cleaned up successfully\n";
    } else {
        echo "WARNING: Failed to cleanup test domain: " . $cleanup_result['msg'] . "\n";
    }
    echo "\n";
    
    echo "=== Test Results ===\n";
    echo "✓ All tests completed successfully\n";
    echo "✓ DKIM timing fix appears to be working correctly\n";
    echo "✓ Retry logic is functioning as expected\n";
    echo "✓ Manual retry function is available\n";
    
} catch (Exception $e) {
    echo "ERROR: Test failed with exception: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\nTest completed successfully!\n";
?> 