<?php
// TODO: The .form-text under form boxes is hard to read on dark backgrounds. Consider using .text-light or a custom color for better contrast.
// Permission checking is handled by the permission system in auto-prepend.php

require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('/docker/whp/web/libs/docker-api.php');
require_once('/docker/whp/web/libs/pdns-integration.php');

$message = '';
$message_type = '';
$is_root = (defined('AUTH_USER') && AUTH_USER === 'root');

// Initialize database connection
$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();

if (!$db) {
    $message = 'Database connection failed';
    $message_type = 'danger';
} else {
    // Initialize Docker API
    $DockerAPI = new docker_api();
    // Initialize PowerDNS Integration
    $PowerDNS = new PowerDNSIntegration($db);
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_site':
                if (!empty($_POST['site_name']) && !empty($_POST['primary_domain_id']) && !empty($_POST['container_type_id'])) {
                    try {
                        $site_name = trim($_POST['site_name']);
                        $primary_fqdn = trim($_POST['primary_domain_id']);
                        $container_type_id = intval($_POST['container_type_id']);
                        $container_count = intval($_POST['container_count'] ?? 1);
                        $cpu_per_container = floatval($_POST['cpu_per_container'] ?? 0.25);
                        $memory_per_container = intval($_POST['memory_per_container'] ?? 256);
                        
                        // Validate container count
                        if ($container_count < 1 || $container_count > 10) {
                            $message = 'Container count must be between 1 and 10';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Get domain info and check ownership
                        $primary_domain_row = null;
                        $all_domains = $db->query("SELECT id, domain_name, username FROM whp.domains WHERE active=1")->fetchAll(PDO::FETCH_ASSOC);
                        foreach ($all_domains as $row) {
                            if ($primary_fqdn === $row['domain_name'] || (str_ends_with($primary_fqdn, '.' . $row['domain_name']))) {
                                $primary_domain_row = $row;
                                break;
                            }
                        }
                        if (!$primary_domain_row) {
                            $message = 'Primary domain/subdomain not found or not owned by you.';
                            $message_type = 'danger';
                            break;
                        }
                        if (!$is_root && $primary_domain_row['username'] !== AUTH_USER) {
                            $message = 'You can only create sites with your own domains/subdomains';
                            $message_type = 'danger';
                            break;
                        }
                        // Validate FQDN is a valid A or CNAME for this domain
                        $valid_fqdn = false;
                        $fqdn_records = $PowerDNS->getRecords($primary_domain_row['domain_name']);
                        foreach ($fqdn_records as $rec) {
                            if (strcasecmp($rec['name'], $primary_fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                                $valid_fqdn = true;
                                break;
                            }
                        }
                        if (!$valid_fqdn) {
                            $message = 'Selected primary FQDN is not a valid A or CNAME record.';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Get container type info
                        $stmt = $db->prepare("SELECT * FROM whp.container_types WHERE id = ? AND active = 1");
                        $stmt->execute([$container_type_id]);
                        $container_type = $stmt->fetch();
                        
                        if (!$container_type) {
                            $message = 'Invalid or inactive container type';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Validate resource requirements
                        if ($cpu_per_container < $container_type['min_cpu']) {
                            $message = 'CPU per container must be at least ' . $container_type['min_cpu'] . ' cores';
                            $message_type = 'danger';
                            break;
                        }
                        
                        if ($memory_per_container < $container_type['min_memory']) {
                            $message = 'Memory per container must be at least ' . $container_type['min_memory'] . ' MB';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check user resource limits
                        $stmt = $db->prepare("SELECT max_cpu_allowance, max_memory_allowance FROM whp.client_allowances WHERE username = ?");
                        $stmt->execute([$primary_domain_row['username']]);
                        $user_limits = $stmt->fetch();
                        
                        if ($user_limits) {
                            // Get current resource usage for the user (all existing sites/containers)
                            $usage = $MySQLMgmt->getUserResourceUsage($primary_domain_row['username']);
                            $current_cpu_used = floatval($usage['cpu_used'] ?? 0);
                            $current_memory_used = intval($usage['memory_used'] ?? 0);
                            // Calculate total resources needed for this new site
                            $new_site_cpu = $cpu_per_container * $container_count;
                            $new_site_memory = $memory_per_container * $container_count;
                            // Calculate projected totals if this site is created
                            $projected_cpu = $current_cpu_used + $new_site_cpu;
                            $projected_memory = $current_memory_used + $new_site_memory;
                            // Enforce total resource limits
                            if ($projected_cpu > $user_limits['max_cpu_allowance']) {
                                $message = 'Total CPU allocation after this site (' . $projected_cpu . ' cores) exceeds your limit (' . $user_limits['max_cpu_allowance'] . ' cores)';
                                $message_type = 'danger';
                                break;
                            }
                            if ($projected_memory > $user_limits['max_memory_allowance']) {
                                $message = 'Total memory allocation after this site (' . $projected_memory . ' MB) exceeds your limit (' . $user_limits['max_memory_allowance'] . ' MB)';
                                $message_type = 'danger';
                                break;
                            }
                        }
                        
                        // Process user variables if provided
                        $user_variables = [];
                        if (!empty($_POST['user_variables'])) {
                            $user_variables = json_decode($_POST['user_variables'], true) ?: [];
                        }
                        
                        // Debug: Log user variables for troubleshooting
                        error_log("User variables received: " . json_encode($user_variables));
                        error_log("Raw POST user_variables: " . ($_POST['user_variables'] ?? 'not set'));
                        
                        // Create site directory using actual FQDN for subdomain support
                        $site_dir = '/docker/users/' . $primary_domain_row['username'] . '/' . $primary_fqdn;
                        if (!is_dir($site_dir)) {
                            if (!mkdir($site_dir, 0755, true)) {
                                $message = 'Failed to create site directory';
                                $message_type = 'danger';
                                break;
                            }
                            
                            // Set ownership
                            $uid = trim(shell_exec("id -u " . escapeshellarg($primary_domain_row['username'])));
                            $gid = trim(shell_exec("id -g " . escapeshellarg($primary_domain_row['username'])));
                            if (is_numeric($uid) && is_numeric($gid)) {
                                shell_exec("chown -R {$uid}:{$gid} " . escapeshellarg($site_dir));
                            }
                        }
                        
                        // Get auto-recreation setting
                        $auto_recreate_on_update = isset($_POST['auto_recreate_on_update']) ? 1 : 0;
                        
                        // Insert site into database (use parent domain_id for primary_domain_id)
                        $stmt = $db->prepare("
                            INSERT INTO whp.sites (site_name, primary_domain_id, username, container_type_id, container_count, cpu_per_container, memory_per_container, user_variables, auto_recreate_on_update) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ");
                        $stmt->execute([
                            $site_name,
                            $primary_domain_row['id'],
                            $primary_domain_row['username'],
                            $container_type_id,
                            $container_count,
                            $cpu_per_container,
                            $memory_per_container,
                            json_encode($user_variables),
                            $auto_recreate_on_update
                        ]);
                        
                        $site_id = $db->lastInsertId();
                        
                        // Link primary FQDN to site
                        $stmt = $db->prepare("
                            INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) 
                            VALUES (?, ?, ?, 1)
                        ");
                        $stmt->execute([$site_id, $primary_domain_row['id'], $primary_fqdn]);
                        
                        // Store the primary FQDN for container naming (prevents conflicts with apex domains)
                        $primary_fqdn_for_containers = $primary_fqdn;
                        
                        // Link additional FQDNs to site
                        $additional_domains_added = [];
                        if (!empty($_POST['additional_domains']) && is_array($_POST['additional_domains'])) {
                            foreach ($_POST['additional_domains'] as $fqdn) {
                                $fqdn = trim($fqdn);
                                if ($fqdn === $primary_fqdn) continue;
                                $parent_row = null;
                                foreach ($all_domains as $row) {
                                    if ($fqdn === $row['domain_name'] || (str_ends_with($fqdn, '.' . $row['domain_name']))) {
                                        $parent_row = $row;
                                        break;
                                    }
                                }
                                if (!$parent_row) continue;
                                if (!$is_root && $parent_row['username'] !== AUTH_USER) continue;
                                // Validate FQDN is a valid A or CNAME for this domain
                                $valid_fqdn = false;
                                $fqdn_records = $PowerDNS->getRecords($parent_row['domain_name']);
                                foreach ($fqdn_records as $rec) {
                                    if (strcasecmp($rec['name'], $fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                                        $valid_fqdn = true;
                                        break;
                                    }
                                }
                                if (!$valid_fqdn) continue;
                                // Check if already linked
                                $stmt = $db->prepare("SELECT id FROM whp.site_domains WHERE site_id = ? AND domain_id = ? AND fqdn = ?");
                                $stmt->execute([$site_id, $parent_row['id'], $fqdn]);
                                if ($stmt->fetch()) continue;
                                // Link additional FQDN to site
                                $stmt = $db->prepare("
                                    INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) 
                                    VALUES (?, ?, ?, 0)
                                ");
                                $stmt->execute([$site_id, $parent_row['id'], $fqdn]);
                                $additional_domains_added[] = $fqdn;
                            }
                        }
                        
                        // Create and start Docker containers for the site
                        $container_creation_success = true;
                        $container_errors = [];
                        
                        try {
                            // Get container type details for Docker creation
                            $stmt = $db->prepare("
                                SELECT ct.base_image, ct.startup_env, ct.mount_options, ct.listen_port 
                                FROM whp.container_types ct 
                                WHERE ct.id = ?
                            ");
                            $stmt->execute([$container_type_id]);
                            $container_type_details = $stmt->fetch();
                            
                            // Get user UID for container configuration
                            $uid = trim(shell_exec("id -u " . escapeshellarg($primary_domain_row['username'])));
                            if (!is_numeric($uid)) {
                                throw new Exception("Invalid user UID");
                            }
                            
                            // Create containers
                            for ($i = 1; $i <= $container_count; $i++) {
                                // Create container name using actual FQDN to prevent conflicts with apex domains
                                $container_name = $primary_fqdn_for_containers . '-' . str_pad($i, 2, '0', STR_PAD_LEFT);
                                
                                // Prepare environment variables
                                $env_vars = [
                                    'WHP_USER=' . $primary_domain_row['username'],
                                    'WHP_UID=' . $uid,
                                    'WHP_DOMAIN=' . $primary_fqdn,
                                    'WHP_CONTAINER_NUMBER=' . $i
                                ];
                                
                                // Add user variables
                                if (!empty($user_variables)) {
                                    foreach ($user_variables as $var_name => $var_value) {
                                        $env_vars[] = $var_name . '=' . $var_value;
                                    }
                                    error_log("Added user variables to container $i: " . json_encode($user_variables));
                                } else {
                                    error_log("No user variables for container $i");
                                }
                                
                                // Process startup environment variables
                                if (!empty($container_type_details['startup_env'])) {
                                    $startup_env = json_decode($container_type_details['startup_env'], true);
                                    if ($startup_env) {
                                        foreach ($startup_env as $var_name => $var_value) {
                                            // Replace dynamic variables using centralized function with actual FQDN
                                            $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, $primary_domain_row['username'], $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                                            $env_vars[] = $var_name . '=' . $var_value;
                                        }
                                    }
                                }
                                
                                // Prepare volumes using actual FQDN path
                                $volumes = [
                                    '/docker/users/' . $primary_domain_row['username'] . '/' . $primary_fqdn . ':/var/www/html'
                                ];
                                
                                // Process mount options
                                if (!empty($container_type_details['mount_options'])) {
                                    $mount_options = json_decode($container_type_details['mount_options'], true);
                                    if ($mount_options && isset($mount_options['volumes'])) {
                                        foreach ($mount_options['volumes'] as $volume) {
                                            // Replace dynamic variables in both source and target paths using actual FQDN
                                            $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], $primary_domain_row['username'], $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                                            $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], $primary_domain_row['username'], $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                                            $volumes[] = $source . ':' . $target;
                                        }
                                    }
                                }
                                
                                // Create container config
                                $container_config = [
                                    'image' => $container_type_details['base_image'],
                                    'name' => $container_name,
                                    'labels' => [
                                        'whp.user' => $primary_domain_row['username'],
                                        'whp.domain' => $primary_fqdn,
                                        'whp.site_id' => strval($site_id),
                                        'whp.container_number' => strval($i)
                                    ],
                                    'env' => $env_vars,
                                    'volumes' => $volumes,
                                    'cpu' => $cpu_per_container,
                                    'memory' => $memory_per_container,
                                    'network' => 'client-net'
                                ];
                                
                                // Debug: Log container config for troubleshooting
                                error_log("Container config for container $i: " . json_encode($container_config));
                                
                                $result = $DockerAPI->create_container($container_config);
                                
                                // Debug: Log Docker API result
                                error_log("Docker API result for container $i: " . json_encode($result));
                                
                                if ($result['success']) {
                                    // Save container info to database
                                    $stmt = $db->prepare("
                                        INSERT INTO whp.site_containers (site_id, container_id, container_number, status) 
                                        VALUES (?, ?, ?, 'running')
                                    ");
                                    $stmt->execute([$site_id, $result['container_id'], $i]);
                                } else {
                                    $container_creation_success = false;
                                    $container_errors[] = "Failed to create container $i: " . $result['error'];
                                }
                            }
                        } catch (Exception $e) {
                            $container_creation_success = false;
                            $container_errors[] = "Container creation error: " . $e->getMessage();
                        }
                        
                        // Configure HAProxy if containers were created successfully
                        if ($container_creation_success) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, [
                                'username' => $primary_domain_row['username'],
                                'primary_domain' => $primary_domain_row['domain_name'],
                                'container_type_id' => $container_type_id
                            ]);
                            
                            if (!$haproxy_result['success']) {
                                $container_creation_success = false;
                                $container_errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        

                        
                        // Enable SSL if requested
                        $ssl_errors = [];
                        if (isset($_POST['enable_ssl']) && $_POST['enable_ssl'] == '1' && $container_creation_success) {
                            // Get all domains for this site (including primary domain)
                            $stmt = $db->prepare("
                                SELECT d.domain_name 
                                FROM whp.site_domains sd
                                JOIN whp.domains d ON sd.domain_id = d.id
                                WHERE sd.site_id = ?
                            ");
                            $stmt->execute([$site_id]);
                            $site_domains = $stmt->fetchAll(PDO::FETCH_COLUMN);
                            
                            // Add primary domain if not already in list
                            if (!in_array($primary_domain_row['domain_name'], $site_domains)) {
                                $site_domains[] = $primary_domain_row['domain_name'];
                            }
                            
                            // Use DNS validator for robust validation
                            require_once('/docker/whp/web/libs/dns_validator.php');
                            $dns_validator = new dns_validator();
                            $validation_results = $dns_validator->validateDomains($site_domains);
                            
                            if (!$validation_results['all_valid']) {
                                $invalid_domains = [];
                                foreach ($validation_results['domains'] as $site_domain => $result) {
                                    if (!$result['valid']) {
                                        $invalid_domains[] = $site_domain;
                                    }
                                }
                                $ssl_errors[] = 'SSL not enabled: The following domains are not pointing to our server: ' . implode(', ', $invalid_domains) . '. Please configure DNS to point to ' . $validation_results['server_ip'] . ' before enabling SSL.';
                            } else {
                                // Enable SSL for all domains (primary and additional)
                                require_once('/docker/whp/web/libs/ssl_manager.php');
                                $ssl_manager = new ssl_manager();
                                $ssl_result = $ssl_manager->enable_ssl($site_id, 'letsencrypt');
                                
                                if (!$ssl_result['success']) {
                                    $ssl_errors[] = 'SSL configuration failed: ' . $ssl_result['error'];
                                }
                            }
                        }
                        
                        if ($container_creation_success) {
                            $domain_info = '';
                            if (!empty($additional_domains_added)) {
                                $domain_info = ' with additional domains: ' . implode(', ', $additional_domains_added);
                            }
                            
                            // Check if SSL was requested and determine appropriate message
                            $ssl_requested = isset($_POST['enable_ssl']) && $_POST['enable_ssl'] == '1';
                            
                            if ($ssl_requested && empty($ssl_errors)) {
                                $message = 'Site created successfully with ' . $container_count . ' container(s) started, HAProxy configured, and SSL enabled: ' . $site_name . $domain_info;
                                $message_type = 'success';
                            } else if ($ssl_requested && !empty($ssl_errors)) {
                                $message = 'Site created successfully with ' . $container_count . ' container(s) started and HAProxy configured: ' . $site_name . $domain_info . '. However, SSL issues: ' . implode(', ', $ssl_errors);
                                $message_type = 'warning';
                            } else {
                                $message = 'Site created successfully with ' . $container_count . ' container(s) started and HAProxy configured: ' . $site_name . $domain_info;
                                $message_type = 'success';
                            }
                            
                            
                        } else {
                            $message = 'Site created successfully: ' . $site_name . '. However, some operations failed: ' . implode(', ', $container_errors);
                            $message_type = 'warning';
                            
                        }
                    } catch (PDOException $e) {
                        $message = 'Error creating site: ' . $e->getMessage();
                        $message_type = 'danger';
                        
                    }
                }
                break;
                
            case 'delete_site':
                if (!empty($_POST['site_id'])) {
                    try {
                        $site_id = intval($_POST['site_id']);
                        
                        // Get site info
                        $stmt = $db->prepare("SELECT s.*, d.domain_name FROM whp.sites s JOIN whp.domains d ON s.primary_domain_id = d.id WHERE s.id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            $message = 'Site not found';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check ownership
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            $message = 'You can only delete your own sites';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Stop and remove all containers for this site
                        require_once('/docker/whp/web/libs/docker-api.php');
                        $DockerAPI = new docker_api();
                        
                        $stmt = $db->prepare("SELECT container_id FROM whp.site_containers WHERE site_id = ?");
                        $stmt->execute([$site_id]);
                        $containers = $stmt->fetchAll();
                        
                        $container_errors = [];
                        foreach ($containers as $container) {
                            // Stop container first
                            $stop_result = $DockerAPI->stop_container($container['container_id']);
                            if (!$stop_result['success']) {
                                $container_errors[] = "Failed to stop container " . $container['container_id'] . ": " . $stop_result['error'];
                            }
                            
                            // Remove container
                            $remove_result = $DockerAPI->remove_container($container['container_id']);
                            if (!$remove_result['success']) {
                                $container_errors[] = "Failed to remove container " . $container['container_id'] . ": " . $remove_result['error'];
                            }
                        }
                        
                        // Remove HAProxy configuration
                        require_once('/docker/whp/web/libs/haproxy_manager.php');
                        $haproxy_manager = new haproxy_manager();
                        $haproxy_result = $haproxy_manager->removeSite($site_id);
                        
                        if (!$haproxy_result['success']) {
                            $container_errors[] = "HAProxy cleanup failed: " . $haproxy_result['error'];
                        }
                        
                        // Delete site (cascades to site_domains and site_containers)
                        $stmt = $db->prepare("DELETE FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        
                        if (empty($container_errors)) {
                            $message = 'Site deleted successfully: ' . $site['site_name'] . '. All containers were stopped and removed, and HAProxy configuration was cleaned up.';
                            $message_type = 'success';
                        } else {
                            $message = 'Site deleted successfully: ' . $site['site_name'] . '. However, some operations had errors: ' . implode(', ', $container_errors);
                            $message_type = 'warning';
                        }
                    } catch (PDOException $e) {
                        $message = 'Error deleting site: ' . $e->getMessage();
                        $message_type = 'danger';
                    }
                }
                break;
                
            case 'add_domain_to_site':
                if (!empty($_POST['site_id']) && !empty($_POST['domain_id'])) {
                    try {
                        $site_id = intval($_POST['site_id']);
                        $domain_id = intval($_POST['domain_id']);
                        
                        // Get site info
                        $stmt = $db->prepare("SELECT username FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            $message = 'Site not found';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check site ownership
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            $message = 'You can only modify your own sites';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Get domain info
                        $stmt = $db->prepare("SELECT domain_name, username FROM whp.domains WHERE id = ?");
                        $stmt->execute([$domain_id]);
                        $domain = $stmt->fetch();
                        
                        if (!$domain) {
                            $message = 'Domain not found';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check domain ownership
                        if (!$is_root && $domain['username'] !== AUTH_USER) {
                            $message = 'You can only add your own domains to sites';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check if domain is already linked to this site
                        $stmt = $db->prepare("SELECT id FROM whp.site_domains WHERE site_id = ? AND domain_id = ?");
                        $stmt->execute([$site_id, $domain_id]);
                        if ($stmt->fetch()) {
                            $message = 'Domain is already linked to this site';
                            $message_type = 'warning';
                            break;
                        }
                        
                        // Link domain to site
                        $stmt = $db->prepare("INSERT INTO whp.site_domains (site_id, domain_id, is_primary) VALUES (?, ?, 0)");
                        $stmt->execute([$site_id, $domain_id]);
                        
                        // Update HAProxy configuration for the site
                        require_once('/docker/whp/web/libs/haproxy_manager.php');
                        $haproxy_manager = new haproxy_manager();
                        $haproxy_result = $haproxy_manager->configureSite($site_id, [
                            'username' => $site['username'],
                            'domain_name' => $domain['domain_name']
                        ]);
                        
                        // Check if the site already has SSL enabled for other domains
                        $ssl_message = '';
                        if ($haproxy_result['success']) {
                            // Check if site has any existing SSL certificates
                            require_once('/docker/whp/web/libs/ssl_manager.php');
                            $ssl_manager = new ssl_manager();
                            
                            // Check if the site already has SSL enabled
                            $stmt = $db->prepare("
                                SELECT ssl_enabled
                                FROM whp.sites
                                WHERE id = ?
                            ");
                            $stmt->execute([$site_id]);
                            $ssl_result = $stmt->fetch();
                            
                            if ($ssl_result && $ssl_result['ssl_enabled'] > 0) {
                                // Site has SSL enabled for other domains, automatically enable for new domain
                                $auto_ssl_result = $ssl_manager->enable_ssl($site_id, 'letsencrypt');
                                
                                if ($auto_ssl_result['success']) {
                                    $ssl_message = ' SSL automatically enabled for new domain.';
                                } else {
                                    $ssl_message = ' SSL auto-enable failed: ' . ($auto_ssl_result['error'] ?? 'Unknown error');
                                }
                            }
                            
                            $message = 'Domain added to site successfully and HAProxy updated: ' . $domain['domain_name'] . $ssl_message;
                            $message_type = 'success';
                        } else {
                            $message = 'Domain added to site successfully: ' . $domain['domain_name'] . '. However, HAProxy configuration failed: ' . $haproxy_result['error'];
                            $message_type = 'warning';
                        }
                    } catch (PDOException $e) {
                        $message = 'Error adding domain to site: ' . $e->getMessage();
                        $message_type = 'danger';
                    }
                }
                break;
                
            case 'remove_domain_from_site':
                if (!empty($_POST['site_id']) && !empty($_POST['domain_id'])) {
                    try {
                        $site_id = intval($_POST['site_id']);
                        $domain_id = intval($_POST['domain_id']);
                        
                        // Get site info
                        $stmt = $db->prepare("SELECT username FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            $message = 'Site not found';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check site ownership
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            $message = 'You can only modify your own sites';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Check if this is the primary domain
                        $stmt = $db->prepare("SELECT is_primary FROM whp.site_domains WHERE site_id = ? AND domain_id = ?");
                        $stmt->execute([$site_id, $domain_id]);
                        $site_domain = $stmt->fetch();
                        
                        if (!$site_domain) {
                            $message = 'Domain is not linked to this site';
                            $message_type = 'warning';
                            break;
                        }
                        
                        if ($site_domain['is_primary']) {
                            $message = 'Cannot remove primary domain from site';
                            $message_type = 'danger';
                            break;
                        }
                        
                        // Remove domain from site
                        $stmt = $db->prepare("DELETE FROM whp.site_domains WHERE site_id = ? AND domain_id = ?");
                        $stmt->execute([$site_id, $domain_id]);
                        
                        // Update HAProxy configuration for the site
                        require_once('/docker/whp/web/libs/haproxy_manager.php');
                        $haproxy_manager = new haproxy_manager();
                        $haproxy_result = $haproxy_manager->configureSite($site_id, [
                            'username' => $site['username']
                        ]);
                        
                        if ($haproxy_result['success']) {
                            $message = 'Domain removed from site successfully and HAProxy updated';
                            $message_type = 'success';
                        } else {
                            $message = 'Domain removed from site successfully. However, HAProxy configuration failed: ' . $haproxy_result['error'];
                            $message_type = 'warning';
                        }
                    } catch (PDOException $e) {
                        $message = 'Error removing domain from site: ' . $e->getMessage();
                        $message_type = 'danger';
                    }
                }
                break;
        }
    }
}

// Get user's sites
try {
    if ($is_root) {
        // Root sees all sites
        $stmt = $db->query("
            SELECT 
                s.*,
                COALESCE(sd.fqdn, d.domain_name) as primary_domain,
                ct.name as container_type_name,
                ct.base_image,
                ct.min_cpu,
                ct.min_memory,
                ct.user_variables as container_user_variables
            FROM whp.sites s
            JOIN whp.domains d ON s.primary_domain_id = d.id
            LEFT JOIN whp.site_domains sd ON s.id = sd.site_id AND sd.is_primary = 1
            JOIN whp.container_types ct ON s.container_type_id = ct.id
            ORDER BY s.site_name ASC
        ");
    } else {
        // Regular users see only their sites
        $stmt = $db->prepare("
            SELECT 
                s.*,
                COALESCE(sd.fqdn, d.domain_name) as primary_domain,
                ct.name as container_type_name,
                ct.base_image,
                ct.min_cpu,
                ct.min_memory,
                ct.user_variables as container_user_variables
            FROM whp.sites s
            JOIN whp.domains d ON s.primary_domain_id = d.id
            LEFT JOIN whp.site_domains sd ON s.id = sd.site_id AND sd.is_primary = 1
            JOIN whp.container_types ct ON s.container_type_id = ct.id
            WHERE s.username = ?
            ORDER BY s.site_name ASC
        ");
        $stmt->execute([AUTH_USER]);
    }
    $sites = $stmt->fetchAll();
} catch (PDOException $e) {
    $sites = [];
    $message = 'Error loading sites: ' . $e->getMessage();
    $message_type = 'danger';
}

// Get user's domains for site creation
try {
    // Exclude domains already used as a primary domain in whp.sites
    $exclude_primary_domains_sql = "SELECT primary_domain_id FROM whp.sites";
    if ($is_root) {
        $stmt = $db->query("
            SELECT d.*, u.username as owner 
            FROM whp.domains d 
            JOIN whp.client_allowances u ON d.username = u.username
            WHERE d.active = 1
              AND d.id NOT IN ($exclude_primary_domains_sql)
            ORDER BY d.domain_name ASC
        ");
    } else {
        $stmt = $db->prepare("
            SELECT d.*, d.username as owner 
            FROM whp.domains d 
            WHERE d.username = ? AND d.active = 1
              AND d.id NOT IN ($exclude_primary_domains_sql)
            ORDER BY d.domain_name ASC
        ");
        $stmt->execute([AUTH_USER]);
    }
    $available_domains = $stmt->fetchAll();
} catch (PDOException $e) {
    $available_domains = [];
}

// Get user's domains for additional domain selection (can include domains already linked to sites)
try {
    if ($is_root) {
        $stmt = $db->query("
            SELECT d.*, u.username as owner 
            FROM whp.domains d 
            JOIN whp.client_allowances u ON d.username = u.username
            WHERE d.active = 1
            ORDER BY d.domain_name ASC
        ");
    } else {
        $stmt = $db->prepare("
            SELECT d.*, d.username as owner 
            FROM whp.domains d 
            WHERE d.username = ? AND d.active = 1
            ORDER BY d.domain_name ASC
        ");
        $stmt->execute([AUTH_USER]);
    }
    $all_available_domains = $stmt->fetchAll();
} catch (PDOException $e) {
    $all_available_domains = [];
}

// Get container types
try {
    $stmt = $db->query("SELECT * FROM whp.container_types WHERE active = 1 ORDER BY name ASC");
    $container_types = $stmt->fetchAll();
    
    // Debug: Log container types for troubleshooting
    foreach ($container_types as $type) {
        error_log("Container type {$type['name']} user_variables: " . $type['user_variables']);
    }
} catch (PDOException $e) {
    $container_types = [];
}

// Get user resource limits
$user_limits = null;
try {
    $stmt = $db->prepare("SELECT * FROM whp.client_allowances WHERE username = ?");
    $stmt->execute([AUTH_USER]);
    $user_limits = $stmt->fetch();
} catch (PDOException $e) {
    // User limits not found
}

// Get all FQDNs (A/CNAME) for the current user
$all_fqdns = [];
$available_fqdns_for_creation = [];
$available_fqdns_for_editing = [];
if ($db && isset($PowerDNS)) {
    $username = $is_root ? 'root' : AUTH_USER;
    $all_fqdns = $PowerDNS->getUserDomainFQDNs($username, ['A','CNAME']);
    
    // Get all FQDNs currently used by existing sites (both primary and additional)
    $used_fqdns = [];
    try {
        if ($is_root) {
            $stmt = $db->query("
                SELECT DISTINCT sd.fqdn
                FROM whp.site_domains sd
                JOIN whp.sites s ON sd.site_id = s.id
            ");
        } else {
            $stmt = $db->prepare("
                SELECT DISTINCT sd.fqdn
                FROM whp.site_domains sd
                JOIN whp.sites s ON sd.site_id = s.id
                WHERE s.username = ?
            ");
            $stmt->execute([AUTH_USER]);
        }
        $used_fqdns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (PDOException $e) {
        // If query fails, assume no domains are used
        $used_fqdns = [];
    }
    
    // Filter out used FQDNs for site creation
    foreach ($all_fqdns as $fqdn) {
        if (!in_array($fqdn['fqdn'], $used_fqdns)) {
            $available_fqdns_for_creation[] = $fqdn;
        }
    }
    
    // For editing, we'll need to dynamically add back domains used by the current site
    // This will be handled via JavaScript when the edit modal opens
    $available_fqdns_for_editing = $available_fqdns_for_creation;
    

}
?>

<?php if (!empty($message)): ?>
<div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
    <?php echo htmlspecialchars($message); ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
<?php endif; ?>

<style>
/* Improve site card and list-group-item visibility */
.card.site-card, .list-group-item.site-list-item {
    background-color: #fff !important;
    /* Use utility class for text color */
    color: inherit !important;
}
.card.site-card .card-header {
    background-color: #f8f9fa !important;
    color: inherit !important;
}
/* Use Bootstrap primary blue for muted info on white */
.text-site-info {
    color: #0d6efd !important;
    font-weight: 500;
}
/* Choices.js select box: ensure readable text and background */
.choices__inner, .choices__list--dropdown, .choices__item--selectable {
    background-color: #fff !important;
    color: #212529 !important;
}
.choices__list--dropdown .choices__item--selectable.is-highlighted {
    background-color: #e9ecef !important;
    color: #212529 !important;
}
.choices__list--dropdown .choices__item--selectable {
    color: #212529 !important;
}
.choices[data-type*=select-one] .choices__inner {
    border-radius: 0.375rem;
    border: 1px solid #ced4da;
}
/* Section headers for site details */
.site-section-header {
    /* Use .text-light for dark background */
    color: #f8f9fa !important;
    background: #212529 !important;
    font-weight: 700;
    font-size: 1.1rem;
    border-radius: 0.375rem;
    padding: 0.5rem 1rem;
    margin-bottom: 0.75rem;
    margin-top: 1.25rem;
    border: none;
}
.form-text {
    color: #f8f9fa !important;
}
</style>

<div class="row">
    <div class="col-md-4">
        <div class="card site-card">
            <div class="card-header">
                <h5 class="card-title mb-0 text-dark">
                    Manage Sites
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($sites)): ?>
                <div class="alert alert-info">
                    <i class="fa fa-info-circle"></i>
                    <strong>No sites found</strong> - Create your first site using the form below.
                </div>
                <?php else: ?>
                <label for="siteDropdown" class="form-label text-dark">Select a Site</label>
                <select id="siteDropdown" class="form-control">
                    <option value="">-- Choose a site --</option>
                    <?php foreach ($sites as $site): ?>
                        <option value="<?php echo $site['id']; ?>">
                            <?php echo htmlspecialchars($site['site_name']); ?> (<?php echo htmlspecialchars($site['primary_domain']); ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
                <div id="siteDetailsContainer" class="mt-3" style="display:none;"></div>
                <div id="siteActionsContainer" class="mt-2" style="display:none;">
                    <button id="manageSiteBtn" class="btn btn-primary me-2"><i class="fa fa-cog"></i> Manage</button>
                    <button id="deleteSiteBtn" class="btn btn-danger"><i class="fa fa-trash"></i> Delete</button>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0" id="createSiteFormTitle">Create New Site</h5>
            </div>
            <div class="card-body">
                <form method="post" id="createSiteForm">
                    <input type="hidden" name="action" value="create_site">
                    <input type="hidden" name="csrf_token" value="<?php echo get_csrf_token(); ?>">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="site_name" class="form-label">Site Name</label>
                                <input type="text" class="form-control" id="site_name" name="site_name" required 
                                       placeholder="My Website">
                                <div class="form-text">A friendly name for your site</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="primary_domain_id" class="form-label">Primary Domain or Subdomain</label>
                                <select class="form-control" id="primary_domain_id" name="primary_domain_id" required>
                                    <option value="">Select a domain or subdomain...</option>
                                    <?php foreach ($available_fqdns_for_creation as $fqdn): ?>
                                    <option value="<?php echo htmlspecialchars($fqdn['fqdn']); ?>">
                                        <?php echo htmlspecialchars($fqdn['fqdn']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">The main domain or subdomain for this site (A or CNAME records only)</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="container_type_id" class="form-label">Container Type</label>
                                <select class="form-control" id="container_type_id" name="container_type_id" required onchange="loadContainerTypeConfig()">
                                    <option value="">Select container type...</option>
                                    <?php foreach ($container_types as $type): ?>
                                    <option value="<?php echo $type['id']; ?>" 
                                            data-min-cpu="<?php echo $type['min_cpu']; ?>"
                                            data-min-memory="<?php echo $type['min_memory']; ?>"
                                            data-user-variables="<?php echo htmlspecialchars($type['user_variables']); ?>">
                                        <?php echo htmlspecialchars($type['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">The type of container to use for this site</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="container_count" class="form-label">Number of Containers</label>
                                <input type="number" class="form-control" id="container_count" name="container_count" 
                                       min="1" max="10" value="1" onchange="updateResourceTotals()">
                                <div class="form-text">How many containers to run for this site (1-10)</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Additional Domains Selection -->
                    <div class="row">
                        <div class="col-md-12">
                            <div class="mb-3">
                                <label for="additional_domains" class="form-label">Additional Domains/Subdomains (Optional)</label>
                                <select class="form-control" id="additional_domains" name="additional_domains[]">
                                    <option value="">Select additional domain/subdomain...</option>
                                    <?php foreach ($available_fqdns_for_creation as $fqdn): ?>
                                    <option value="<?php echo htmlspecialchars($fqdn['fqdn']); ?>" class="additional-domain-option">
                                        <?php echo htmlspecialchars($fqdn['fqdn']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">Select additional domains/subdomains to route to this site. Selected domains will appear in the list below.</div>
                                <ul id="selectedAdditionalDomains" class="list-group mt-2"></ul>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="cpu_per_container" class="form-label">CPU per Container (cores)</label>
                                <input type="number" class="form-control" id="cpu_per_container" name="cpu_per_container" 
                                       step="0.25" min="0.25" max="8" value="0.25" onchange="updateResourceTotals()">
                                <div class="form-text">CPU cores allocated to each container</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="memory_per_container" class="form-label">Memory per Container (MB)</label>
                                <input type="number" class="form-control" id="memory_per_container" name="memory_per_container" 
                                       min="128" max="8192" value="256" onchange="updateResourceTotals()">
                                <div class="form-text">Memory allocated to each container</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Resource Totals Display -->
                    <div class="alert alert-info" id="resourceTotals" style="display: none;">
                        <strong>Total Resource Allocation:</strong>
                        <span id="totalCpu">0</span> CPU cores, 
                        <span id="totalMemory">0</span> MB RAM
                        <?php if ($user_limits): ?>
                        <br><small>
                            Your limits: <?php echo $user_limits['max_cpu_allowance']; ?> CPU cores, 
                            <?php echo $user_limits['max_memory_allowance']; ?> MB RAM
                        </small>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Container Type Configuration -->
                    <div id="containerTypeConfig" style="display: none;">
                        <h6 class="fw-bold mb-3">Container Configuration</h6>
                        <div id="userVariablesContainer">
                            <!-- User variables will be loaded here -->
                        </div>
                        <textarea class="form-control d-none" id="user_variables" name="user_variables"></textarea>
                    </div>
                    
                    <!-- SSL/HAProxy Configuration -->
                    <div class="mt-4">
                        <h6 class="fw-bold mb-3">SSL/HAProxy Configuration</h6>
                        <?php
                        // Get server IP for display
                        require_once('/docker/whp/web/libs/dns_validator.php');
                        $dns_validator = new dns_validator();
                        $server_ip = $dns_validator->getServerIPAddress();
                        ?>
                        <div class="alert alert-warning">
                            <i class="fa fa-exclamation-triangle"></i>
                            <strong>Important:</strong> Before enabling SSL, ensure your domain(s) are pointed to our server (<?php echo htmlspecialchars($server_ip); ?>). SSL certificates cannot be issued for domains that are not properly configured.
                        </div>
                        <div class="mt-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="1" id="enable_ssl" name="enable_ssl">
                                <label class="form-check-label" for="enable_ssl">
                                    <i class="fa fa-lock"></i> Enable SSL for all domains
                                </label>
                            </div>
                            <div class="form-text">SSL certificates will be requested for the primary domain and all additional domains after the site is created</div>
                        </div>
                        <div class="alert alert-info mt-3">
                            <i class="fa fa-info-circle"></i>
                            <strong>Note:</strong> SSL certificates are managed through HAProxy Manager and will be automatically configured for your domains.
                        </div>
                    </div>
                    
                    <!-- Container Management Options -->
                    <div class="mt-4">
                        <h6 class="fw-bold mb-3">Container Management</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" value="1" id="auto_recreate_on_update" name="auto_recreate_on_update">
                                        <label class="form-check-label" for="auto_recreate_on_update">
                                            Auto-recreate containers when image updates are available
                                        </label>
                                    </div>
                                    <div class="form-text">When enabled, containers will be automatically recreated when new image versions are pulled</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="d-grid gap-2 mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fa fa-plus"></i> Create Site
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Site Details Modal -->
<div class="modal fade" id="siteDetailsModal" tabindex="-1" aria-labelledby="siteDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="siteDetailsModalLabel">Site Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="siteDetailsContent">
                <!-- Content will be loaded here -->
            </div>
        </div>
    </div>
</div>

<!-- Delete Site Modal -->
<div class="modal fade" id="deleteSiteModal" tabindex="-1" aria-labelledby="deleteSiteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteSiteModalLabel">Delete Site</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the site "<span id="deleteSiteName"></span>"?</p>
                <p class="text-danger"><strong>Warning:</strong> This action cannot be undone. All containers and configurations will be permanently removed.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form method="post" style="display: inline;">
                    <input type="hidden" name="action" value="delete_site">
                    <input type="hidden" name="csrf_token" value="<?php echo get_csrf_token(); ?>">
                    <input type="hidden" name="site_id" id="deleteSiteId">
                    <button type="submit" class="btn btn-danger">Delete Site</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Edit Site Modal -->
<div class="modal fade" id="editSiteModal" tabindex="-1" aria-labelledby="editSiteModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editSiteModalLabel">Edit Site</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="editSiteForm">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_site_name" class="form-label">Site Name</label>
                                <input type="text" class="form-control" id="edit_site_name" name="site_name" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_primary_domain" class="form-label">Primary Domain</label>
                                <input type="text" class="form-control" id="edit_primary_domain" readonly>
                                <div class="form-text">The main domain for this site (cannot be changed)</div>
                            </div>
                        </div>
                    </div>
                    <!-- Additional Domains Selection -->
                    <div class="row">
                        <div class="col-md-12">
                            <div class="mb-3">
                                <label for="edit_additional_domains" class="form-label">Additional Domains/Subdomains (Optional)</label>
                                <select class="form-control" id="edit_additional_domains" name="additional_domains[]">
                                    <option value="">Select additional domain/subdomain...</option>
                                    <?php foreach ($available_fqdns_for_editing as $fqdn): ?>
                                    <option value="<?php echo htmlspecialchars($fqdn['fqdn']); ?>" class="edit-additional-domain-option">
                                        <?php echo htmlspecialchars($fqdn['fqdn']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">Select additional domains/subdomains to route to this site. Selected domains will appear in the list below.</div>
                                <ul id="editSelectedAdditionalDomains" class="list-group mt-2"></ul>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_container_type_id" class="form-label">Container Type</label>
                                <select class="form-control" id="edit_container_type_id" name="container_type_id" required>
                                    <?php foreach ($container_types as $type): ?>
                                    <option value="<?php echo $type['id']; ?>" 
                                            data-min-cpu="<?php echo $type['min_cpu']; ?>"
                                            data-min-memory="<?php echo $type['min_memory']; ?>">
                                        <?php echo htmlspecialchars($type['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_container_count" class="form-label">Number of Containers</label>
                                <input type="number" class="form-control" id="edit_container_count" name="container_count" 
                                       min="1" max="10" value="1" onchange="updateEditResourceTotals()">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_cpu_per_container" class="form-label">CPU per Container (cores)</label>
                                <input type="number" class="form-control" id="edit_cpu_per_container" name="cpu_per_container" 
                                       step="0.25" min="0.25" max="8" value="0.25" onchange="updateEditResourceTotals()">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="edit_memory_per_container" class="form-label">Memory per Container (MB)</label>
                                <input type="number" class="form-control" id="edit_memory_per_container" name="memory_per_container" 
                                       min="128" max="8192" value="256" onchange="updateEditResourceTotals()">
                            </div>
                        </div>
                    </div>
                    
                    <!-- Resource Totals Display -->
                    <div class="alert alert-info" id="editResourceTotals" style="display:none;">
                        <strong>Total Resources:</strong> <span id="editTotalCpu">0</span> CPU cores, <span id="editTotalMemory">0</span> MB memory
                    </div>
                    
                    <div id="editUserVariablesContainer" style="display:none;"></div>
                    
                    <!-- Container Management Options -->
                    <div class="mt-3">
                        <h6>Container Management</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" value="1" id="edit_auto_recreate_on_update" name="auto_recreate_on_update">
                                        <label class="form-check-label" for="edit_auto_recreate_on_update">
                                            Auto-recreate containers when image updates are available
                                        </label>
                                    </div>
                                    <div class="form-text">When enabled, containers will be automatically recreated when new image versions are pulled</div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <button type="button" class="btn btn-info" id="recreateContainersBtn">
                                        <i class="fa fa-refresh"></i> Recreate All Containers
                                    </button>
                                    <div class="form-text">Manually recreate all containers with latest images</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- SSL Configuration -->
                    <div class="mt-3 mb-3">
                        <h6>SSL Configuration</h6>
                        <div class="d-flex gap-3 mb-4"> <!-- Add gap-3 and mb-4 for spacing -->
                            <button type="button" class="btn btn-success btn-sm" onclick="enableSSL()">
                                <i class="fa fa-lock"></i> Enable SSL
                            </button>
                            <button type="button" class="btn btn-info btn-sm" onclick="renewSSL()">
                                <i class="fa fa-refresh"></i> Renew SSL
                            </button>
                        </div>
                        <div class="alert alert-info">
                            <i class="fa fa-info-circle"></i>
                            <strong>Note:</strong> SSL certificates are managed automatically after enabling. Changes will take effect immediately.
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveSiteBtn">
                    <i class="fa fa-save"></i> Save Changes
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Correct Choices.js/CSS URLs for document root -->
<link rel="stylesheet" href="/css/choices.min.css">
<script src="/js/choices.min.js"></script>

<script>
// Define CSRF token for AJAX requests
const nonce = '<?php echo get_csrf_token(); ?>';

let currentEditSiteId = null;
let currentEditSiteData = null;

function loadContainerTypeConfig() {
    const select = document.getElementById('container_type_id');
    const configDiv = document.getElementById('containerTypeConfig');
    const userVariablesContainer = document.getElementById('userVariablesContainer');
    const userVariablesInput = document.getElementById('user_variables');
    
    if (select.value) {
        const option = select.options[select.selectedIndex];
        const userVariables = option.getAttribute('data-user-variables');
        
        if (userVariables && userVariables !== '{}') {
            try {
                const variables = JSON.parse(userVariables);
                console.log('Container type user variables:', variables);
                let html = '';
                
                for (const [varName, config] of Object.entries(variables)) {
                    html += '<div class="mb-3">';
                    html += '<label for="var_' + varName + '" class="form-label">' + config.label + '</label>';
                    
                    if (config.type === 'select') {
                        html += '<select class="form-control" id="var_' + varName + '" name="var_' + varName + '"';
                        if (config.required) html += ' required';
                        html += '>';
                        for (const option of config.options) {
                            const selected = option === config.default ? ' selected' : '';
                            html += '<option value="' + option + '"' + selected + '>' + option + '</option>';
                        }
                        html += '</select>';
                    } else if (config.type === 'number') {
                        html += '<input type="number" class="form-control" id="var_' + varName + '" name="var_' + varName + '"';
                        if (config.default) html += ' value="' + config.default + '"';
                        if (config.min) html += ' min="' + config.min + '"';
                        if (config.max) html += ' max="' + config.max + '"';
                        if (config.required) html += ' required';
                        html += '>';
                    } else if (config.type === 'checkbox') {
                        html += '<div class="form-check">';
                        html += '<input type="checkbox" class="form-check-input" id="var_' + varName + '" name="var_' + varName + '"';
                        if (config.default) html += ' checked';
                        html += '>';
                        html += '<label class="form-check-label" for="var_' + varName + '">' + config.label + '</label>';
                        html += '</div>';
                    } else {
                        html += '<input type="text" class="form-control" id="var_' + varName + '" name="var_' + varName + '"';
                        if (config.default) html += ' value="' + config.default + '"';
                        if (config.required) html += ' required';
                        html += '>';
                    }
                    
                    if (config.description) {
                        html += '<div class="form-text">' + config.description + '</div>';
                    }
                    
                    html += '</div>';
                }
                
                userVariablesContainer.innerHTML = html;
                configDiv.style.display = 'block';
                
                // Update user variables JSON when form is submitted
                // Store the variables in a global scope so they can be accessed during form submission
                window.currentUserVariables = variables;
                
                // Set up a function to collect user variables
                window.collectUserVariables = function() {
                    const collectedVariables = {};
                    
                    console.log('Processing user variables:', window.currentUserVariables);
                    
                    for (const [varName, config] of Object.entries(window.currentUserVariables)) {
                        const input = document.getElementById('var_' + varName);
                        if (input) {
                            if (config.type === 'checkbox') {
                                collectedVariables[varName] = input.checked;
                            } else {
                                collectedVariables[varName] = input.value;
                            }
                        }
                    }
                    
                    console.log('Collected user variables:', collectedVariables);
                    return collectedVariables;
                };
                
            } catch (e) {
                console.error('Error parsing user variables:', e);
                configDiv.style.display = 'none';
            }
        } else {
            configDiv.style.display = 'none';
        }
        
        updateResourceTotals();
    } else {
        configDiv.style.display = 'none';
    }
}

function updateResourceTotals() {
    // Calculate total resources for all containers (not just per-container)
    const containerCount = parseInt(document.getElementById('container_count').value) || 0;
    const cpuPerContainer = parseFloat(document.getElementById('cpu_per_container').value) || 0;
    const memoryPerContainer = parseInt(document.getElementById('memory_per_container').value) || 0;

    const totalCpu = containerCount * cpuPerContainer;
    const totalMemory = containerCount * memoryPerContainer;

    document.getElementById('totalCpu').textContent = totalCpu.toFixed(2);
    document.getElementById('totalMemory').textContent = totalMemory;

    const totalsDiv = document.getElementById('resourceTotals');
    if (containerCount > 0) {
        totalsDiv.style.display = 'block';
        // Check against user limits
        <?php if ($user_limits): ?>
        const userCpuLimit = <?php echo $user_limits['max_cpu_allowance']; ?>;
        const userMemoryLimit = <?php echo $user_limits['max_memory_allowance']; ?>;
        // Enforce total resource limits
        if (totalCpu > userCpuLimit || totalMemory > userMemoryLimit) {
            totalsDiv.className = 'alert alert-warning';
        } else {
            totalsDiv.className = 'alert alert-info';
        }
        <?php endif; ?>
    } else {
        totalsDiv.style.display = 'none';
    }
}

function viewSiteDetails(siteId) {
    // Load site details via AJAX and show modal
    fetch('api/sites.php?action=get_site_details&site_id=' + siteId)
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                document.getElementById('siteDetailsContent').innerHTML = data.html;
                new bootstrap.Modal(document.getElementById('siteDetailsModal')).show();
            } else {
                alert('Error loading site details: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading site details');
        });
}

function deleteSite(siteId, siteName) {
    document.getElementById('deleteSiteId').value = siteId;
    document.getElementById('deleteSiteName').textContent = siteName;
    new bootstrap.Modal(document.getElementById('deleteSiteModal')).show();
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    const createSiteForm = document.getElementById('createSiteForm');
    const originalFormHtml = createSiteForm.innerHTML;
    // Initialize Choices.js for the site dropdown
    const siteDropdown = document.getElementById('siteDropdown');
    if (siteDropdown) {
        const choices = new Choices(siteDropdown, {
            searchEnabled: true,
            itemSelectText: '',
            shouldSort: false
        });
        siteDropdown.addEventListener('change', function() {
            const siteId = this.value;
            if (siteId) {
                fetch('api/sites.php?action=get_site_details&site_id=' + siteId)
                    .then(response => response.json())
                    .then(data => {
                        if (data.status === 'success') {
                            document.getElementById('siteDetailsContainer').innerHTML = data.html;
                            document.getElementById('siteDetailsContainer').style.display = '';
                            document.getElementById('siteActionsContainer').style.display = '';
                            // Set up Manage button
                            document.getElementById('manageSiteBtn').onclick = function() {
                                const site = data.site;
                                showEditSiteModal(site);
                            };
                            // Set up Delete button
                            document.getElementById('deleteSiteBtn').onclick = function() {
                                deleteSite(siteId, data.site.site_name || '');
                            };
                        } else {
                            document.getElementById('siteDetailsContainer').innerHTML = '<div class="alert alert-danger">Error loading site details: ' + data.message + '</div>';
                            document.getElementById('siteDetailsContainer').style.display = '';
                            document.getElementById('siteActionsContainer').style.display = 'none';
                        }
                    })
                    .catch(error => {
                        document.getElementById('siteDetailsContainer').innerHTML = '<div class="alert alert-danger">Error loading site details</div>';
                        document.getElementById('siteDetailsContainer').style.display = '';
                        document.getElementById('siteActionsContainer').style.display = 'none';
                    });
            } else {
                document.getElementById('siteDetailsContainer').style.display = 'none';
                document.getElementById('siteDetailsContainer').innerHTML = '';
                document.getElementById('siteActionsContainer').style.display = 'none';
            }
        });
    }
    // Edit Site Modal Functions
    
    function populateEditDomainsDropdown(site) {
        const editAdditionalDomains = document.getElementById('edit_additional_domains');
        
        // Clear existing options except the first placeholder
        const placeholder = editAdditionalDomains.querySelector('option[value=""]');
        editAdditionalDomains.innerHTML = '';
        if (placeholder) {
            editAdditionalDomains.appendChild(placeholder);
        }
        
        // Get all available FQDNs for editing (from PHP)
        const availableFqdns = <?php echo json_encode($available_fqdns_for_editing); ?>;
        
        // Get domains currently used by this site
        const siteDomains = [];
        if (site.fqdns) {
            site.fqdns.forEach(fqdnObj => {
                siteDomains.push(fqdnObj.fqdn);
            });
        }
        
        // Add available domains
        availableFqdns.forEach(fqdn => {
            const option = document.createElement('option');
            option.value = fqdn.fqdn;
            option.textContent = fqdn.fqdn;
            option.className = 'edit-additional-domain-option';
            editAdditionalDomains.appendChild(option);
        });
        
        // Add domains used by this site (if not already in available list)
        siteDomains.forEach(domain => {
            const existingOption = editAdditionalDomains.querySelector(`option[value="${domain}"]`);
            if (!existingOption) {
                const option = document.createElement('option');
                option.value = domain;
                option.textContent = domain;
                option.className = 'edit-additional-domain-option';
                editAdditionalDomains.appendChild(option);
            }
        });
    }
    
    function showEditSiteModal(site) {
        currentEditSiteId = site.id;
        currentEditSiteData = site;
        
        // Populate modal form
        document.getElementById('edit_site_name').value = site.site_name;
        document.getElementById('edit_primary_domain').value = site.primary_domain;
        document.getElementById('edit_container_type_id').value = site.container_type_id;
        document.getElementById('edit_container_count').value = site.container_count;
        document.getElementById('edit_cpu_per_container').value = site.cpu_per_container;
        document.getElementById('edit_memory_per_container').value = site.memory_per_container;
        document.getElementById('edit_auto_recreate_on_update').checked = site.auto_recreate_on_update == 1;
        
        // Update resource totals
        updateEditResourceTotals();
        
        // Set up recreate containers button
        document.getElementById('recreateContainersBtn').onclick = function() {
            if (!confirm('Are you sure you want to recreate all containers for this site? This will stop and recreate all containers with the current image.')) {
                return;
            }
            
            const btn = document.getElementById('recreateContainersBtn');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Recreating...';
            btn.disabled = true;
            
            fetch('api/sites.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': nonce
                },
                body: JSON.stringify({
                    action: 'recreate_all_containers',
                    site_id: site.id
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    alert('All containers recreated successfully');
                    location.reload();
                } else {
                    alert('Error recreating containers: ' + data.message);
                }
            })
            .catch(error => {
                alert('Network error: ' + error.message);
            })
            .finally(() => {
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
        };
        
        // Show modal
        new bootstrap.Modal(document.getElementById('editSiteModal')).show();

        // Pre-select additional domains using new single-select behavior
        const editAdditionalDomains = document.getElementById('edit_additional_domains');
        const editSelectedList = document.getElementById('editSelectedAdditionalDomains');
        
        // Clear previous selections and list
        editSelectedList.innerHTML = '';
        
        // Dynamically populate dropdown with domains used by this site plus available domains
        populateEditDomainsDropdown(site);
        
        // Update dropdown to exclude primary domain
        updateEditAdditionalDomainsDropdown();
        
        // Pre-populate list with existing additional domains
        if (site.fqdns) {
            site.fqdns.forEach(fqdnObj => {
                if (!fqdnObj.is_primary) {
                    // Find the option and add it to the list
                    const option = editAdditionalDomains.querySelector(`option[value="${fqdnObj.fqdn}"]`);
                    if (option) {
                        // Add to list
                        const li = document.createElement('li');
                        li.className = 'list-group-item d-flex justify-content-between align-items-center text-light';
                        li.textContent = option.text;
                        const btn = document.createElement('button');
                        btn.type = 'button';
                        btn.className = 'btn btn-sm btn-danger';
                        btn.innerHTML = '<i class="fa fa-times"></i>';
                        btn.onclick = function() {
                            // Remove from list
                            li.remove();
                            // Re-enable in dropdown
                            option.style.display = '';
                            option.disabled = false;
                        };
                        li.appendChild(btn);
                        editSelectedList.appendChild(li);
                        
                        // Remove from dropdown
                        option.style.display = 'none';
                        option.disabled = true;
                    }
                }
            });
        }
        
        // Set up event listener for new selections
        editAdditionalDomains.addEventListener('change', addEditDomainToList);
    }
    
    window.updateEditResourceTotals = function() {
        const containerCount = parseInt(document.getElementById('edit_container_count').value) || 0;
        const cpuPerContainer = parseFloat(document.getElementById('edit_cpu_per_container').value) || 0;
        const memoryPerContainer = parseInt(document.getElementById('edit_memory_per_container').value) || 0;

        const totalCpu = containerCount * cpuPerContainer;
        const totalMemory = containerCount * memoryPerContainer;

        document.getElementById('editTotalCpu').textContent = totalCpu.toFixed(2);
        document.getElementById('editTotalMemory').textContent = totalMemory;

        const totalsDiv = document.getElementById('editResourceTotals');
        if (containerCount > 0) {
            totalsDiv.style.display = 'block';
        } else {
            totalsDiv.style.display = 'none';
        }
    }
    
    // Handle save button click
    document.getElementById('saveSiteBtn').onclick = function() {
        const formData = new FormData(document.getElementById('editSiteForm'));
        formData.append('action', 'update_site');
        formData.append('site_id', currentEditSiteId);
        const primaryFqdn = document.getElementById('edit_primary_domain').value;
        formData.set('primary_domain_id', primaryFqdn);
        // Collect all additional domains from the removable list
        const editSelectedList = document.getElementById('editSelectedAdditionalDomains');
        const additionalFqdns = Array.from(editSelectedList.querySelectorAll('li')).map(li => li.textContent.replace(/\s*\u00d7\s*$/, ''));
        // Remove any empty or duplicate values
        const uniqueFqdns = [...new Set(additionalFqdns.filter(fqdn => fqdn && fqdn !== primaryFqdn))];
        // Remove any existing additional_domains from the form
        formData.delete('additional_domains[]');
        uniqueFqdns.forEach(fqdn => formData.append('additional_domains[]', fqdn));
        
        // Disable save button and show loading
        const saveBtn = document.getElementById('saveSiteBtn');
        const originalText = saveBtn.innerHTML;
        saveBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Saving...';
        saveBtn.disabled = true;
        
        fetch('api/sites.php', {
            method: 'POST',
            headers: {
                'X-CSRF-Token': nonce
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                // Close modal
                bootstrap.Modal.getInstance(document.getElementById('editSiteModal')).hide();
                
                // Show success message
                const alertDiv = document.createElement('div');
                alertDiv.className = 'alert alert-success alert-dismissible fade show';
                alertDiv.innerHTML = `
                    Site updated successfully. Containers will be recreated.
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                `;
                document.querySelector('.container-fluid').insertBefore(alertDiv, document.querySelector('.container-fluid').firstChild);
                
                // Refresh site list
                location.reload();
            } else {
                alert('Error updating site: ' + data.message);
            }
        })
        .catch(error => {
            alert('Error updating site');
        })
        .finally(() => {
            // Re-enable save button
            saveBtn.innerHTML = originalText;
            saveBtn.disabled = false;
        });
    };
    // Initialize Choices.js for primary domain dropdown
    const primaryDomainDropdown = document.getElementById('primary_domain_id');
    if (primaryDomainDropdown) {
        window.primaryDomainChoices = new Choices(primaryDomainDropdown, {
            searchEnabled: true,
            itemSelectText: '',
            shouldSort: false
        });
        
        // No longer needed - SSL is now a checkbox
    }
});

// SSL Management Functions for New Sites
// Note: SSL is now handled via checkbox during site creation

// SSL Management Functions
function enableSSL() {
    if (!currentEditSiteId) {
        alert('No site selected');
        return;
    }
    
    if (confirm('Are you sure you want to enable SSL for this site? This will request Let\'s Encrypt certificates for all domains.')) {
        fetch('/api/sites.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': nonce
            },
            body: JSON.stringify({
                action: 'ssl_enable',
                site_id: currentEditSiteId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                alert('SSL enabled successfully');
                location.reload();
            } else if (data.invalid_domains && data.invalid_domains.length > 0) {
                alert('SSL could not be enabled because the following domains are not pointed to this server:\n' + data.invalid_domains.join(', ') + '\n\nPlease update your DNS settings and try again.');
            } else {
                alert('Failed to enable SSL: ' + (data.message || 'Unknown error'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while enabling SSL');
        });
    }
}


function renewSSL() {
    if (!currentEditSiteId) {
        alert('No site selected');
        return;
    }
    
    if (confirm('Are you sure you want to renew SSL certificates for this site?')) {
        fetch('/api/sites.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': nonce
            },
            body: JSON.stringify({
                action: 'ssl_renew',
                site_id: currentEditSiteId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                alert('SSL certificates renewed successfully');
                location.reload();
            } else {
                alert('Failed to renew SSL: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while renewing SSL');
        });
    }
}

// Single-select with removable list for additional domains
const additionalDomainsSelect = document.getElementById('additional_domains');
const selectedList = document.getElementById('selectedAdditionalDomains');
const primaryDomainSelect = document.getElementById('primary_domain_id');

// Function to update additional domains dropdown (exclude primary domain)
function updateAdditionalDomainsDropdown() {
    if (!additionalDomainsSelect || !primaryDomainSelect) return;
    const primaryDomain = primaryDomainSelect.value;
    const additionalOptions = additionalDomainsSelect.querySelectorAll('.additional-domain-option');
    
    additionalOptions.forEach(option => {
        if (option.value === primaryDomain) {
            option.style.display = 'none';
            option.disabled = true;
        } else {
            option.style.display = '';
            option.disabled = false;
        }
    });
}

// Function to add selected domain to list and remove from dropdown
function addDomainToList() {
    if (!additionalDomainsSelect || !selectedList) return;
    const selectedOption = additionalDomainsSelect.options[additionalDomainsSelect.selectedIndex];
    if (selectedOption && selectedOption.value) {
        // Add to list
        const li = document.createElement('li');
        li.className = 'list-group-item d-flex justify-content-between align-items-center text-light';
        li.textContent = selectedOption.text;
        const btn = document.createElement('button');
        btn.type = 'button';
        btn.className = 'btn btn-sm btn-danger';
        btn.innerHTML = '<i class="fa fa-times"></i>';
        btn.onclick = function() {
            // Remove from list
            li.remove();
            // Re-enable in dropdown
            selectedOption.style.display = '';
            selectedOption.disabled = false;
            // Reset dropdown selection
            additionalDomainsSelect.value = '';
        };
        li.appendChild(btn);
        selectedList.appendChild(li);
        
        // Remove from dropdown
        selectedOption.style.display = 'none';
        selectedOption.disabled = true;
        
        // Reset dropdown selection
        additionalDomainsSelect.value = '';
    }
}

// Event listeners
if (primaryDomainSelect) {
    primaryDomainSelect.addEventListener('change', updateAdditionalDomainsDropdown);
}
if (additionalDomainsSelect) {
    additionalDomainsSelect.addEventListener('change', addDomainToList);
}

// Edit modal functions
function updateEditAdditionalDomainsDropdown() {
    const editPrimaryDomain = document.getElementById('edit_primary_domain');
    const editAdditionalDomains = document.getElementById('edit_additional_domains');
    if (!editPrimaryDomain || !editAdditionalDomains) return;
    
    const editPrimaryDomainValue = editPrimaryDomain.value;
    const editAdditionalOptions = editAdditionalDomains.querySelectorAll('.edit-additional-domain-option');
    
    editAdditionalOptions.forEach(option => {
        if (option.value === editPrimaryDomainValue) {
            option.style.display = 'none';
            option.disabled = true;
        } else {
            option.style.display = '';
            option.disabled = false;
        }
    });
}

function addEditDomainToList() {
    const editAdditionalDomains = document.getElementById('edit_additional_domains');
    const editSelectedList = document.getElementById('editSelectedAdditionalDomains');
    if (!editAdditionalDomains || !editSelectedList) return;
    
    const selectedOption = editAdditionalDomains.options[editAdditionalDomains.selectedIndex];
    
    if (selectedOption && selectedOption.value) {
        // Add to list
        const li = document.createElement('li');
        li.className = 'list-group-item d-flex justify-content-between align-items-center text-light';
        li.textContent = selectedOption.text;
        const btn = document.createElement('button');
        btn.type = 'button';
        btn.className = 'btn btn-sm btn-danger';
        btn.innerHTML = '<i class="fa fa-times"></i>';
        btn.onclick = function() {
            // Remove from list
            li.remove();
            // Re-enable in dropdown
            selectedOption.style.display = '';
            selectedOption.disabled = false;
            // Reset dropdown selection
            editAdditionalDomains.value = '';
        };
        li.appendChild(btn);
        editSelectedList.appendChild(li);
        
        // Remove from dropdown
        selectedOption.style.display = 'none';
        selectedOption.disabled = true;
        
        // Reset dropdown selection
        editAdditionalDomains.value = '';
    }
}

// Initialize on page load
if (primaryDomainSelect) {
    updateAdditionalDomainsDropdown();
}

// Container readiness checking function
function checkContainerReadiness(siteId, expectedContainerCount, attempt = 1) {
    const maxAttempts = 30; // 30 attempts = 15 seconds maximum wait
    const checkInterval = 500; // Check every 500ms
    
    if (attempt > maxAttempts) {
        // Timeout - containers taking too long to start
        const submitBtn = document.querySelector('#createSiteForm button[type="submit"]');
        submitBtn.innerHTML = '<i class="fa fa-check"></i> Site Created (containers starting...)';
        submitBtn.disabled = false;
        
        // Show a message and reload after short delay
        alert('Site created successfully! Containers are still starting up. The page will reload in a moment.');
        setTimeout(() => location.reload(), 2000);
        return;
    }
    
    fetch(`api/container-status.php?site_id=${siteId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.all_ready) {
                // All containers are ready!
                const submitBtn = document.querySelector('#createSiteForm button[type="submit"]');
                submitBtn.innerHTML = '<i class="fa fa-check"></i> Site Ready!';
                submitBtn.disabled = false;
                
                // Brief success feedback before reload
                setTimeout(() => location.reload(), 1000);
            } else if (data.success) {
                // Containers still starting, show progress
                const submitBtn = document.querySelector('#createSiteForm button[type="submit"]');
                const readyCount = data.ready_count || 0;
                const totalCount = data.container_count || expectedContainerCount;
                submitBtn.innerHTML = `<i class="fa fa-spinner fa-spin"></i> Containers starting (${readyCount}/${totalCount})...`;
                
                // Check again after delay
                setTimeout(() => checkContainerReadiness(siteId, expectedContainerCount, attempt + 1), checkInterval);
            } else {
                // Error checking container status
                console.warn('Container status check failed:', data.error);
                setTimeout(() => checkContainerReadiness(siteId, expectedContainerCount, attempt + 1), checkInterval);
            }
        })
        .catch(error => {
            console.warn('Container status check error:', error);
            // Continue trying despite errors
            setTimeout(() => checkContainerReadiness(siteId, expectedContainerCount, attempt + 1), checkInterval);
        });
}

// 1. After successful site creation, fetch and display new site details
// Find the form submission handler for createSiteForm (look for 'onsubmit' or submit event)
createSiteForm.onsubmit = function(e) {
    e.preventDefault();
    const formData = new FormData(createSiteForm);
    formData.append('action', 'create_site');
    
    // Collect additional domains from the selected list
    const selectedList = document.getElementById('selectedAdditionalDomains');
    const additionalFqdns = Array.from(selectedList.querySelectorAll('li')).map(li => li.textContent.replace(/\s*\u00d7\s*$/, ''));
    
    // Remove any existing additional_domains from the form
    formData.delete('additional_domains[]');
    additionalFqdns.forEach(fqdn => formData.append('additional_domains[]', fqdn));
    
    // Collect user variables if available
    if (window.collectUserVariables) {
        const userVariables = window.collectUserVariables();
        formData.set('user_variables', JSON.stringify(userVariables));
    }
    
    // Debug: Log form data for troubleshooting
    console.log('Form data being sent:');
    for (let [key, value] of formData.entries()) {
        console.log(key + ': ' + value);
    }
    
    // Disable submit button
    const submitBtn = createSiteForm.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Creating...';
    submitBtn.disabled = true;
    fetch('api/sites.php', {
        method: 'POST',
        headers: {
            'X-CSRF-Token': nonce
        },
        body: formData
    })
    .then(response => {
        // Check if response is JSON (API) or HTML (fallback)
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return response.json();
        } else {
            // Fall back to regular form submission if we got HTML response
            throw new Error('Received HTML response, falling back to form submission');
        }
    })
    .then(data => {
        if (data.success && data.site_id) {
            // Site created successfully, now check container readiness
            submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Waiting for containers...';
            checkContainerReadiness(data.site_id, data.container_count);
        } else {
            // Site creation failed or partial failure
            alert(data.message || data.error || 'Error creating site');
            location.reload();
        }
    })
    .catch(error => {
        console.log('AJAX failed, falling back to regular form submission:', error.message);
        // Reset button before form submission
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
        // Fall back to regular form submission
        createSiteForm.submit();
    });
};

// 2. In the container status display, use the real container name from the database
// Find the section that renders container status (look for 'Container 1', 'Container 2', etc.)
// Replace with:
// site_containers.forEach(function(container) {
//   html += '<div>' + container.name + ' (' + container.status + ')</div>';
// });
</script> 