#!/bin/bash

# Container Manager Script for WHP
# This script manages system Docker containers (MySQL, HAProxy Manager, Memcache)
# with support for API key configuration and container lifecycle operations

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to get HAProxy API key from settings
get_haproxy_api_key() {
    local api_key=""
    if [ -f "/docker/whp/settings.json" ]; then
        # Try jq first (more reliable for JSON parsing)
        if command -v jq >/dev/null 2>&1; then
            api_key=$(jq -r '.haproxy_api_key // empty' /docker/whp/settings.json 2>/dev/null || echo "")
        else
            # Fallback to grep/sed method
            api_key=$(sed -n 's/.*"haproxy_api_key":"\([^"]*\)".*/\1/p' /docker/whp/settings.json 2>/dev/null || echo "")
        fi
    fi
    echo "$api_key"
}

# Function to get MySQL password
get_mysql_password() {
    local mysql_password=""
    if [ -f "/root/.my.cnf" ]; then
        mysql_password=$(grep '^password=' /root/.my.cnf | cut -d'=' -f2-)
    fi
    echo "$mysql_password"
}

# Function to get PostgreSQL password
get_postgresql_password() {
    local postgres_password=""
    if [ -f "/root/.pgpass" ]; then
        postgres_password=$(cut -d: -f5 /root/.pgpass | head -1)
    fi
    echo "$postgres_password"
}

# Function to recreate HAProxy container
recreate_haproxy() {
    print_status "Recreating HAProxy container..."
    
    # Stop and remove existing container
    if docker ps -a --filter name=haproxy-manager --format '{{.Names}}' | grep -q haproxy-manager; then
        print_status "Stopping existing HAProxy container..."
        docker stop haproxy-manager 2>/dev/null || true
        print_status "Removing existing HAProxy container..."
        docker rm haproxy-manager 2>/dev/null || true
    fi
    
    # Pull latest image
    print_status "Pulling latest HAProxy image..."
    docker pull repo.anhonesthost.net/cloud-hosting-platform/haproxy-manager-base:latest
    
    # Get API key
    local api_key=$(get_haproxy_api_key)
    local api_key_env=""
    
    if [ -n "$api_key" ]; then
        api_key_env="-e HAPROXY_API_KEY=$api_key "
        print_status "Using API key from settings"
    else
        print_warning "No API key found in settings - container will start without API key"
    fi
    
    # Start new container
    print_status "Starting new HAProxy container..."
    docker run -d --network=client-net --add-host=host.docker.internal:host-gateway \
        -p 80:80 -p 443:443 -p 8000:8000 \
        -v lets-encrypt:/etc/letsencrypt -v haproxy:/etc/haproxy \
        $api_key_env--name haproxy-manager --restart unless-stopped \
        repo.anhonesthost.net/cloud-hosting-platform/haproxy-manager-base:latest
    
    # Verify container is running
    if docker ps --filter name=haproxy-manager --format '{{.Status}}' | grep -q "Up"; then
        print_success "HAProxy container recreated and running successfully"
    else
        print_error "Failed to start HAProxy container"
        exit 1
    fi
}

# Function to recreate MySQL container
recreate_mysql() {
    print_status "Recreating MySQL container..."
    
    # Stop and remove existing container
    if docker ps -a --filter name=mysql --format '{{.Names}}' | grep -q mysql; then
        print_status "Stopping existing MySQL container..."
        docker stop mysql 2>/dev/null || true
        print_status "Removing existing MySQL container..."
        docker rm mysql 2>/dev/null || true
    fi
    
    # Pull latest image
    print_status "Pulling latest MySQL image..."
    docker pull mariadb:10.11
    
    # Get MySQL password
    local mysql_password=$(get_mysql_password)
    if [ -z "$mysql_password" ]; then
        print_error "Could not determine MySQL password from /root/.my.cnf"
        exit 1
    fi
    
    # Start new container (bind to localhost only for security)
    print_status "Starting new MySQL container..."
    docker run -d --network=client-net --name mysql \
        --env MARIADB_ROOT_PASSWORD="$mysql_password" \
        -v /docker/mysql:/var/lib/mysql:Z -p 127.0.0.1:3306:3306 \
        --restart unless-stopped mariadb:10.11
    
    # Verify container is running
    if docker ps --filter name=mysql --format '{{.Status}}' | grep -q "Up"; then
        print_success "MySQL container recreated and running successfully"
    else
        print_error "Failed to start MySQL container"
        exit 1
    fi
}

# Function to recreate Memcache container
recreate_memcache() {
    print_status "Recreating Memcache container..."
    
    # Stop and remove existing container
    if docker ps -a --filter name=memcache --format '{{.Names}}' | grep -q memcache; then
        print_status "Stopping existing Memcache container..."
        docker stop memcache 2>/dev/null || true
        print_status "Removing existing Memcache container..."
        docker rm memcache 2>/dev/null || true
    fi
    
    # Pull latest image
    print_status "Pulling latest Memcache image..."
    docker pull repo.anhonesthost.net/cloud-hosting-platform/cmc:latest
    
    # Start new container
    print_status "Starting new Memcache container..."
    docker run -d --network=client-net --add-host=host.docker.internal:host-gateway \
        -p 11211:11211 --restart unless-stopped --name memcache \
        repo.anhonesthost.net/cloud-hosting-platform/cmc:latest
    
    # Verify container is running
    if docker ps --filter name=memcache --format '{{.Status}}' | grep -q "Up"; then
        print_success "Memcache container recreated and running successfully"
    else
        print_error "Failed to start Memcache container"
        exit 1
    fi
}

# Function to recreate PostgreSQL container
recreate_postgresql() {
    print_status "Recreating PostgreSQL container..."
    
    # Stop and remove existing container
    if docker ps -a --filter name=postgresql --format '{{.Names}}' | grep -q postgresql; then
        print_status "Stopping existing PostgreSQL container..."
        docker stop postgresql 2>/dev/null || true
        print_status "Removing existing PostgreSQL container..."
        docker rm postgresql 2>/dev/null || true
    fi
    
    # Pull latest image
    print_status "Pulling latest PostgreSQL image..."
    docker pull postgres:latest
    
    # Get PostgreSQL password
    local postgres_password=$(get_postgresql_password)
    if [ -z "$postgres_password" ]; then
        print_warning "PostgreSQL password not found in /root/.pgpass - skipping PostgreSQL recreation"
        print_warning "If PostgreSQL is installed, please check /root/.pgpass configuration"
        return 0
    fi
    
    # Start new container (bind to localhost only for security)
    print_status "Starting new PostgreSQL container..."
    docker run -d --network=client-net --name postgresql \
        --env POSTGRES_PASSWORD="$postgres_password" \
        -v /docker/postgresql:/var/lib/postgresql/data:Z -p 127.0.0.1:5432:5432 \
        --restart unless-stopped postgres:latest
    
    # Verify container is running
    if docker ps --filter name=postgresql --format '{{.Status}}' | grep -q "Up"; then
        print_success "PostgreSQL container recreated and running successfully"
    else
        print_error "Failed to start PostgreSQL container"
        exit 1
    fi
}

# Function to show container status
show_status() {
    print_status "Container Status:"
    echo ""
    
    local containers=("mysql" "postgresql" "haproxy-manager" "memcache")
    
    for container in "${containers[@]}"; do
        local status=$(docker ps --filter name=$container --format '{{.Status}}' 2>/dev/null)
        if [ -n "$status" ]; then
            print_success "$container: Running ($status)"
        else
            local exists=$(docker ps -a --filter name=$container --format '{{.Names}}' 2>/dev/null)
            if [ -n "$exists" ]; then
                print_warning "$container: Stopped"
            else
                print_error "$container: Not found"
            fi
        fi
    done
}

# Function to show help
show_help() {
    echo "Container Manager Script for WHP"
    echo ""
    echo "Usage: $0 [COMMAND] [CONTAINER]"
    echo ""
    echo "Commands:"
    echo "  status                    Show status of all containers"
    echo "  start <container>         Start a specific container"
    echo "  stop <container>          Stop a specific container"
    echo "  restart <container>       Restart a specific container"
    echo "  recreate <container>      Pull latest image and recreate container"
    echo "  recreate-all              Recreate all containers"
    echo "  help                      Show this help message"
    echo ""
    echo "Containers:"
    echo "  mysql                     MariaDB database container"
    echo "  postgresql                PostgreSQL database container"
    echo "  haproxy-manager           HAProxy Manager container (with API key support)"
    echo "  memcache                  Memcache container"
    echo ""
    echo "Examples:"
    echo "  $0 status"
    echo "  $0 start mysql"
    echo "  $0 recreate haproxy-manager"
    echo "  $0 recreate-all"
}

# Main script logic
case "${1:-help}" in
    "status")
        show_status
        ;;
    "start")
        if [ -z "$2" ]; then
            print_error "Container name required"
            exit 1
        fi
        print_status "Starting container: $2"
        docker start "$2"
        print_success "Container $2 started"
        ;;
    "stop")
        if [ -z "$2" ]; then
            print_error "Container name required"
            exit 1
        fi
        print_status "Stopping container: $2"
        docker stop "$2"
        print_success "Container $2 stopped"
        ;;
    "restart")
        if [ -z "$2" ]; then
            print_error "Container name required"
            exit 1
        fi
        print_status "Restarting container: $2"
        docker restart "$2"
        print_success "Container $2 restarted"
        ;;
    "recreate")
        if [ -z "$2" ]; then
            print_error "Container name required"
            exit 1
        fi
        case "$2" in
            "mysql")
                recreate_mysql
                ;;
            "haproxy-manager")
                recreate_haproxy
                ;;
            "memcache")
                recreate_memcache
                ;;
            "postgresql")
                recreate_postgresql
                ;;
            *)
                print_error "Unknown container: $2"
                print_error "Valid containers: mysql, postgresql, haproxy-manager, memcache"
                exit 1
                ;;
        esac
        ;;
    "recreate-all")
        print_status "Recreating all containers..."
        recreate_mysql
        # Only recreate PostgreSQL if it's configured
        if [ -f "/root/.pgpass" ] && [ -n "$(get_postgresql_password)" ]; then
            recreate_postgresql
        fi
        recreate_haproxy
        recreate_memcache
        print_success "All containers recreated successfully"
        ;;
    "help"|*)
        show_help
        ;;
esac 