<?php

namespace WHPBackup;

require_once(__DIR__ . '/../configs/config.php');
require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('BackupTarget.php');
require_once('BackupHistory.php');
require_once('BackupStorage.php');
require_once('BackupLocalStorage.php');

use Exception;

class BackupJob {
    private $db;
    private $tempDir;
    private $localStorage;

    public function __construct() {
        $mysql = new \mysqlmgmt();
        $this->db = $mysql->getMySQLConnection();
        $this->db->exec("USE whp");
        $this->tempDir = '/docker/tmp';
        $this->localStorage = new BackupLocalStorage($this->db);

        if (!is_dir($this->tempDir)) {
            mkdir($this->tempDir, 0755, true);
        }
    }
    
    public function processBackup($backupId) {
        // Set generous time limit for backup operations (30 minutes)
        set_time_limit(1800);

        $history = new BackupHistory($this->db, $backupId);
        $data = $history->getData();

        if ($data['status'] !== 'pending' && $data['status'] !== 'claimed') {
            throw new Exception("Backup is not in pending or claimed status, current status: " . $data['status']);
        }

        // Update status to running and creation_status to creating
        $history->updateStatus('running');
        $this->updateCreationStatus($backupId, 'creating');

        try {
            // Phase 1: Create backup locally
            $localPath = $this->createBackupLocally($data);

            // Phase 2: Queue for upload
            $this->queueForUpload($backupId, $localPath);

            // Mark creation as completed (backup is now ready for upload)
            $this->updateCreationStatus($backupId, 'created');
            $history->updateStatus('completed'); // This means creation completed, not full upload

        } catch (Exception $e) {
            // Log the failure before updating status
            error_log("Backup creation job {$backupId} failed: " . $e->getMessage());

            // Update status to failed
            $this->updateCreationStatus($backupId, 'failed');
            $history->updateStatus('failed', $e->getMessage());

            // Clean up any partial local backup
            if (isset($localPath) && file_exists($localPath)) {
                $this->localStorage->markFailed($localPath);
            }

            throw $e;
        }
    }

    /**
     * Create backup locally without uploading
     */
    private function createBackupLocally($data) {
        // Estimate backup size for space allocation
        $estimatedSize = $this->estimateBackupSize($data);

        // Allocate local storage space
        $localPath = $this->localStorage->allocateSpace($data['backup_name'], $estimatedSize);

        // Update database with local path
        $stmt = $this->db->prepare("UPDATE backup_history SET local_path = ? WHERE id = ?");
        $stmt->execute([$localPath, $data['id']]);

        try {
            switch ($data['backup_type']) {
                case 'site':
                    $this->createSiteBackup($data, $localPath);
                    break;
                case 'userfiles':
                    $this->createUserfilesBackup($data, $localPath);
                    break;
                case 'database':
                    $this->createDatabaseBackup($data, $localPath);
                    break;
                case 'wordpress':
                    $this->createWordPressBackup($data, $localPath);
                    break;
                default:
                    throw new Exception("Unknown backup type: " . $data['backup_type']);
            }

            // Get actual file size and mark as created
            $actualSize = filesize($localPath);
            $readyPath = $this->localStorage->markCreated($localPath, $actualSize);

            // Update database with final path and size
            $stmt = $this->db->prepare("
                UPDATE backup_history
                SET local_path = ?, local_size = ?, backup_size = ?
                WHERE id = ?
            ");
            $stmt->execute([$readyPath, $actualSize, $actualSize, $data['id']]);

            return $readyPath;

        } catch (Exception $e) {
            // Clean up failed backup
            $this->localStorage->markFailed($localPath);
            throw $e;
        }
    }

    /**
     * Queue backup for upload
     */
    private function queueForUpload($backupId, $localPath) {
        // Add to upload queue
        $stmt = $this->db->prepare("
            INSERT INTO backup_upload_queue (backup_id, priority, status)
            VALUES (?, 5, 'queued')
            ON DUPLICATE KEY UPDATE
            status = 'queued', retry_count = 0, error_message = NULL
        ");
        $stmt->execute([$backupId]);

        // Update backup status
        $stmt = $this->db->prepare("
            UPDATE backup_history
            SET upload_status = 'pending'
            WHERE id = ?
        ");
        $stmt->execute([$backupId]);

        error_log("Backup {$backupId} created locally ({$localPath}) and queued for upload");
    }

    /**
     * Update creation status
     */
    private function updateCreationStatus($backupId, $status) {
        $stmt = $this->db->prepare("
            UPDATE backup_history
            SET creation_status = ?
            WHERE id = ?
        ");
        $stmt->execute([$status, $backupId]);
    }

    /**
     * Estimate backup size for space allocation
     */
    private function estimateBackupSize($data) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);

        switch ($data['backup_type']) {
            case 'site':
            case 'wordpress':
                // Estimate based on directory size
                $sourcePath = $metadata['source_path'] ?? null;
                if ($sourcePath && is_dir($sourcePath)) {
                    return $this->getDirectorySize($sourcePath);
                }
                break;

            case 'userfiles':
                $sourcePath = $metadata['source_path'] ?? null;
                if ($sourcePath && is_dir($sourcePath)) {
                    return $this->getDirectorySize($sourcePath);
                }
                break;

            case 'database':
                // Estimate database size
                $database = $metadata['database'] ?? '';
                return $this->estimateDatabaseSize($database);
        }

        // Default estimate: 100MB
        return 100 * 1024 * 1024;
    }

    private function getDirectorySize($directory) {
        $size = 0;
        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $size += $file->getSize();
                }
            }
        } catch (Exception $e) {
            // If we can't read the directory, use a default estimate
            error_log("Failed to calculate directory size for {$directory}: " . $e->getMessage());
            $size = 50 * 1024 * 1024; // 50MB default
        }

        return $size;
    }

    private function estimateDatabaseSize($database) {
        try {
            $stmt = $this->db->prepare("
                SELECT ROUND(SUM(data_length + index_length), 0) as size
                FROM information_schema.tables
                WHERE table_schema = ?
            ");
            $stmt->execute([$database]);
            $result = $stmt->fetch();

            return $result['size'] ?? (10 * 1024 * 1024); // 10MB default
        } catch (Exception $e) {
            error_log("Failed to estimate database size for {$database}: " . $e->getMessage());
            return 10 * 1024 * 1024; // 10MB default
        }
    }
    
    private function createSiteBackup($data, $localPath) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $user = $data['user'];
        $domain = $metadata['domain'];

        // Create archive directly to local storage
        $this->createSiteArchive($sourcePath, $localPath, $user, $domain);
    }
    
    private function createUserfilesBackup($data, $localPath) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $user = $data['user'];

        // Create archive directly to local storage (use zip for better compatibility)
        $this->createUserfilesArchiveZip($sourcePath, $localPath, $user);
    }
    
    private function createDatabaseBackup($data, $localPath) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $database = $metadata['database'];
        $user = $data['user'];

        // Create SQL dump directly to local storage
        $this->createDatabaseDump($database, $localPath);
    }

    private function createWordPressBackup($data, $localPath) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $database = $metadata['database'] ?? null;
        $domain = $metadata['domain'];
        $user = $data['user'];

        // Validate user and domain for security
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $user)) {
            throw new Exception("Invalid user format");
        }
        if (!preg_match('/^[a-zA-Z0-9.-]+$/', $domain)) {
            throw new Exception("Invalid domain format");
        }

        // Validate source path is within expected directory structure
        $expectedBasePath = "/docker/users/$user/";
        $realSourcePath = realpath($sourcePath);
        if (!$realSourcePath || strpos($realSourcePath, $expectedBasePath) !== 0) {
            throw new Exception("Invalid source path");
        }

        // Create combined WordPress archive directly to local storage
        $this->createWordPressArchive($sourcePath, $database, $localPath, $user, $domain);
    }
    
    private function createSiteArchive($sourcePath, $tempFile, $user, $domain) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Site directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        $domainName = basename($sourcePath);
        
        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/" . $domainName) . " && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression (level 6 for better speed/ratio balance)
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tempFile) . " --use-compress-program='gzip -6' " . escapeshellarg($userName . "/" . $domainName);
        }

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception("Failed to create site archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Site archive file is empty or doesn't exist");
        }
    }
    
    private function createUserfilesArchive($sourcePath, $tempFile, $user) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Userfiles directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        
        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/userfiles") . " 2>&1 && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression (level 6 for better speed/ratio balance)
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -czf " . escapeshellarg($tempFile) . " --use-compress-program='gzip -6' " . escapeshellarg($userName . "/userfiles") . " 2>&1";
        }

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception("Failed to create userfiles archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Userfiles archive file is empty or doesn't exist");
        }
        
    }
    
    private function createUserfilesArchiveZip($sourcePath, $tempFile, $user) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Userfiles directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        
        // Create zip archive
        $command = "cd " . escapeshellarg(dirname($userDir)) . " && zip -r " . escapeshellarg($tempFile) . " " . escapeshellarg($userName . "/userfiles") . " 2>&1";

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception("Failed to create userfiles archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Userfiles archive file is empty or doesn't exist");
        }
        
    }
    
    private function createDatabaseDump($database, $tempFile) {
        // Validate database name for security (prevent SQL injection and command injection)
        // Allow alphanumeric characters, underscores, and hyphens (common in WordPress database names)
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $database)) {
            throw new Exception("Invalid database name format");
        }

        // Get database credentials
        $mysql = new \mysqlmgmt();
        $host = '127.0.0.1';
        $port = 3306;
        $rootUser = 'root';
        $rootPass = defined('MYSQL_PASS') ? trim(MYSQL_PASS) : '';
        
        // First, test if the database exists
        try {
            $testStmt = $this->db->prepare("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?");
            $testStmt->execute([$database]);
            if (!$testStmt->fetch()) {
                throw new Exception("Database '$database' does not exist");
            }
        } catch (Exception $e) {
            throw new Exception("Failed to verify database existence: " . $e->getMessage());
        }
        
        // Create a temporary SQL file first, then compress it
        $tempSqlFile = $tempFile . '.sql';
        
        // Create mysqldump command with proper options for full database recreation
        $passwordArg = $rootPass ? "--password=" . escapeshellarg($rootPass) : "";
        
        $command = "mysqldump -h " . escapeshellarg($host) . 
                  " -P " . escapeshellarg($port) . 
                  " -u " . escapeshellarg($rootUser) . 
                  " " . $passwordArg . 
                  " --single-transaction --routines --triggers --events" .
                  " --add-drop-database --create-options --complete-insert" .
                  " --databases " . escapeshellarg($database) . 
                  " > " . escapeshellarg($tempSqlFile) . " 2>&1";
        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            // Clean up temp file
            if (file_exists($tempSqlFile)) {
                unlink($tempSqlFile);
            }
            
            throw new Exception("Failed to create database dump: " . implode("\n", $output));
        }
        
        // Check if SQL file was created and has content
        if (!file_exists($tempSqlFile) || filesize($tempSqlFile) === 0) {
            throw new Exception("Database dump SQL file is empty or doesn't exist");
        }
        
        // Compress the SQL file - use pigz if available for better performance
        if (shell_exec('which pigz 2>/dev/null')) {
            $gzipCommand = "pigz -6 -c " . escapeshellarg($tempSqlFile) . " > " . escapeshellarg($tempFile);
        } else {
            $gzipCommand = "gzip -6 -c " . escapeshellarg($tempSqlFile) . " > " . escapeshellarg($tempFile);
        }

        exec($gzipCommand, $gzipOutput, $gzipReturn);

        // Clean up temp SQL file
        if (file_exists($tempSqlFile)) {
            unlink($tempSqlFile);
        }
        if ($gzipReturn !== 0) {
            throw new Exception("Failed to compress database dump: " . implode("\n", $gzipOutput));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Compressed database dump file is empty or doesn't exist");
        }
    }

    private function createWordPressArchive($sourcePath, $database, $tempFile, $user, $domain) {
        if (!is_dir($sourcePath)) {
            throw new Exception("WordPress site directory not found: $sourcePath");
        }

        // Create temporary directory for organizing the backup content
        // Use random_bytes for cryptographically secure temporary directory name
        $randomSuffix = bin2hex(random_bytes(16));
        $tempWorkDir = $this->tempDir . '/wp_backup_' . $randomSuffix;
        if (!mkdir($tempWorkDir, 0700, true)) { // More restrictive permissions
            throw new Exception("Failed to create temporary work directory");
        }

        try {
            // Step 1: Create site files archive
            $siteArchivePath = $tempWorkDir . '/site.tar.gz';
            $this->createSiteArchiveForWordPress($sourcePath, $siteArchivePath, $user, $domain);

            // Step 2: Create database dump if database exists
            $dbDumpPath = null;
            if (!empty($database)) {
                $dbDumpPath = $tempWorkDir . '/database.sql.gz';
                $this->createDatabaseDump($database, $dbDumpPath);
            }

            // Step 3: Create manifest file with backup information
            $manifestPath = $tempWorkDir . '/backup-manifest.json';
            $manifest = [
                'backup_type' => 'wordpress',
                'domain' => $domain,
                'user' => $user,
                'created_at' => date('Y-m-d H:i:s'),
                'files' => [
                    'site' => 'site.tar.gz',
                    'database' => $database ? 'database.sql.gz' : null
                ],
                'database_name' => $database
            ];
            file_put_contents($manifestPath, json_encode($manifest, JSON_PRETTY_PRINT));

            // Step 4: Create final combined archive
            $filesToArchive = ['backup-manifest.json', 'site.tar.gz'];
            if ($dbDumpPath && file_exists($dbDumpPath)) {
                $filesToArchive[] = 'database.sql.gz';
            }

            // Use tar with gzip compression
            $filesArg = implode(' ', array_map('escapeshellarg', $filesToArchive));
            $command = "cd " . escapeshellarg($tempWorkDir) . " && tar -czf " . escapeshellarg($tempFile) . " " . $filesArg;

            exec($command, $output, $returnCode);

            if ($returnCode !== 0) {
                throw new Exception("Failed to create WordPress archive: " . implode("\n", $output));
            }

            if (!file_exists($tempFile) || filesize($tempFile) === 0) {
                throw new Exception("WordPress archive file is empty or doesn't exist");
            }

        } finally {
            // Clean up temporary work directory
            if (is_dir($tempWorkDir)) {
                $this->removeDirectory($tempWorkDir);
            }
        }
    }

    private function createSiteArchiveForWordPress($sourcePath, $tempFile, $user, $domain) {
        // This is similar to createSiteArchive but creates the archive directly in the specified location
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        $domainName = basename($sourcePath);

        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/" . $domainName) . " && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -czf " . escapeshellarg($tempFile) . " " . escapeshellarg($userName . "/" . $domainName);
        }

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception("Failed to create site archive: " . implode("\n", $output));
        }

        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Site archive file is empty or doesn't exist");
        }
    }

    private function removeDirectory($dir) {
        if (!is_dir($dir)) {
            return;
        }

        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }

    // uploadToStorage method removed - uploads now handled by separate upload worker
}