<?php

namespace WHPBackup;

require_once(__DIR__ . '/../configs/config.php');
require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('BackupTarget.php');
require_once('BackupHistory.php');
require_once('BackupStorage.php');

use Exception;

class BackupJob {
    private $db;
    private $tempDir;
    
    public function __construct() {
        $mysql = new \mysqlmgmt();
        $this->db = $mysql->getMySQLConnection();
        $this->db->exec("USE whp");
        $this->tempDir = '/docker/tmp';
        
        if (!is_dir($this->tempDir)) {
            mkdir($this->tempDir, 0755, true);
        }
    }
    
    public function processBackup($backupId) {
        $history = new BackupHistory($this->db, $backupId);
        $data = $history->getData();
        
        if ($data['status'] !== 'pending' && $data['status'] !== 'claimed') {
            throw new Exception("Backup is not in pending or claimed status, current status: " . $data['status']);
        }
        
        // Update status to running
        $history->updateStatus('running');
        
        try {
            switch ($data['backup_type']) {
                case 'site':
                    $this->processSiteBackup($history, $data);
                    break;
                case 'userfiles':
                    $this->processUserfilesBackup($history, $data);
                    break;
                case 'database':
                    $this->processDatabaseBackup($history, $data);
                    break;
                case 'wordpress':
                    $this->processWordPressBackup($history, $data);
                    break;
                default:
                    throw new Exception("Unknown backup type: " . $data['backup_type']);
            }
            
            // Update status to completed
            $history->updateStatus('completed');
            
        } catch (Exception $e) {
            // Update status to failed
            $history->updateStatus('failed', $e->getMessage());
            throw $e;
        }
    }
    
    private function processSiteBackup($history, $data) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $user = $data['user'];
        $domain = $metadata['domain'];
        
        // Step 1: Create archive
        $tempFile = $this->tempDir . '/' . $data['backup_name'] . '.tar.gz';
        $this->createSiteArchive($sourcePath, $tempFile, $user, $domain);
        
        // Step 2: Upload to storage
        $this->uploadToStorage($history, $data, $tempFile);
        
        // Step 3: Clean up temp file
        if (file_exists($tempFile)) {
            unlink($tempFile);
        }
    }
    
    private function processUserfilesBackup($history, $data) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $user = $data['user'];
        
        // Step 1: Create archive
        // Try zip format for better compatibility
        $tempFile = $this->tempDir . '/' . $data['backup_name'] . '.zip';
        $this->createUserfilesArchiveZip($sourcePath, $tempFile, $user);
        
        // Step 2: Upload to storage
        $this->uploadToStorage($history, $data, $tempFile);
        
        // Step 3: Clean up temp file
        if (file_exists($tempFile)) {
            unlink($tempFile);
        }
    }
    
    private function processDatabaseBackup($history, $data) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $database = $metadata['database'];
        $user = $data['user'];

        // Step 1: Create SQL dump
        $tempFile = $this->tempDir . '/' . $data['backup_name'] . '.sql.gz';
        $this->createDatabaseDump($database, $tempFile);

        // Step 2: Upload to storage
        $this->uploadToStorage($history, $data, $tempFile);

        // Step 3: Clean up temp file
        if (file_exists($tempFile)) {
            unlink($tempFile);
        }
    }

    private function processWordPressBackup($history, $data) {
        $metadata = is_array($data['metadata']) ? $data['metadata'] : json_decode($data['metadata'], true);
        $sourcePath = $metadata['source_path'];
        $database = $metadata['database'] ?? null;
        $domain = $metadata['domain'];
        $user = $data['user'];

        // Validate user and domain for security
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $user)) {
            throw new Exception("Invalid user format");
        }
        if (!preg_match('/^[a-zA-Z0-9.-]+$/', $domain)) {
            throw new Exception("Invalid domain format");
        }

        // Validate source path is within expected directory structure
        $expectedBasePath = "/docker/users/$user/";
        $realSourcePath = realpath($sourcePath);
        if (!$realSourcePath || strpos($realSourcePath, $expectedBasePath) !== 0) {
            throw new Exception("Invalid source path");
        }

        // Step 1: Create combined WordPress archive
        $tempFile = $this->tempDir . '/' . $data['backup_name'] . '.tar.gz';
        $this->createWordPressArchive($sourcePath, $database, $tempFile, $user, $domain);

        // Step 2: Upload to storage
        $this->uploadToStorage($history, $data, $tempFile);

        // Step 3: Clean up temp file
        if (file_exists($tempFile)) {
            unlink($tempFile);
        }
    }
    
    private function createSiteArchive($sourcePath, $tempFile, $user, $domain) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Site directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        $domainName = basename($sourcePath);
        
        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/" . $domainName) . " && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression (level 6 for better speed/ratio balance)
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tempFile) . " --use-compress-program='gzip -6' " . escapeshellarg($userName . "/" . $domainName);
        }
        
        exec($command, $output, $returnCode);
        
        if ($returnCode !== 0) {
            throw new Exception("Failed to create site archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Site archive file is empty or doesn't exist");
        }
    }
    
    private function createUserfilesArchive($sourcePath, $tempFile, $user) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Userfiles directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        
        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/userfiles") . " 2>&1 && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression (level 6 for better speed/ratio balance)
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -czf " . escapeshellarg($tempFile) . " --use-compress-program='gzip -6' " . escapeshellarg($userName . "/userfiles") . " 2>&1";
        }
        
        exec($command, $output, $returnCode);
        
        if ($returnCode !== 0) {
            throw new Exception("Failed to create userfiles archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Userfiles archive file is empty or doesn't exist");
        }
        
    }
    
    private function createUserfilesArchiveZip($sourcePath, $tempFile, $user) {
        if (!is_dir($sourcePath)) {
            throw new Exception("Userfiles directory not found: $sourcePath");
        }
        
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        
        // Create zip archive
        $command = "cd " . escapeshellarg(dirname($userDir)) . " && zip -r " . escapeshellarg($tempFile) . " " . escapeshellarg($userName . "/userfiles") . " 2>&1";
        
        exec($command, $output, $returnCode);
        
        if ($returnCode !== 0) {
            throw new Exception("Failed to create userfiles archive: " . implode("\n", $output));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Userfiles archive file is empty or doesn't exist");
        }
        
    }
    
    private function createDatabaseDump($database, $tempFile) {
        // Validate database name for security (prevent SQL injection and command injection)
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $database)) {
            throw new Exception("Invalid database name format");
        }

        // Get database credentials
        $mysql = new \mysqlmgmt();
        $host = '127.0.0.1';
        $port = 3306;
        $rootUser = 'root';
        $rootPass = defined('MYSQL_PASS') ? trim(MYSQL_PASS) : '';
        
        // First, test if the database exists
        try {
            $testStmt = $this->db->prepare("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?");
            $testStmt->execute([$database]);
            if (!$testStmt->fetch()) {
                throw new Exception("Database '$database' does not exist");
            }
        } catch (Exception $e) {
            throw new Exception("Failed to verify database existence: " . $e->getMessage());
        }
        
        // Create a temporary SQL file first, then compress it
        $tempSqlFile = $tempFile . '.sql';
        
        // Create mysqldump command with proper options for full database recreation
        $passwordArg = $rootPass ? "--password=" . escapeshellarg($rootPass) : "";
        
        $command = "mysqldump -h " . escapeshellarg($host) . 
                  " -P " . escapeshellarg($port) . 
                  " -u " . escapeshellarg($rootUser) . 
                  " " . $passwordArg . 
                  " --single-transaction --routines --triggers --events" .
                  " --add-drop-database --create-options --complete-insert" .
                  " --databases " . escapeshellarg($database) . 
                  " > " . escapeshellarg($tempSqlFile) . " 2>&1";
        
        exec($command, $output, $returnCode);
        
        if ($returnCode !== 0) {
            // Clean up temp file
            if (file_exists($tempSqlFile)) {
                unlink($tempSqlFile);
            }
            
            throw new Exception("Failed to create database dump: " . implode("\n", $output));
        }
        
        // Check if SQL file was created and has content
        if (!file_exists($tempSqlFile) || filesize($tempSqlFile) === 0) {
            throw new Exception("Database dump SQL file is empty or doesn't exist");
        }
        
        // Compress the SQL file - use pigz if available for better performance
        if (shell_exec('which pigz 2>/dev/null')) {
            $gzipCommand = "pigz -6 -c " . escapeshellarg($tempSqlFile) . " > " . escapeshellarg($tempFile);
        } else {
            $gzipCommand = "gzip -6 -c " . escapeshellarg($tempSqlFile) . " > " . escapeshellarg($tempFile);
        }
        exec($gzipCommand, $gzipOutput, $gzipReturn);
        
        // Clean up temp SQL file
        if (file_exists($tempSqlFile)) {
            unlink($tempSqlFile);
        }
        
        if ($gzipReturn !== 0) {
            throw new Exception("Failed to compress database dump: " . implode("\n", $gzipOutput));
        }
        
        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Compressed database dump file is empty or doesn't exist");
        }
    }

    private function createWordPressArchive($sourcePath, $database, $tempFile, $user, $domain) {
        if (!is_dir($sourcePath)) {
            throw new Exception("WordPress site directory not found: $sourcePath");
        }

        // Create temporary directory for organizing the backup content
        // Use random_bytes for cryptographically secure temporary directory name
        $randomSuffix = bin2hex(random_bytes(16));
        $tempWorkDir = $this->tempDir . '/wp_backup_' . $randomSuffix;
        if (!mkdir($tempWorkDir, 0700, true)) { // More restrictive permissions
            throw new Exception("Failed to create temporary work directory");
        }

        try {
            // Step 1: Create site files archive
            $siteArchivePath = $tempWorkDir . '/site.tar.gz';
            $this->createSiteArchiveForWordPress($sourcePath, $siteArchivePath, $user, $domain);

            // Step 2: Create database dump if database exists
            $dbDumpPath = null;
            if (!empty($database)) {
                $dbDumpPath = $tempWorkDir . '/database.sql.gz';
                $this->createDatabaseDump($database, $dbDumpPath);
            }

            // Step 3: Create manifest file with backup information
            $manifestPath = $tempWorkDir . '/backup-manifest.json';
            $manifest = [
                'backup_type' => 'wordpress',
                'domain' => $domain,
                'user' => $user,
                'created_at' => date('Y-m-d H:i:s'),
                'files' => [
                    'site' => 'site.tar.gz',
                    'database' => $database ? 'database.sql.gz' : null
                ],
                'database_name' => $database
            ];
            file_put_contents($manifestPath, json_encode($manifest, JSON_PRETTY_PRINT));

            // Step 4: Create final combined archive
            $filesToArchive = ['backup-manifest.json', 'site.tar.gz'];
            if ($dbDumpPath && file_exists($dbDumpPath)) {
                $filesToArchive[] = 'database.sql.gz';
            }

            // Use tar with gzip compression
            $filesArg = implode(' ', array_map('escapeshellarg', $filesToArchive));
            $command = "cd " . escapeshellarg($tempWorkDir) . " && tar -czf " . escapeshellarg($tempFile) . " " . $filesArg;

            exec($command, $output, $returnCode);

            if ($returnCode !== 0) {
                throw new Exception("Failed to create WordPress archive: " . implode("\n", $output));
            }

            if (!file_exists($tempFile) || filesize($tempFile) === 0) {
                throw new Exception("WordPress archive file is empty or doesn't exist");
            }

        } finally {
            // Clean up temporary work directory
            if (is_dir($tempWorkDir)) {
                $this->removeDirectory($tempWorkDir);
            }
        }
    }

    private function createSiteArchiveForWordPress($sourcePath, $tempFile, $user, $domain) {
        // This is similar to createSiteArchive but creates the archive directly in the specified location
        $userDir = dirname($sourcePath);
        $userName = basename($userDir);
        $domainName = basename($sourcePath);

        // Use pigz for parallel compression if available, otherwise fall back to gzip
        if (shell_exec('which pigz 2>/dev/null')) {
            // Create tar without compression, then compress with pigz
            $tarFile = $tempFile . '.tar';
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -cf " . escapeshellarg($tarFile) . " " . escapeshellarg($userName . "/" . $domainName) . " && pigz -6 -c " . escapeshellarg($tarFile) . " > " . escapeshellarg($tempFile) . " && rm -f " . escapeshellarg($tarFile);
        } else {
            // Fall back to standard tar with gzip compression
            $command = "cd " . escapeshellarg(dirname($userDir)) . " && tar -czf " . escapeshellarg($tempFile) . " " . escapeshellarg($userName . "/" . $domainName);
        }

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new Exception("Failed to create site archive: " . implode("\n", $output));
        }

        if (!file_exists($tempFile) || filesize($tempFile) === 0) {
            throw new Exception("Site archive file is empty or doesn't exist");
        }
    }

    private function removeDirectory($dir) {
        if (!is_dir($dir)) {
            return;
        }

        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }

    private function uploadToStorage($history, $data, $tempFile) {
        $target = new BackupTarget($this->db, $data['target_id']);
        $storage = new BackupStorage($target->getCredentials());
        
        $result = $storage->uploadFile($tempFile, $data['backup_path']);
        
        if (!$result['success']) {
            throw new Exception("Failed to upload backup: " . $result['error']);
        }
        
        // Update backup size
        $history->updateSize($result['size']);
    }
}