#!/usr/bin/env php
<?php
/**
 * WordPress Auto-Update Script
 * 
 * Runs scheduled WordPress auto-updates for sites with auto-update enabled.
 * Creates backups before updates and logs all activities.
 * 
 * Usage: php wordpress-auto-update.php [--dry-run] [--site-id=ID] [--type=core|plugins|themes]
 */

// Add the web-files directory to include path
ini_set('include_path', ini_get('include_path') . PATH_SEPARATOR . '/docker/whp/web');

// Set up CLI-friendly environment variables before including auto-prepend
if (php_sapi_name() === 'cli') {
    $_SERVER['REMOTE_ADDR'] = '127.0.0.1';
    $_SERVER['REQUEST_URI'] = '/cli-script';
    $_SERVER['SERVER_NAME'] = 'localhost';
    $_SERVER['SCRIPT_FILENAME'] = __FILE__;
}

// Include the auto-prepend configuration (needed for database constants)
require_once('/docker/whp/web/auto-prepend.php');

require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('/docker/whp/web/libs/WordPressManager.php');
require_once('/docker/whp/web/libs/BackupEngine.php');

use WHP\WordPressManager;
use WHPBackup\BackupEngine;

class WordPressAutoUpdater {
    private $db;
    private $wpManager;
    private $backupEngine;
    private $dryRun = false;
    private $logFile;
    
    public function __construct($dryRun = false) {
        $this->dryRun = $dryRun;
        $this->logFile = '/var/log/wordpress-auto-update.log';
        
        // Initialize database connection
        $mysql = new mysqlmgmt();
        $this->db = $mysql->getMySQLConnection();
        $this->db->exec("USE whp");
        
        // Initialize managers
        $this->wpManager = new WordPressManager();
        $this->backupEngine = new BackupEngine();
        
        $this->log("WordPress Auto-Updater started" . ($this->dryRun ? " (DRY RUN)" : ""));
    }
    
    /**
     * Log messages to file and optionally to stdout
     */
    private function log($message, $level = 'INFO') {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
        
        // Write to log file
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
        
        // Only output to stdout if running interactively (not from cron)
        if (php_sapi_name() === 'cli' && function_exists('posix_isatty') && posix_isatty(STDOUT)) {
            echo $logMessage;
        } elseif (php_sapi_name() === 'cli') {
            // Fallback for systems without POSIX functions - always output for CLI
            echo $logMessage;
        }
    }
    
    /**
     * Get sites that have auto-updates enabled
     */
    public function getSitesWithAutoUpdates($updateType = null, $siteId = null) {
        $conditions = ["status = 'active'"];
        $params = [];
        
        if ($updateType) {
            switch ($updateType) {
                case 'core':
                    $conditions[] = "auto_update_core = 1";
                    break;
                case 'plugins':
                    $conditions[] = "auto_update_plugins = 1";
                    break;
                case 'themes':
                    $conditions[] = "auto_update_themes = 1";
                    break;
                default:
                    throw new Exception("Invalid update type: {$updateType}");
            }
        } else {
            $conditions[] = "(auto_update_core = 1 OR auto_update_plugins = 1 OR auto_update_themes = 1)";
        }
        
        if ($siteId) {
            $conditions[] = "id = ?";
            $params[] = $siteId;
        }
        
        $sql = "SELECT * FROM wordpress_sites WHERE " . implode(' AND ', $conditions) . " ORDER BY user, domain";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Create backup before update
     */
    private function createBackup($site) {
        if (!$site['auto_backup_before_update']) {
            $this->log("Skipping backup for {$site['domain']} - auto backup disabled", 'INFO');
            return true;
        }
        
        $this->log("Creating backup for {$site['domain']} before update", 'INFO');
        
        if ($this->dryRun) {
            $this->log("DRY RUN: Would create backup for site {$site['id']}", 'INFO');
            return true;
        }
        
        try {
            // Create site backup using the backup engine
            $backupResult = $this->backupEngine->createWordPressBackup(
                $site['id'],
                'auto-update-backup-' . date('Y-m-d-H-i-s'),
                $site['preferred_backup_target_id']
            );
            
            if ($backupResult['success']) {
                $this->log("Backup created successfully for {$site['domain']}: {$backupResult['backup_id']}", 'SUCCESS');
                return true;
            } else {
                $this->log("Backup failed for {$site['domain']}: " . $backupResult['error'], 'ERROR');
                return false;
            }
        } catch (Exception $e) {
            // Check if this is a "no backup target available" error
            if (strpos($e->getMessage(), 'No backup target available') !== false) {
                $this->log("Warning: No backup targets configured for {$site['domain']} - skipping backup and continuing with update", 'WARNING');
                return true; // Continue with update even without backup
            }
            $this->log("Backup exception for {$site['domain']}: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Update WordPress core
     */
    private function updateCore($site) {
        if (!$site['auto_update_core']) {
            return true;
        }
        
        $this->log("Updating WordPress core for {$site['domain']}", 'INFO');
        
        if ($this->dryRun) {
            $this->log("DRY RUN: Would update core for {$site['domain']}", 'INFO');
            return true;
        }
        
        try {
            // Check if update is available first
            $checkResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp core check-update --format=json"
            );
            
            if ($checkResult['success']) {
                $updateInfo = json_decode($checkResult['output'], true);
                if (empty($updateInfo)) {
                    $this->log("WordPress core is already up to date for {$site['domain']}", 'INFO');
                    return true;
                }
            }
            
            // Perform the update
            $updateResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp core update --quiet"
            );
            
            if ($updateResult['success']) {
                $this->log("WordPress core updated successfully for {$site['domain']}", 'SUCCESS');
                return true;
            } else {
                $this->log("WordPress core update failed for {$site['domain']}: " . $updateResult['output'], 'ERROR');
                return false;
            }
        } catch (Exception $e) {
            $this->log("Core update exception for {$site['domain']}: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Update WordPress plugins
     */
    private function updatePlugins($site) {
        if (!$site['auto_update_plugins']) {
            return true;
        }
        
        $this->log("Updating WordPress plugins for {$site['domain']}", 'INFO');
        
        if ($this->dryRun) {
            $this->log("DRY RUN: Would update plugins for {$site['domain']}", 'INFO');
            return true;
        }
        
        try {
            // Check for plugin updates first
            $checkResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp plugin list --update=available --format=count"
            );
            
            if ($checkResult['success'] && trim($checkResult['output']) === '0') {
                $this->log("All plugins are up to date for {$site['domain']}", 'INFO');
                return true;
            }
            
            // Update all plugins
            $updateResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp plugin update --all --quiet"
            );
            
            if ($updateResult['success']) {
                $this->log("WordPress plugins updated successfully for {$site['domain']}", 'SUCCESS');
                return true;
            } else {
                $this->log("WordPress plugins update failed for {$site['domain']}: " . $updateResult['output'], 'ERROR');
                return false;
            }
        } catch (Exception $e) {
            $this->log("Plugin update exception for {$site['domain']}: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Update WordPress themes
     */
    private function updateThemes($site) {
        if (!$site['auto_update_themes']) {
            return true;
        }
        
        $this->log("Updating WordPress themes for {$site['domain']}", 'INFO');
        
        if ($this->dryRun) {
            $this->log("DRY RUN: Would update themes for {$site['domain']}", 'INFO');
            return true;
        }
        
        try {
            // Check for theme updates first
            $checkResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp theme list --update=available --format=count"
            );
            
            if ($checkResult['success'] && trim($checkResult['output']) === '0') {
                $this->log("All themes are up to date for {$site['domain']}", 'INFO');
                return true;
            }
            
            // Update all themes
            $updateResult = $this->wpManager->executeWpCliCommand(
                $site['container_name'],
                $site['user'],
                "wp theme update --all --quiet"
            );
            
            if ($updateResult['success']) {
                $this->log("WordPress themes updated successfully for {$site['domain']}", 'SUCCESS');
                return true;
            } else {
                $this->log("WordPress themes update failed for {$site['domain']}: " . $updateResult['output'], 'ERROR');
                return false;
            }
        } catch (Exception $e) {
            $this->log("Theme update exception for {$site['domain']}: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Process updates for a single site
     * @param array $site The site data
     * @param string|null $updateType Specific update type to run (core|plugins|themes), or null for all
     */
    public function processSite($site, $updateType = null) {
        $this->log("Processing site: {$site['domain']} (ID: {$site['id']})", 'INFO');

        $success = true;

        // Create backup if needed
        if (!$this->createBackup($site)) {
            $this->log("Skipping updates for {$site['domain']} due to backup failure", 'ERROR');
            return false;
        }

        // If a specific update type is specified, only run that type
        // Otherwise, run all enabled update types for the site
        if ($updateType) {
            switch ($updateType) {
                case 'core':
                    if (!$this->updateCore($site)) {
                        $success = false;
                    }
                    break;
                case 'plugins':
                    if (!$this->updatePlugins($site)) {
                        $success = false;
                    }
                    break;
                case 'themes':
                    if (!$this->updateThemes($site)) {
                        $success = false;
                    }
                    break;
            }
        } else {
            // Run all update types that are enabled for the site
            // Update core
            if (!$this->updateCore($site)) {
                $success = false;
            }

            // Update plugins
            if (!$this->updatePlugins($site)) {
                $success = false;
            }

            // Update themes
            if (!$this->updateThemes($site)) {
                $success = false;
            }
        }

        if ($success) {
            $this->log("All updates completed successfully for {$site['domain']}", 'SUCCESS');
        } else {
            $this->log("Some updates failed for {$site['domain']}", 'WARNING');
        }

        return $success;
    }
    
    /**
     * Run auto-updates for all eligible sites
     */
    public function runAutoUpdates($updateType = null, $siteId = null) {
        $sites = $this->getSitesWithAutoUpdates($updateType, $siteId);
        
        if (empty($sites)) {
            $this->log("No sites found with auto-updates enabled", 'INFO');
            return;
        }
        
        $this->log("Found " . count($sites) . " site(s) with auto-updates enabled", 'INFO');
        
        $successCount = 0;
        $failureCount = 0;
        
        foreach ($sites as $site) {
            if ($this->processSite($site, $updateType)) {
                $successCount++;
            } else {
                $failureCount++;
            }

            // Add a small delay between sites to prevent overwhelming the system
            if (!$this->dryRun) {
                sleep(2);
            }
        }
        
        $this->log("Auto-update run completed. Success: {$successCount}, Failures: {$failureCount}", 'INFO');
    }
}

// Parse command line arguments
$options = getopt('', ['dry-run', 'site-id:', 'type:']);
$dryRun = isset($options['dry-run']);
$siteId = $options['site-id'] ?? null;
$updateType = $options['type'] ?? null;

// Validate update type
if ($updateType && !in_array($updateType, ['core', 'plugins', 'themes'])) {
    echo "Error: Invalid update type. Must be one of: core, plugins, themes\n";
    exit(1);
}

try {
    $updater = new WordPressAutoUpdater($dryRun);
    $updater->runAutoUpdates($updateType, $siteId);
} catch (Exception $e) {
    echo "Fatal error: " . $e->getMessage() . "\n";
    exit(1);
}