<?php

namespace WHPBackup;

require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('BackupTarget.php');
require_once('BackupHistory.php');
require_once('BackupStorage.php');

use Exception;

class BackupEngine {
    private $db;
    private $tempDir;
    
    public function __construct() {
        $mysql = new \mysqlmgmt();
        $this->db = $mysql->getMySQLConnection();
        $this->db->exec("USE whp");
        $this->tempDir = '/docker/tmp';
        
        if (!is_dir($this->tempDir)) {
            mkdir($this->tempDir, 0755, true);
        }
    }
    
    public function createSiteBackup($user, $domain, $targetId) {
        try {
            $sitePath = "/docker/users/$user/$domain";

            if (!is_dir($sitePath)) {
                return [
                    'success' => false,
                    'error' => "Site directory not found: $sitePath"
                ];
            }

            $target = new BackupTarget($this->db, $targetId);
            if (!$target->canAccess($user)) {
                return [
                    'success' => false,
                    'error' => "Access denied to backup target"
                ];
            }

            $backupName = "{$user}_{$domain}_site_" . date('Y-m-d_H-i-s');
            $tempFile = $this->tempDir . '/' . $backupName . '.tar.gz';
            $remotePath = "$user/sites/$domain/" . $backupName . '.tar.gz';

            // Create backup history record
            $history = new BackupHistory($this->db);
            $result = $history->create([
                'target_id' => $targetId,
                'user' => $user,
                'backup_type' => 'site',
                'backup_name' => $backupName,
                'backup_path' => $remotePath,
                'status' => 'pending',
                'metadata' => [
                    'domain' => $domain,
                    'source_path' => $sitePath
                ]
            ]);

            if (!$result) {
                return [
                    'success' => false,
                    'error' => "Failed to create backup record"
                ];
            }

            // Trigger background processing
            $this->triggerBackgroundProcessing();

            return [
                'success' => true,
                'backup_id' => $history->getId(),
                'backup_name' => $backupName,
                'status' => 'pending'
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function createUserfilesBackup($user, $targetId) {
        try {
            $userfilesPath = "/docker/users/$user/userfiles";

            if (!is_dir($userfilesPath)) {
                return [
                    'success' => false,
                    'error' => "Userfiles directory not found: $userfilesPath"
                ];
            }

            $target = new BackupTarget($this->db, $targetId);
            if (!$target->canAccess($user)) {
                return [
                    'success' => false,
                    'error' => "Access denied to backup target"
                ];
            }

            $backupName = "{$user}_userfiles_" . date('Y-m-d_H-i-s');
            $tempFile = $this->tempDir . '/' . $backupName . '.zip';
            $remotePath = "$user/userfiles/" . $backupName . '.zip';

            // Create backup history record
            $history = new BackupHistory($this->db);
            $result = $history->create([
                'target_id' => $targetId,
                'user' => $user,
                'backup_type' => 'userfiles',
                'backup_name' => $backupName,
                'backup_path' => $remotePath,
                'status' => 'pending',
                'metadata' => [
                    'source_path' => $userfilesPath
                ]
            ]);

            if (!$result) {
                return [
                    'success' => false,
                    'error' => "Failed to create backup record"
                ];
            }

            // Trigger background processing
            $this->triggerBackgroundProcessing();

            return [
                'success' => true,
                'backup_id' => $history->getId(),
                'backup_name' => $backupName,
                'status' => 'pending'
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function createDatabaseBackup($user, $database, $targetId) {
        try {
            $target = new BackupTarget($this->db, $targetId);
            if (!$target->canAccess($user)) {
                return [
                    'success' => false,
                    'error' => "Access denied to backup target"
                ];
            }

            // Ensure database name is properly formatted and not empty
            if (empty($database)) {
                return [
                    'success' => false,
                    'error' => "Database name cannot be empty"
                ];
            }

            // Log database backup creation for debugging
            error_log("Creating database backup - User: $user, Database: $database, Target: $targetId");

            $backupName = "{$user}_{$database}_db_" . date('Y-m-d_H-i-s');
            $tempFile = $this->tempDir . '/' . $backupName . '.sql.gz';
            $remotePath = "$user/databases/" . $backupName . '.sql.gz';

            // Create backup history record
            $history = new BackupHistory($this->db);
            $result = $history->create([
                'target_id' => $targetId,
                'user' => $user,
                'backup_type' => 'database',
                'backup_name' => $backupName,
                'backup_path' => $remotePath,
                'status' => 'pending',
                'metadata' => [
                    'database' => $database
                ]
            ]);

            if (!$result) {
                return [
                    'success' => false,
                    'error' => "Failed to create backup record"
                ];
            }

            // Trigger background processing
            $this->triggerBackgroundProcessing();

            return [
                'success' => true,
                'backup_id' => $history->getId(),
                'backup_name' => $backupName,
                'status' => 'pending'
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function downloadBackup($backupId, $user) {
        $history = new BackupHistory($this->db, $backupId);
        
        if (!$history->canAccess($user)) {
            throw new Exception("Access denied to backup");
        }
        
        $data = $history->getData();
        if ($data['status'] !== 'completed') {
            throw new Exception("Backup is not available for download");
        }
        
        $target = new BackupTarget($this->db, $data['target_id']);
        $storage = new BackupStorage($target->getCredentials());
        
        // Get backup info
        $info = $storage->getFileInfo($data['backup_path']);
        if (!$info['success'] || !$info['exists']) {
            throw new Exception("Backup file not found in storage");
        }
        
        // Set headers for download
        // Use the actual filename from the backup path
        $filename = basename($data['backup_path']);
        
        // Clean any output buffers to ensure no extra content
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . $info['size']);
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        
        
        // Stream the file directly to output
        $result = $storage->streamFile($data['backup_path']);
        
        if (!$result['success']) {
            throw new Exception("Failed to stream backup: " . $result['error']);
        }
        
        return true;
    }
    
    public function deleteBackup($backupId, $user) {
        $history = new BackupHistory($this->db, $backupId);
        
        if (!$history->canAccess($user)) {
            throw new Exception("Access denied to backup");
        }
        
        $data = $history->getData();
        
        // Handle different backup statuses
        if ($data['status'] === 'completed') {
            // Delete completed backups from storage
            $target = new BackupTarget($this->db, $data['target_id']);
            $storage = new BackupStorage($target->getCredentials());
            
            // Delete from storage
            $result = $storage->deleteFile($data['backup_path']);
            
            if (!$result['success']) {
                throw new Exception("Failed to delete backup from storage: " . $result['error']);
            }
        } elseif ($data['status'] === 'failed') {
            // For failed backups, just remove from database
            // No need to delete from storage as upload likely failed
        } elseif ($data['status'] === 'running' || $data['status'] === 'pending') {
            // Allow deletion of stuck running/pending backups
            // These likely failed or got stuck, so clean them up
        }
        
        // Mark as deleted in database
        $history->delete();
        
        return true;
    }
    
    public function getUserSites($user) {
        $userDir = "/docker/users/$user";
        $sites = [];
        
        if (is_dir($userDir)) {
            $directories = glob($userDir . '/*', GLOB_ONLYDIR);
            foreach ($directories as $dir) {
                $dirname = basename($dir);
                // Skip userfiles directory
                if ($dirname !== 'userfiles') {
                    $sites[] = $dirname;
                }
            }
        }
        
        return $sites;
    }
    
    public function getUserDatabases($user) {
        try {
            $stmt = $this->db->prepare("SHOW DATABASES");
            $stmt->execute();
            
            $databases = [];
            while ($row = $stmt->fetch(\PDO::FETCH_ASSOC)) {
                $dbName = $row['Database'];
                // Only include databases that belong to the user
                if (strpos($dbName, $user . '_') === 0) {
                    $databases[] = $dbName;
                }
            }
            
            return $databases;
            
        } catch (Exception $e) {
            throw new Exception("Failed to get user databases: " . $e->getMessage());
        }
    }
    
    public function cleanupOldBackups($targetId = null) {
        $targets = [];
        
        if ($targetId) {
            $targets[] = new BackupTarget($this->db, $targetId);
        } else {
            $targetList = BackupTarget::listTargets($this->db);
            foreach ($targetList as $targetData) {
                $targets[] = new BackupTarget($this->db, $targetData['id']);
            }
        }
        
        foreach ($targets as $target) {
            $targetData = $target->getData();
            $retentionDays = $targetData['retention_days'];
            $maxBackups = $targetData['max_backups'];
            
            // Delete old backups based on retention days
            if ($retentionDays > 0) {
                $oldBackups = BackupHistory::getOldBackups($this->db, $target->getId(), $retentionDays);
                foreach ($oldBackups as $backup) {
                    try {
                        $this->deleteBackup($backup['id'], $backup['user']);
                    } catch (Exception $e) {
                        // Log error but continue
                    }
                }
            }
            
            // Delete excess backups based on max count
            if ($maxBackups > 0) {
                $backupTypes = ['site', 'userfiles', 'database'];
                $users = $this->getAllUsers();
                
                foreach ($users as $user) {
                    foreach ($backupTypes as $type) {
                        $excessBackups = BackupHistory::getExcessBackups($this->db, $target->getId(), $user, $type, $maxBackups);
                        foreach ($excessBackups as $backup) {
                            try {
                                $this->deleteBackup($backup['id'], $backup['user']);
                            } catch (Exception $e) {
                                // Log error but continue
                            }
                        }
                    }
                }
            }
        }
    }
    
    public function cleanupScheduledBackups($targetId, $user, $backupType, $maxRetention) {
        // Get backups for this specific user, target, and type
        $filters = [
            'user' => $user,
            'target_id' => $targetId,
            'backup_type' => $backupType,
            'status' => 'completed'
        ];
        
        $backups = BackupHistory::listBackups($this->db, $filters);
        
        // Sort by creation date (newest first)
        usort($backups, function($a, $b) {
            return strtotime($b['started_at']) - strtotime($a['started_at']);
        });
        
        // Delete excess backups
        for ($i = $maxRetention; $i < count($backups); $i++) {
            try {
                $this->deleteBackup($backups[$i]['id'], $user);
            } catch (Exception $e) {
                // Log error but continue
                error_log("Failed to delete scheduled backup {$backups[$i]['id']}: " . $e->getMessage());
            }
        }
    }
    
    private function getAllUsers() {
        $users = [];
        $userDir = '/docker/users';
        
        if (is_dir($userDir)) {
            $directories = glob($userDir . '/*', GLOB_ONLYDIR);
            foreach ($directories as $dir) {
                $users[] = basename($dir);
            }
        }
        
        return $users;
    }
    
    private function triggerBackgroundProcessing() {
        // Trigger background processing asynchronously
        $scriptPath = dirname(__DIR__) . '/scripts/backup-worker.php';
        if (file_exists($scriptPath)) {
            // Run in background without blocking the response
            exec("php " . escapeshellarg($scriptPath) . " > /dev/null 2>&1 &");
        }
    }
    
    /**
     * Create a combined WordPress backup (site files + database)
     */
    public function createWordPressBackup($siteId, $backupName = null, $targetId = null) {
        // Get WordPress site information
        $stmt = $this->db->prepare("SELECT * FROM wordpress_sites WHERE id = ?");
        $stmt->execute([$siteId]);
        $site = $stmt->fetch();
        
        if (!$site) {
            throw new Exception("WordPress site not found");
        }
        
        // Use default backup target if none specified
        if (!$targetId) {
            $targetId = $site['preferred_backup_target_id'];
            if (!$targetId) {
                // Get user's default backup target
                $stmt = $this->db->prepare("SELECT id FROM backup_targets WHERE owner = ? ORDER BY id LIMIT 1");
                $stmt->execute([$site['user']]);
                $target = $stmt->fetch();
                if (!$target) {
                    // Try to get any available backup target (including global ones)
                    $stmt = $this->db->prepare("SELECT id FROM backup_targets WHERE owner = ? OR is_global = 1 ORDER BY id LIMIT 1");
                    $stmt->execute([$site['user']]);
                    $target = $stmt->fetch();
                    if (!$target) {
                        throw new Exception("No backup target available. Please configure a backup target before importing WordPress sites with auto-backup enabled.");
                    }
                }
                $targetId = $target['id'];
            }
        }
        
        // Generate backup name if not provided
        if (!$backupName) {
            $backupName = "wp_{$site['domain']}_" . date('Y-m-d_H-i-s');
        }
        
        try {
            // Create site backup (files)
            $siteBackupResult = $this->createSiteBackup($site['user'], $site['domain'], $targetId);

            // Create database backup only if database name is available
            $dbBackupResult = null;
            if (!empty($site['database_name'])) {
                $dbBackupResult = $this->createDatabaseBackup($site['user'], $site['database_name'], $targetId);
            } else {
                // Log warning but continue with file backup only
                error_log("Warning: WordPress site {$site['domain']} has no database configured, creating file backup only");
                $dbBackupResult = ['success' => true, 'backup_id' => null, 'size' => 0];
            }
            
            // Create combined backup record
            $history = new BackupHistory($this->db);
            $combinedBackupId = $history->create([
                'target_id' => $targetId,
                'user' => $site['user'],
                'backup_type' => 'wordpress',
                'backup_name' => $backupName,
                'backup_path' => "wordpress/{$site['domain']}/",
                'backup_size' => ($siteBackupResult['size'] ?? 0) + ($dbBackupResult['size'] ?? 0),
                'status' => 'completed',
                'metadata' => [
                    'wordpress_site_id' => $siteId,
                    'domain' => $site['domain'],
                    'database' => $site['database_name'] ?? null,
                    'site_backup_id' => $siteBackupResult['backup_id'] ?? null,
                    'db_backup_id' => $dbBackupResult['backup_id'] ?? null,
                    'wp_version' => $site['wp_version'] ?? null,
                    'source_path' => "/docker/users/{$site['user']}/{$site['domain']}"
                ]
            ]);
            
            return [
                'success' => true,
                'backup_id' => $combinedBackupId,
                'site_backup_id' => $siteBackupResult['backup_id'] ?? null,
                'db_backup_id' => $dbBackupResult['backup_id'] ?? null,
                'message' => "WordPress backup created successfully"
            ];
            
        } catch (Exception $e) {
            error_log("WordPress backup failed for site {$siteId}: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function __destruct() {
        // Clean up temp directory
        if (is_dir($this->tempDir)) {
            $files = glob($this->tempDir . '/*');
            foreach ($files as $file) {
                if (is_file($file) && (time() - filemtime($file)) > 3600) {
                    unlink($file);
                }
            }
        }
    }
}