<?php

class security_manager {
    
    private $db_path = '/docker/whp/sql/security.db';
    private $db;
    private $max_failed_attempts = 5;
    private $block_duration = 3600; // 1 hour in seconds
    private $whitelist_ips = array('127.0.0.1', '::1'); // Localhost
    
    public function __construct() {
        $this->init_database();
    }
    
    private function init_database() {
        if (!file_exists($this->db_path)) {
            $this->db = new SQLite3($this->db_path);
            $this->db->busyTimeout(6000);
            
            // Create failed login attempts table
            $this->db->exec('CREATE TABLE failed_logins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT NOT NULL,
                username TEXT NOT NULL,
                attempt_time INTEGER NOT NULL,
                user_agent TEXT
            )');
            
            // Create blocked IPs table
            $this->db->exec('CREATE TABLE blocked_ips (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT UNIQUE NOT NULL,
                block_time INTEGER NOT NULL,
                unblock_time INTEGER NOT NULL,
                reason TEXT,
                failed_attempts INTEGER DEFAULT 0,
                manual_block INTEGER DEFAULT 0
            )');
            
            // Create whitelist IPs table
            $this->db->exec('CREATE TABLE whitelist_ips (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT UNIQUE NOT NULL,
                description TEXT,
                added_time INTEGER NOT NULL
            )');
            
            // Create successful logins table
            $this->db->exec('CREATE TABLE successful_logins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT NOT NULL,
                ip_address TEXT NOT NULL,
                login_time INTEGER NOT NULL,
                user_agent TEXT
            )');
            
            // Create indexes for better performance
            $this->db->exec('CREATE INDEX idx_failed_logins_ip ON failed_logins(ip_address)');
            $this->db->exec('CREATE INDEX idx_failed_logins_time ON failed_logins(attempt_time)');
            $this->db->exec('CREATE INDEX idx_blocked_ips_time ON blocked_ips(unblock_time)');
            $this->db->exec('CREATE INDEX idx_successful_logins_user ON successful_logins(username)');
            $this->db->exec('CREATE INDEX idx_successful_logins_time ON successful_logins(login_time)');
        } else {
            $this->db = new SQLite3($this->db_path);
            $this->db->busyTimeout(6000);
            
            // Check if successful_logins table exists, create if not (for existing databases)
            $this->ensure_successful_logins_table();
        }
    }
    
    private function ensure_successful_logins_table() {
        // Check if successful_logins table exists
        $result = $this->db->query("SELECT name FROM sqlite_master WHERE type='table' AND name='successful_logins'");
        if (!$result->fetchArray()) {
            // Create the table and indexes
            $this->db->exec('CREATE TABLE successful_logins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT NOT NULL,
                ip_address TEXT NOT NULL,
                login_time INTEGER NOT NULL,
                user_agent TEXT
            )');
            
            $this->db->exec('CREATE INDEX idx_successful_logins_user ON successful_logins(username)');
            $this->db->exec('CREATE INDEX idx_successful_logins_time ON successful_logins(login_time)');
        }
    }
    
    public function record_failed_login($ip_address, $username, $user_agent = '') {
        // Don't track whitelisted IPs
        if ($this->is_ip_whitelisted($ip_address)) {
            return true;
        }
        
        $stmt = $this->db->prepare('INSERT INTO failed_logins (ip_address, username, attempt_time, user_agent) VALUES (:ip, :user, :time, :agent)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':user', $username, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->bindValue(':agent', $user_agent, SQLITE3_TEXT);
        $stmt->execute();
        
        // Check if IP should be blocked
        $this->check_and_block_ip($ip_address);
        
        return true;
    }
    
    public function record_successful_login($ip_address, $username = '', $user_agent = '') {
        // Clear failed attempts for this IP on successful login
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        // Record successful login if username is provided
        if (!empty($username)) {
            // Ensure table exists before trying to use it
            $this->ensure_successful_logins_table();
            
            // Insert new login record
            $stmt = $this->db->prepare('INSERT INTO successful_logins (username, ip_address, login_time, user_agent) VALUES (:user, :ip, :time, :agent)');
            if ($stmt) {
                $stmt->bindValue(':user', $username, SQLITE3_TEXT);
                $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
                $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
                $stmt->bindValue(':agent', $user_agent, SQLITE3_TEXT);
                $stmt->execute();
                
                // Keep only the last 3 logins for this user
                $stmt = $this->db->prepare('DELETE FROM successful_logins WHERE username = :user AND id NOT IN (
                    SELECT id FROM successful_logins WHERE username = :user ORDER BY login_time DESC LIMIT 3
                )');
                if ($stmt) {
                    $stmt->bindValue(':user', $username, SQLITE3_TEXT);
                    $stmt->execute();
                }
            }
        }
        
        return true;
    }
    
    private function check_and_block_ip($ip_address) {
        // Count recent failed attempts
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM failed_logins WHERE ip_address = :ip AND attempt_time > :time');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time() - 3600, SQLITE3_INTEGER); // Last hour
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        if ($row['count'] >= $this->max_failed_attempts) {
            $this->block_ip($ip_address, 'Too many failed login attempts', $row['count']);
        }
    }
    
    public function block_ip($ip_address, $reason = 'Manual block', $failed_attempts = 0, $indefinite = false, $skip_service_blocks = false) {
        // Don't block whitelisted IPs
        if ($this->is_ip_whitelisted($ip_address)) {
            return false;
        }
        
        // Check if IP is already blocked in security manager to prevent duplicates
        if ($this->is_ip_blocked($ip_address)) {
            return true; // Already blocked, return success
        }
        
        // Store in local database
        $unblock_time = $indefinite ? 0 : (time() + $this->block_duration); // 0 means indefinite/manual unblock only
        $stmt = $this->db->prepare('INSERT OR REPLACE INTO blocked_ips (ip_address, block_time, unblock_time, reason, failed_attempts, manual_block) VALUES (:ip, :block_time, :unblock_time, :reason, :attempts, :manual)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':block_time', time(), SQLITE3_INTEGER);
        $stmt->bindValue(':unblock_time', $unblock_time, SQLITE3_INTEGER);
        $stmt->bindValue(':reason', $reason, SQLITE3_TEXT);
        $stmt->bindValue(':attempts', $failed_attempts, SQLITE3_INTEGER);
        $stmt->bindValue(':manual', ($reason === 'Manual block') ? 1 : 0, SQLITE3_INTEGER);
        $stmt->execute();
        
        // Also block in HAProxy for full service-wide blocking
        try {
            require_once('/docker/whp/web/libs/haproxy_manager.php');
            $haproxy = new haproxy_manager();
            $haproxy->block_ip($ip_address, $reason, 'WHP Security Manager');
        } catch (Exception $e) {
            error_log("Failed to block IP in HAProxy: " . $e->getMessage());
            // Continue even if HAProxy blocking fails - local block is still active
        }
        
        // Also block in SSH/FTP services via log monitor (only if not called from service blocks to prevent loops)
        if (!$skip_service_blocks) {
            try {
                require_once('/docker/whp/web/libs/log_monitor_manager.php');
                $log_monitor = new log_monitor_manager();
                // Block on both SSH and FTP services
                $log_monitor->block_service_ip('ssh', $ip_address, $reason, $failed_attempts);
                $log_monitor->block_service_ip('ftp', $ip_address, $reason, $failed_attempts);
            } catch (Exception $e) {
                error_log("Failed to block IP in SSH/FTP services: " . $e->getMessage());
                // Continue even if service blocking fails
            }
        }
        
        return true;
    }
    
    public function unblock_ip($ip_address) {
        // Remove from local database
        $stmt = $this->db->prepare('DELETE FROM blocked_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        // Also clear failed attempts for this IP
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        // Also unblock from HAProxy
        try {
            require_once('/docker/whp/web/libs/haproxy_manager.php');
            $haproxy = new haproxy_manager();
            $haproxy->unblock_ip($ip_address);
        } catch (Exception $e) {
            error_log("Failed to unblock IP from HAProxy: " . $e->getMessage());
            // Continue even if HAProxy unblocking fails
        }
        
        // Also unblock from SSH/FTP services
        try {
            require_once('/docker/whp/web/libs/log_monitor_manager.php');
            $log_monitor = new log_monitor_manager();
            // Unblock from both SSH and FTP services
            $log_monitor->unblock_service_ip('ssh', $ip_address);
            $log_monitor->unblock_service_ip('ftp', $ip_address);
        } catch (Exception $e) {
            error_log("Failed to unblock IP from SSH/FTP services: " . $e->getMessage());
            // Continue even if service unblocking fails
        }
        
        return true;
    }
    
    public function is_ip_blocked($ip_address) {
        // Check if IP is whitelisted
        if ($this->is_ip_whitelisted($ip_address)) {
            return false;
        }
        
        // Check if IP is blocked
        $stmt = $this->db->prepare('SELECT * FROM blocked_ips WHERE ip_address = :ip AND (unblock_time > :time OR unblock_time = 0)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        if ($row) {
            // Auto-unblock if time has expired (but not for indefinite blocks with unblock_time = 0)
            if ($row['unblock_time'] > 0 && $row['unblock_time'] <= time()) {
                $this->unblock_ip($ip_address);
                return false;
            }
            return $row;
        }
        
        return false;
    }
    
    public function is_ip_whitelisted($ip_address) {
        // Check built-in whitelist
        if (in_array($ip_address, $this->whitelist_ips)) {
            return true;
        }
        
        // Check database whitelist
        $stmt = $this->db->prepare('SELECT * FROM whitelist_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        return $row !== false;
    }
    
    public function add_whitelist_ip($ip_address, $description = '') {
        $stmt = $this->db->prepare('INSERT OR REPLACE INTO whitelist_ips (ip_address, description, added_time) VALUES (:ip, :desc, :time)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':desc', $description, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->execute();
        
        return true;
    }
    
    public function remove_whitelist_ip($ip_address) {
        $stmt = $this->db->prepare('DELETE FROM whitelist_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        return true;
    }
    
    public function get_failed_attempts($ip_address, $hours = 1) {
        $stmt = $this->db->prepare('SELECT * FROM failed_logins WHERE ip_address = :ip AND attempt_time > :time ORDER BY attempt_time DESC');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time() - ($hours * 3600), SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        $attempts = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $attempts[] = $row;
        }
        
        return $attempts;
    }
    
    public function get_blocked_ips() {
        $stmt = $this->db->prepare('SELECT * FROM blocked_ips WHERE unblock_time > :time OR unblock_time = 0 ORDER BY block_time DESC');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        $blocked = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $blocked[] = $row;
        }
        
        return $blocked;
    }
    
    public function get_whitelist_ips() {
        $stmt = $this->db->prepare('SELECT * FROM whitelist_ips ORDER BY added_time DESC');
        $result = $stmt->execute();
        
        $whitelist = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $whitelist[] = $row;
        }
        
        return $whitelist;
    }
    
    public function cleanup_expired_blocks() {
        $stmt = $this->db->prepare('DELETE FROM blocked_ips WHERE unblock_time <= :time');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->execute();
        
        // Clean up old failed attempts (older than 24 hours)
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE attempt_time <= :time');
        $stmt->bindValue(':time', time() - 86400, SQLITE3_INTEGER);
        $stmt->execute();
        
        // Clean up old permission logs (older than 90 days)
        if (file_exists('/docker/whp/sql/permissions.db')) {
            require_once('/docker/whp/web/libs/permission_manager.php');
            $PermManager = new permission_manager();
            $PermManager->cleanup_old_logs(90);
        }
        
        return true;
    }
    
    public function get_last_logins($username) {
        $this->ensure_successful_logins_table();
        
        $stmt = $this->db->prepare('SELECT * FROM successful_logins WHERE username = :user ORDER BY login_time DESC LIMIT 3');
        if (!$stmt) {
            return array(); // Return empty array if table doesn't exist
        }
        
        $stmt->bindValue(':user', $username, SQLITE3_TEXT);
        $result = $stmt->execute();
        
        $logins = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $logins[] = $row;
        }
        
        return $logins;
    }
    
    public function get_all_recent_logins($limit = 10) {
        $this->ensure_successful_logins_table();
        
        $stmt = $this->db->prepare('SELECT * FROM successful_logins ORDER BY login_time DESC LIMIT :limit');
        if (!$stmt) {
            return array(); // Return empty array if table doesn't exist
        }
        
        $stmt->bindValue(':limit', $limit, SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        $logins = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $logins[] = $row;
        }
        
        return $logins;
    }
    
    public function get_security_stats() {
        // Get current blocked IPs count (including indefinite blocks)
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM blocked_ips WHERE unblock_time > :time OR unblock_time = 0');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        $blocked_count = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        // Get failed attempts in last 24 hours from both old and new tables
        $cutoff_time = time() - 86400;
        
        // Get from old failed_logins table
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM failed_logins WHERE attempt_time > :time');
        $stmt->bindValue(':time', $cutoff_time, SQLITE3_INTEGER);
        $result = $stmt->execute();
        $failed_24h_old = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        // Get from new service_failed_attempts table (if it exists)
        $failed_24h_new = 0;
        try {
            $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM service_failed_attempts WHERE attempt_time > :time');
            $stmt->bindValue(':time', $cutoff_time, SQLITE3_INTEGER);
            $result = $stmt->execute();
            $failed_24h_new = $result->fetchArray(SQLITE3_ASSOC)['count'];
        } catch (Exception $e) {
            // Table might not exist in older installations
        }
        
        $failed_24h = $failed_24h_old + $failed_24h_new;
        
        // Get exemption list count
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM whitelist_ips');
        $result = $stmt->execute();
        $exemption_count = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        return array(
            'blocked_ips' => $blocked_count,
            'failed_attempts_24h' => $failed_24h,
            'exemption_ips' => $exemption_count,
            'whitelist_ips' => $exemption_count // Backward compatibility
        );
    }
    
    // New exemption terminology methods (for clearer semantics)
    
    /**
     * Adds an IP address to the exemption list (modern terminology)
     * 
     * Exempted IPs are never blocked by any monitoring system. This method
     * provides modern, clear terminology as an alias for add_whitelist_ip().
     * 
     * @param string $ip_address IPv4 address to exempt from blocking
     * @param string $description Optional description for the exemption
     * @return bool True on success, false on failure
     * @since 2025.08.73
     * 
     * @example
     * $security->add_exemption_ip('192.168.1.100', 'Office gateway');
     */
    public function add_exemption_ip($ip_address, $description = '') {
        return $this->add_whitelist_ip($ip_address, $description);
    }
    
    /**
     * Removes an IP address from the exemption list (modern terminology)
     * 
     * Removes IP from exemption list, allowing it to be blocked by monitoring
     * systems. Provides modern, clear terminology as alias for remove_whitelist_ip().
     * 
     * @param string $ip_address IPv4 address to remove from exemption list
     * @return bool True on success, false on failure
     * @since 2025.08.73
     * 
     * @example
     * $security->remove_exemption_ip('192.168.1.100');
     */
    public function remove_exemption_ip($ip_address) {
        return $this->remove_whitelist_ip($ip_address);
    }
    
    /**
     * Retrieves all IP addresses in the exemption list (modern terminology)
     * 
     * Returns array of exempted IP addresses with descriptions. Provides
     * modern, clear terminology as alias for get_whitelist_ips().
     * 
     * @return array Array of exemption records with ip_address and description
     * @since 2025.08.73
     * 
     * @example
     * $exempted = $security->get_exemption_ips();
     * foreach ($exempted as $exempt) {
     *     echo $exempt['ip_address'] . ' - ' . $exempt['description'];
     * }
     */
    public function get_exemption_ips() {
        return $this->get_whitelist_ips();
    }
    
    /**
     * Checks if an IP address is in the exemption list (modern terminology)
     * 
     * Returns true if IP is exempted from blocking. Provides modern, clear
     * terminology as alias for is_ip_whitelisted().
     * 
     * @param string $ip_address IPv4 address to check
     * @return bool True if IP is exempted, false otherwise
     * @since 2025.08.73
     * 
     * @example
     * if ($security->is_ip_exempted('192.168.1.100')) {
     *     echo "IP is exempted from blocking";
     * }
     */
    public function is_ip_exempted($ip_address) {
        return $this->is_ip_whitelisted($ip_address);
    }
    
    public function __destruct() {
        if ($this->db) {
            $this->db->close();
        }
    }
} 