#!/usr/bin/env bash

# PowerDNS Password Manager for WHP
# This script provides secure password management for PowerDNS services

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

PASSWORDS_FILE="/docker/powerdns/config/passwords.conf"
CONFIG_FILE="/docker/powerdns/config/whp-integration.conf"

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}This script must be run as root${NC}"
   exit 1
fi

# Check if passwords file exists
if [[ ! -f "$PASSWORDS_FILE" ]]; then
    echo -e "${RED}PowerDNS passwords file not found: $PASSWORDS_FILE${NC}"
    echo -e "${YELLOW}Please run the PowerDNS setup script first${NC}"
    exit 1
fi

# Function to get password from file
get_password() {
    local key=$1
    local password=$(grep "^${key}=" "$PASSWORDS_FILE" | cut -d'=' -f2)
    echo "$password"
}

# Function to update password
update_password() {
    local key=$1
    local new_password=$2
    
    # Generate new password if not provided
    if [[ -z "$new_password" ]]; then
        new_password=$(tr -dc 'A-Za-z0-9!@#$%^&*(-_=+)' </dev/urandom | head -c 32)
    fi
    
    # Update password in file
    sed -i "s/^${key}=.*/${key}=${new_password}/" "$PASSWORDS_FILE"
    
    echo -e "${GREEN}Updated ${key} password${NC}"
    return 0
}

# Function to regenerate all passwords
regenerate_all_passwords() {
    echo -e "${BLUE}Regenerating all PowerDNS passwords...${NC}"
    
    # Generate new passwords
    local new_pdns_db_password=$(tr -dc 'A-Za-z0-9!@#$%^&*(-_=+)' </dev/urandom | head -c 32)
    local new_pdns_admin_db_password=$(tr -dc 'A-Za-z0-9!@#$%^&*(-_=+)' </dev/urandom | head -c 32)
    local new_api_key=$(tr -dc 'A-Za-z0-9' </dev/urandom | head -c 32)
    local new_secret_key=$(tr -dc 'A-Za-z0-9!@#$%^&*(-_=+)' </dev/urandom | head -c 50)
    
    # Get MySQL root password
    local mysql_root_password=$(grep password /root/.my.cnf | cut -d= -f2- | tr -d ' ')
    
    # Update passwords in file
    sed -i "s/^PDNS_DB_PASSWORD=.*/PDNS_DB_PASSWORD=${new_pdns_db_password}/" "$PASSWORDS_FILE"
    sed -i "s/^PDNS_ADMIN_DB_PASSWORD=.*/PDNS_ADMIN_DB_PASSWORD=${new_pdns_admin_db_password}/" "$PASSWORDS_FILE"
    sed -i "s/^PDNS_API_KEY=.*/PDNS_API_KEY=${new_api_key}/" "$PASSWORDS_FILE"
    sed -i "s/^SECRET_KEY=.*/SECRET_KEY=${new_secret_key}/" "$PASSWORDS_FILE"
    
    # Update MySQL user passwords
    mysql -h 127.0.0.1 -u root -p"${mysql_root_password}" -e "ALTER USER 'pdns'@'%' IDENTIFIED BY '${new_pdns_db_password}';"
    mysql -h 127.0.0.1 -u root -p"${mysql_root_password}" -e "ALTER USER 'pdns_admin'@'%' IDENTIFIED BY '${new_pdns_admin_db_password}';"
    mysql -h 127.0.0.1 -u root -p"${mysql_root_password}" -e "FLUSH PRIVILEGES;"
    
    # Update PowerDNS configuration
    sed -i "s/^gmysql-password=.*/gmysql-password=${new_pdns_db_password}/" /docker/powerdns/config/pdns.conf
    
    # Restart PowerDNS services
    echo -e "${BLUE}Restarting PowerDNS services...${NC}"
    docker restart powerdns 2>/dev/null || true
    
    echo -e "${GREEN}All passwords regenerated successfully!${NC}"
    echo -e "${YELLOW}Please update any external configurations that use these passwords.${NC}"
}

# Function to show current passwords
show_passwords() {
    echo -e "${BLUE}Current PowerDNS Passwords:${NC}"
    echo ""
    
    local pdns_db_password=$(get_password "PDNS_DB_PASSWORD")
    local pdns_admin_db_password=$(get_password "PDNS_ADMIN_DB_PASSWORD")
    local api_key=$(get_password "PDNS_API_KEY")
    local secret_key=$(get_password "SECRET_KEY")
    
    echo -e "PowerDNS Database Password: ${YELLOW}${pdns_db_password}${NC}"
    echo -e "PowerDNS-Admin Database Password: ${YELLOW}${pdns_admin_db_password}${NC}"
    echo -e "PowerDNS API Key: ${YELLOW}${api_key}${NC}"
    echo -e "PowerDNS-Admin Secret Key: ${YELLOW}${secret_key}${NC}"
    echo ""
    echo -e "${YELLOW}Note: These passwords are stored in: $PASSWORDS_FILE${NC}"
}

# Function to show configuration status
show_status() {
    echo -e "${BLUE}PowerDNS Configuration Status:${NC}"
    echo ""
    
    # Check if files exist
    if [[ -f "$PASSWORDS_FILE" ]]; then
        echo -e "Passwords file: ${GREEN}✓ Found${NC}"
    else
        echo -e "Passwords file: ${RED}✗ Missing${NC}"
    fi
    
    if [[ -f "$CONFIG_FILE" ]]; then
        echo -e "Integration config: ${GREEN}✓ Found${NC}"
    else
        echo -e "Integration config: ${RED}✗ Missing${NC}"
    fi
    
    if [[ -f "/docker/powerdns/config/pdns.conf" ]]; then
        echo -e "PowerDNS config: ${GREEN}✓ Found${NC}"
    else
        echo -e "PowerDNS config: ${RED}✗ Missing${NC}"
    fi
    
    # Check if containers are running
    if docker ps -q -f name=powerdns | grep -q .; then
        echo -e "PowerDNS container: ${GREEN}✓ Running${NC}"
    else
        echo -e "PowerDNS container: ${RED}✗ Not running${NC}"
    fi
    
    # Check API connectivity
    local api_key=$(get_password "PDNS_API_KEY")
    if curl -s -H "X-API-Key: ${api_key}" http://localhost:8081/api/v1/servers/localhost > /dev/null 2>&1; then
        echo -e "PowerDNS API: ${GREEN}✓ Responding${NC}"
    else
        echo -e "PowerDNS API: ${RED}✗ Not responding${NC}"
    fi
}

# Function to backup passwords
backup_passwords() {
    local backup_dir="/docker/backups/powerdns"
    local timestamp=$(date +%Y%m%d_%H%M%S)
    local backup_file="${backup_dir}/passwords_backup_${timestamp}.conf"
    
    mkdir -p "$backup_dir"
    cp "$PASSWORDS_FILE" "$backup_file"
    chmod 600 "$backup_file"
    
    echo -e "${GREEN}Passwords backed up to: $backup_file${NC}"
}

# Function to restore passwords
restore_passwords() {
    local backup_file=$1
    
    if [[ -z "$backup_file" ]]; then
        echo -e "${RED}Please specify a backup file to restore from${NC}"
        echo -e "Usage: $0 restore <backup_file>"
        exit 1
    fi
    
    if [[ ! -f "$backup_file" ]]; then
        echo -e "${RED}Backup file not found: $backup_file${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}Restoring passwords from: $backup_file${NC}"
    cp "$backup_file" "$PASSWORDS_FILE"
    chmod 600 "$PASSWORDS_FILE"
    chown root:root "$PASSWORDS_FILE"
    
    echo -e "${GREEN}Passwords restored successfully${NC}"
    echo -e "${YELLOW}You may need to restart PowerDNS services for changes to take effect.${NC}"
}

# Function to show help
show_help() {
    echo -e "${BLUE}PowerDNS Password Manager for WHP${NC}"
    echo ""
    echo "Usage: $0 [COMMAND] [OPTIONS]"
    echo ""
    echo "Commands:"
    echo "  show           Show current passwords"
    echo "  status         Show configuration status"
    echo "  backup         Backup current passwords"
    echo "  restore <file> Restore passwords from backup file"
    echo "  regenerate     Regenerate all passwords"
    echo "  help           Show this help message"
    echo ""
    echo "Examples:"
    echo "  $0 show                    # Display current passwords"
    echo "  $0 status                  # Check system status"
    echo "  $0 backup                  # Backup passwords"
    echo "  $0 restore backup.conf     # Restore from backup"
    echo "  $0 regenerate              # Generate new passwords"
    echo ""
    echo -e "${YELLOW}Security Note:${NC}"
    echo "  - Passwords are stored in: $PASSWORDS_FILE"
    echo "  - Only root can access password files"
    echo "  - Backup files are created with restricted permissions"
}

# Main script logic
case "${1:-help}" in
    show)
        show_passwords
        ;;
    status)
        show_status
        ;;
    backup)
        backup_passwords
        ;;
    restore)
        restore_passwords "$2"
        ;;
    regenerate)
        echo -e "${YELLOW}This will regenerate all PowerDNS passwords and restart services.${NC}"
        echo -e "${YELLOW}Are you sure? (y/N)${NC}"
        read -r response
        if [[ "$response" =~ ^[Yy]$ ]]; then
            regenerate_all_passwords
        else
            echo -e "${BLUE}Operation cancelled.${NC}"
        fi
        ;;
    help|--help|-h)
        show_help
        ;;
    *)
        echo -e "${RED}Unknown command: $1${NC}"
        echo ""
        show_help
        exit 1
        ;;
esac 