<?php
/**
 * PowerDNS API Example Usage
 * 
 * This file demonstrates how to use the PowerDNS API class
 * to manage DNS records from within the panel.
 */

require_once('pdns-api.php');

// Function to get PowerDNS API key (reused from dns-manager.php)
function getPdnsApiKey() {
    $config_file = '/docker/powerdns/config/pdns.conf';
    if (file_exists($config_file)) {
        $content = file_get_contents($config_file);
        if (preg_match('/api-key=(.+)/', $content, $matches)) {
            return trim($matches[1]);
        }
    }
    return null;
}

// Get PDNS API key
$api_key = getPdnsApiKey();

// Create API instance
// Note: Adjust the API URL based on your PowerDNS installation
$pdnsApi = new PowerDNSAPI('http://127.0.0.1:8081/api/v1', $api_key);

// Example 1: List all zones
echo "<h2>Example 1: List all zones</h2>";
$zones = $pdnsApi->getZones();
if ($zones === false) {
    echo "<p>Error: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
} else {
    echo "<ul>";
    foreach ($zones as $zone) {
        echo "<li>" . htmlspecialchars($zone['name']) . " (Kind: " . htmlspecialchars($zone['kind']) . ")</li>";
    }
    echo "</ul>";
}

// Example 2: Create a new zone
echo "<h2>Example 2: Create a new zone</h2>";
$domainName = "example-" . time() . ".com";
$nameservers = ["ns1.example.com.", "ns2.example.com."];
$result = $pdnsApi->createZone($domainName, $nameservers);

if ($result === false) {
    echo "<p>Error creating zone: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
} else {
    echo "<p>Zone created successfully: " . htmlspecialchars($domainName) . "</p>";
    
    // Example 3: Add an A record
    echo "<h2>Example 3: Add A record</h2>";
    $recordName = $domainName;
    $recordType = "A";
    $ttl = 3600;
    $records = ["192.168.1.1"];
    
    $addResult = $pdnsApi->addRecord($domainName, $recordName, $recordType, $ttl, $records);
    if ($addResult) {
        echo "<p>A record added successfully.</p>";
    } else {
        echo "<p>Error adding A record: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    }
    
    // Example 4: Add CNAME record
    echo "<h2>Example 4: Add CNAME record</h2>";
    $cnameResult = $pdnsApi->addRecord(
        $domainName, 
        "www." . $domainName, 
        "CNAME", 
        3600, 
        [$domainName . "."]
    );
    
    if ($cnameResult) {
        echo "<p>CNAME record added successfully.</p>";
    } else {
        echo "<p>Error adding CNAME record: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    }
    
    // Example 5: List all records in the zone
    echo "<h2>Example 5: List records</h2>";
    $records = $pdnsApi->getRecords($domainName);
    
    if ($records === false) {
        echo "<p>Error retrieving records: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    } else {
        echo "<table border='1'>";
        echo "<tr><th>Name</th><th>Type</th><th>TTL</th><th>Content</th></tr>";
        
        foreach ($records as $record) {
            foreach ($record['records'] as $content) {
                echo "<tr>";
                echo "<td>" . htmlspecialchars($record['name']) . "</td>";
                echo "<td>" . htmlspecialchars($record['type']) . "</td>";
                echo "<td>" . htmlspecialchars($record['ttl']) . "</td>";
                echo "<td>" . htmlspecialchars($content['content']) . "</td>";
                echo "</tr>";
            }
        }
        
        echo "</table>";
    }
    
    // Example 6: Enable DNSSEC
    echo "<h2>Example 6: Enable DNSSEC</h2>";
    $dnssecResult = $pdnsApi->enableDNSSEC($domainName);
    
    if ($dnssecResult) {
        echo "<p>DNSSEC enabled successfully.</p>";
        
        // Show DNSSEC keys
        $keys = $pdnsApi->getDNSSEC($domainName);
        if ($keys !== false) {
            echo "<p>DNSSEC Keys:</p>";
            echo "<ul>";
            foreach ($keys as $key) {
                echo "<li>Key ID: " . htmlspecialchars($key['id']) . ", Type: " . htmlspecialchars($key['type']) . 
                     ", Algorithm: " . htmlspecialchars($key['algorithm']) . ", Active: " . 
                     ($key['active'] ? 'Yes' : 'No') . "</li>";
            }
            echo "</ul>";
        }
    } else {
        echo "<p>Error enabling DNSSEC: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    }
    
    // Example 7: Export zone
    echo "<h2>Example 7: Export zone</h2>";
    $zoneFile = $pdnsApi->exportZone($domainName);
    
    if ($zoneFile === false) {
        echo "<p>Error exporting zone: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    } else {
        echo "<pre>" . htmlspecialchars($zoneFile) . "</pre>";
    }
    
    // Example 8: Delete a record
    echo "<h2>Example 8: Delete a record</h2>";
    $deleteResult = $pdnsApi->deleteRecord($domainName, "www." . $domainName, "CNAME");
    
    if ($deleteResult) {
        echo "<p>CNAME record deleted successfully.</p>";
    } else {
        echo "<p>Error deleting CNAME record: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    }
    
    // Example 9: Delete zone
    echo "<h2>Example 9: Delete zone</h2>";
    $deleteZoneResult = $pdnsApi->deleteZone($domainName);
    
    if ($deleteZoneResult) {
        echo "<p>Zone deleted successfully.</p>";
    } else {
        echo "<p>Error deleting zone: " . htmlspecialchars($pdnsApi->getLastError()) . "</p>";
    }
}