-- HAProxy Integration Schema for WHP
-- This script adds HAProxy integration features to the existing database schema

USE whp;

-- Add listen_port column to container_types table
ALTER TABLE container_types 
ADD COLUMN IF NOT EXISTS listen_port INT DEFAULT 80 COMMENT 'Port that containers listen on for HAProxy backend configuration';

-- Add listen_port_tls column to container_types table for TLS/SSL flag
ALTER TABLE container_types 
ADD COLUMN IF NOT EXISTS listen_port_tls TINYINT(1) DEFAULT 0 COMMENT 'If set, this port uses TLS/SSL for HAProxy backend';

-- Update existing container types with appropriate default ports
UPDATE container_types SET listen_port = 443, listen_port_tls = 1 WHERE name IN ('PHP 7.4', 'PHP 8.0', 'PHP 8.1', 'PHP 8.2', 'PHP 8.3', 'PHP 8.4');
UPDATE container_types SET listen_port = 80, listen_port_tls = 0 WHERE name IN ('Static Apache');

-- Create HAProxy domain configuration table
CREATE TABLE IF NOT EXISTS haproxy_domains (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_name VARCHAR(255) NOT NULL,
    backend_name VARCHAR(255) NOT NULL,
    ssl_enabled TINYINT(1) DEFAULT 0,
    ssl_certificate_path VARCHAR(255) DEFAULT NULL,
    ssl_last_renewal TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_domain (domain_name),
    INDEX idx_backend_name (backend_name),
    INDEX idx_ssl_enabled (ssl_enabled)
);

-- Create HAProxy backend servers table
CREATE TABLE IF NOT EXISTS haproxy_backend_servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    backend_name VARCHAR(255) NOT NULL,
    server_name VARCHAR(255) NOT NULL,
    server_address VARCHAR(255) NOT NULL,
    server_port INT NOT NULL,
    server_options VARCHAR(255) DEFAULT 'check',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_backend_server (backend_name, server_name),
    INDEX idx_backend_name (backend_name)
);

-- Create SSL certificate renewal log table
CREATE TABLE IF NOT EXISTS ssl_renewal_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_name VARCHAR(255) NOT NULL,
    renewal_status ENUM('success', 'failed', 'pending') NOT NULL,
    renewal_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    error_message TEXT DEFAULT NULL,
    certificate_expires TIMESTAMP NULL,
    INDEX idx_domain_name (domain_name),
    INDEX idx_renewal_status (renewal_status),
    INDEX idx_renewal_date (renewal_date)
);

-- Create system service SSL certificates table
CREATE TABLE IF NOT EXISTS system_ssl_certificates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    service_name VARCHAR(50) NOT NULL,
    domain_name VARCHAR(255) NOT NULL,
    certificate_path VARCHAR(255) NOT NULL,
    private_key_path VARCHAR(255) NOT NULL,
    ssl_enabled TINYINT(1) DEFAULT 0,
    auto_renew TINYINT(1) DEFAULT 1,
    last_renewal TIMESTAMP NULL,
    expires_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_service_domain (service_name, domain_name),
    INDEX idx_service_name (service_name),
    INDEX idx_ssl_enabled (ssl_enabled),
    INDEX idx_auto_renew (auto_renew)
);

SELECT 'HAProxy integration schema applied successfully' as status; 