<?php
/**
 * HAProxy API Manager
 * 
 * Manages communication with the HAProxy manager service for IP blocking/unblocking
 * across all services. Provides a unified interface for managing IP blocks at the
 * load balancer level.
 * 
 * @package WHPSecuritySystem
 * @since 2025.08.22
 */

class haproxy_api_manager {
    private $api_url;
    private $api_key;
    private $timeout = 10;
    
    /**
     * Initialize HAProxy API manager
     * 
     * @param string $api_url Base URL for HAProxy manager API (default: http://haproxy-manager:8000)
     * @param string $api_key Optional API key for authentication
     */
    public function __construct($api_url = null, $api_key = null) {
        // Use environment variables or defaults
        $this->api_url = $api_url ?: getenv('HAPROXY_API_URL') ?: 'http://haproxy-manager:8000';
        $this->api_key = $api_key ?: getenv('HAPROXY_API_KEY') ?: '';
    }
    
    /**
     * Block an IP address in HAProxy
     * 
     * @param string $ip_address IP address to block
     * @param string $reason Reason for blocking
     * @param string $blocked_by Who initiated the block
     * @return bool True on success, false on failure
     */
    public function block_ip($ip_address, $reason = 'Manual block', $blocked_by = 'WHP System') {
        if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
            error_log("HAProxy API: Invalid IP address provided for blocking: $ip_address");
            return false;
        }
        
        $endpoint = '/api/blocked-ips';
        $data = [
            'ip_address' => $ip_address,
            'reason' => $reason,
            'blocked_by' => $blocked_by
        ];
        
        $response = $this->make_request('POST', $endpoint, $data);
        
        if ($response && isset($response['status'])) {
            if ($response['status'] === 'success') {
                error_log("HAProxy API: Successfully blocked IP $ip_address");
                return true;
            } elseif ($response['status'] === 'error' && strpos($response['message'], 'already blocked') !== false) {
                // IP is already blocked, consider this a success
                error_log("HAProxy API: IP $ip_address is already blocked");
                return true;
            }
        }
        
        error_log("HAProxy API: Failed to block IP $ip_address - " . json_encode($response));
        return false;
    }
    
    /**
     * Unblock an IP address in HAProxy
     * 
     * @param string $ip_address IP address to unblock
     * @return bool True on success, false on failure
     */
    public function unblock_ip($ip_address) {
        if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
            error_log("HAProxy API: Invalid IP address provided for unblocking: $ip_address");
            return false;
        }
        
        $endpoint = '/api/blocked-ips';
        $data = [
            'ip_address' => $ip_address
        ];
        
        $response = $this->make_request('DELETE', $endpoint, $data);
        
        if ($response && isset($response['status']) && $response['status'] === 'success') {
            error_log("HAProxy API: Successfully unblocked IP $ip_address");
            return true;
        }
        
        error_log("HAProxy API: Failed to unblock IP $ip_address - " . json_encode($response));
        return false;
    }
    
    /**
     * Get list of blocked IPs from HAProxy
     * 
     * @return array Array of blocked IP information or empty array on failure
     */
    public function get_blocked_ips() {
        $endpoint = '/api/blocked-ips';
        $response = $this->make_request('GET', $endpoint);
        
        if (is_array($response)) {
            return $response;
        }
        
        error_log("HAProxy API: Failed to get blocked IPs list");
        return [];
    }
    
    /**
     * Check if an IP is blocked in HAProxy
     * 
     * @param string $ip_address IP address to check
     * @return bool True if blocked, false otherwise
     */
    public function is_ip_blocked($ip_address) {
        if (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
            return false;
        }
        
        $blocked_ips = $this->get_blocked_ips();
        foreach ($blocked_ips as $blocked) {
            if (isset($blocked['ip_address']) && $blocked['ip_address'] === $ip_address) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Reload HAProxy configuration
     * 
     * @return bool True on success, false on failure
     */
    public function reload_haproxy() {
        $endpoint = '/api/reload';
        $response = $this->make_request('GET', $endpoint);
        
        if ($response && isset($response['status']) && $response['status'] === 'success') {
            error_log("HAProxy API: Successfully reloaded HAProxy");
            return true;
        }
        
        error_log("HAProxy API: Failed to reload HAProxy - " . json_encode($response));
        return false;
    }
    
    /**
     * Check HAProxy service health
     * 
     * @return array Health status information or false on failure
     */
    public function check_health() {
        $endpoint = '/health';
        $response = $this->make_request('GET', $endpoint, null, false); // No auth for health check
        
        if ($response && isset($response['status'])) {
            return $response;
        }
        
        return false;
    }
    
    /**
     * Make HTTP request to HAProxy API
     * 
     * @param string $method HTTP method (GET, POST, DELETE)
     * @param string $endpoint API endpoint path
     * @param array $data Optional data to send
     * @param bool $use_auth Whether to use authentication
     * @return mixed Response data or false on failure
     */
    private function make_request($method, $endpoint, $data = null, $use_auth = true) {
        $url = rtrim($this->api_url, '/') . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        
        // Add authentication header if API key is set
        $headers = ['Content-Type: application/json'];
        if ($use_auth && !empty($this->api_key)) {
            $headers[] = 'Authorization: Bearer ' . $this->api_key;
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        // Add data for POST/DELETE requests
        if ($data !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            error_log("HAProxy API: cURL error - $error");
            return false;
        }
        
        if ($http_code >= 200 && $http_code < 300) {
            $decoded = json_decode($response, true);
            return $decoded !== null ? $decoded : $response;
        }
        
        error_log("HAProxy API: HTTP error $http_code - $response");
        
        // Try to decode error response
        $decoded = json_decode($response, true);
        return $decoded !== null ? $decoded : false;
    }
    
    /**
     * Test API connectivity
     * 
     * @return bool True if API is accessible, false otherwise
     */
    public function test_connection() {
        $health = $this->check_health();
        return $health !== false;
    }
}