#!/usr/bin/env bash

# Setup SSH/FTP Log Monitoring System
# This script installs the log monitoring cron job and initializes the system

set -euo pipefail

# Logging
LOG_FILE="/var/log/whp-setup.log"
SCRIPT_NAME="setup-ssh-ftp-monitoring"

# Function to log messages
log_message() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [$SCRIPT_NAME] $1" >> "$LOG_FILE"
}

# Function to check if WHP is properly installed
check_whp_installation() {
    if [[ ! -d "/docker/whp/web" ]]; then
        log_message "ERROR: WHP web directory not found. Please ensure WHP is properly installed."
        echo "ERROR: WHP web directory not found at /docker/whp/web"
        exit 1
    fi
    
    if [[ ! -f "/docker/whp/web/libs/log_monitor_manager.php" ]]; then
        log_message "ERROR: Log monitor manager not found. Please ensure the log monitoring system is properly installed."
        echo "ERROR: Log monitor manager not found"
        exit 1
    fi
    
    log_message "WHP installation verified"
}

# Function to create necessary directories and set permissions
setup_directories() {
    log_message "Setting up directories and permissions"
    
    # Ensure log directories exist
    mkdir -p /var/log/proftpd
    mkdir -p /docker/whp/sql
    
    # Set proper permissions
    chmod 755 /var/log/proftpd
    chown -R root:root /var/log/proftpd
    
    # Make sure the security database directory is accessible
    chmod 755 /docker/whp/sql
    
    log_message "Directories and permissions configured"
}

# Function to setup cron job for log monitoring
setup_cron_job() {
    log_message "Setting up cron job for log monitoring"
    
    # Check if cron job already exists
    if crontab -l 2>/dev/null | grep -q "log_monitor_cron.sh"; then
        log_message "Cron job already exists, skipping setup"
        echo "Log monitoring cron job already configured"
        return 0
    fi
    
    # Add cron job to run every 5 minutes
    (crontab -l 2>/dev/null; echo "*/5 * * * * /root/whp/scripts/log_monitor_cron.sh >/dev/null 2>&1") | crontab -
    
    log_message "Cron job added successfully"
    echo "Log monitoring cron job configured to run every 5 minutes"
}

# Function to test the log monitoring system
test_system() {
    log_message "Testing log monitoring system"
    
    # Create a temporary PHP script to test the system with secure permissions
    local test_script
    test_script=$(mktemp -t ssh_ftp_test.XXXXXX)
    chmod 600 "$test_script"
    
    cat > "$test_script" << 'EOF'
<?php
chdir('/docker/whp/web');
require_once('/docker/whp/web/libs/log_monitor_manager.php');

try {
    $monitor = new log_monitor_manager();
    
    // Test configuration loading
    $config = $monitor->get_all_config();
    if (empty($config)) {
        throw new Exception("Configuration not loaded");
    }
    
    // Test stats retrieval
    $stats = $monitor->get_service_stats();
    if (!isset($stats['ssh']) || !isset($stats['ftp'])) {
        throw new Exception("Service stats not available");
    }
    
    echo "SSH monitoring enabled: " . ($stats['ssh']['monitoring_enabled'] ? 'Yes' : 'No') . "\n";
    echo "FTP monitoring enabled: " . ($stats['ftp']['monitoring_enabled'] ? 'Yes' : 'No') . "\n";
    echo "System test completed successfully\n";
    
} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}
?>
EOF
    
    # Run the test
    if php "$test_script" 2>&1; then
        log_message "System test passed"
        echo "System test completed successfully"
    else
        log_message "ERROR: System test failed"
        echo "ERROR: System test failed"
        rm -f "$test_script"
        exit 1
    fi
    
    # Clean up
    rm -f "$test_script"
}

# Function to display final instructions
show_instructions() {
    echo ""
    echo "=============================================="
    echo "SSH/FTP Log Monitoring Setup Complete!"
    echo "=============================================="
    echo ""
    echo "What was configured:"
    echo "  ✓ Log monitoring system installed"
    echo "  ✓ Database tables created"
    echo "  ✓ Cron job configured (runs every 5 minutes)"
    echo "  ✓ Default settings applied"
    echo ""
    echo "Next steps:"
    echo "  1. Access the WHP Security Management page"
    echo "  2. Configure SSH/FTP monitoring settings as needed"
    echo "  3. Monitor the system logs at /var/log/whp-log-monitor.log"
    echo ""
    echo "Default configuration:"
    echo "  - SSH monitoring: Enabled"
    echo "  - FTP monitoring: Enabled"
    echo "  - Max attempts: 5 (both SSH and FTP)"
    echo "  - Block duration: 1 hour"
    echo "  - iptables blocking: Disabled (software-only blocking)"
    echo "  - Docker network preservation: Enabled"
    echo ""
    echo "Log files monitored:"
    echo "  - SSH: /var/log/auth.log"
    echo "  - FTP: /var/log/proftpd/proftpd.log"
    echo ""
    echo "To manually trigger log checking:"
    echo "  /root/whp/scripts/log_monitor_cron.sh"
    echo ""
}

# Main execution
main() {
    log_message "Starting SSH/FTP log monitoring setup"
    
    echo "Setting up SSH/FTP log monitoring system..."
    
    # Check prerequisites
    check_whp_installation
    
    # Setup system
    setup_directories
    setup_cron_job
    
    # Test the system
    test_system
    
    # Show final instructions
    show_instructions
    
    log_message "SSH/FTP log monitoring setup completed successfully"
}

# Execute main function
main "$@"