#!/usr/bin/env php
<?php
/**
 * Simple Server IP Detection Test
 * 
 * This script tests IP detection methods without requiring database connection.
 */

echo "=== Server IP Detection Test ===\n\n";

echo "1. Settings Configuration:\n";
$settings_file = '/docker/whp/settings.json';
if (file_exists($settings_file)) {
    $settings_content = file_get_contents($settings_file);
    if ($settings_content) {
        $settings = json_decode($settings_content, true);
        if ($settings && isset($settings['server_ip'])) {
            echo "   Configured server_ip: " . $settings['server_ip'] . "\n";
        } else {
            echo "   No server_ip configured in settings.json\n";
        }
    } else {
        echo "   settings.json exists but is empty\n";
    }
} else {
    echo "   settings.json not found\n";
}
echo "\n";

echo "2. IP Detection Methods:\n";

// Method 1: ip route
$cmd1 = "ip route get 8.8.8.8 2>/dev/null | awk '{print \$7}' | head -1";
$ip1 = trim(shell_exec($cmd1));
echo "   ip route method: " . ($ip1 ?: 'failed') . "\n";

// Method 2: hostname -I
$cmd2 = "hostname -I 2>/dev/null | awk '{print \$1}'";
$ip2 = trim(shell_exec($cmd2));
echo "   hostname -I method: " . ($ip2 ?: 'failed') . "\n";

// Method 3: SERVER_ADDR
$ip3 = $_SERVER['SERVER_ADDR'] ?? 'not set';
echo "   \$_SERVER['SERVER_ADDR']: " . $ip3 . "\n";

// Method 4: ip addr
$cmd4 = "ip -4 addr show scope global | grep inet | head -1 | awk '{print \$2}' | cut -d/ -f1 2>/dev/null";
$ip4 = trim(shell_exec($cmd4));
echo "   ip addr method: " . ($ip4 ?: 'failed') . "\n";

// Method 5: All interfaces
$cmd5 = "ip addr show 2>/dev/null | grep 'inet ' | grep -v '127.0.0.1' | awk '{print \$2}' | cut -d/ -f1";
$interfaces = shell_exec($cmd5);
echo "   All non-localhost IPs: " . str_replace("\n", ", ", trim($interfaces)) . "\n";

echo "\n3. Simulated Final Detection (following PowerDNS logic):\n";

// Simulate the PowerDNS getServerIP logic
$server_ip = null;

// Check settings first
if (file_exists($settings_file)) {
    $settings_content = file_get_contents($settings_file);
    if ($settings_content) {
        $settings = json_decode($settings_content, true);
        if ($settings && !empty($settings['server_ip'])) {
            $server_ip = $settings['server_ip'];
            echo "   Using configured IP: " . $server_ip . "\n";
        }
    }
}

if (!$server_ip) {
    // Method 1: Try primary network interface
    $output = shell_exec("ip route get 8.8.8.8 2>/dev/null | awk '{print \$7}' | head -1");
    if ($output && trim($output) !== '') {
        $server_ip = trim($output);
        echo "   Using ip route result: " . $server_ip . "\n";
    }
}

if (!$server_ip) {
    // Method 2: Try getting primary interface IP
    $output = shell_exec("hostname -I 2>/dev/null | awk '{print \$1}'");
    if ($output && trim($output) !== '') {
        $server_ip = trim($output);
        echo "   Using hostname -I result: " . $server_ip . "\n";
    }
}

if (!$server_ip) {
    // Method 3: Try SERVER_ADDR environment variable
    $server_ip = $_SERVER['SERVER_ADDR'] ?? null;
    if ($server_ip) {
        echo "   Using SERVER_ADDR: " . $server_ip . "\n";
    }
}

if (!$server_ip || $server_ip === '127.0.0.1' || $server_ip === '::1') {
    // Method 4: Try getting from primary network interface directly
    $output = shell_exec("ip -4 addr show scope global | grep inet | head -1 | awk '{print \$2}' | cut -d/ -f1 2>/dev/null");
    if ($output && trim($output) !== '') {
        $server_ip = trim($output);
        echo "   Using ip addr result: " . $server_ip . "\n";
    }
}

if (!$server_ip || $server_ip === '127.0.0.1') {
    // Method 5: For development/test environments, check common internal ranges
    $interfaces = shell_exec("ip addr show 2>/dev/null | grep 'inet ' | grep -v '127.0.0.1' | awk '{print \$2}' | cut -d/ -f1");
    if ($interfaces) {
        $ips = explode("\n", trim($interfaces));
        foreach ($ips as $ip) {
            $ip = trim($ip);
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) && $ip !== '127.0.0.1') {
                $server_ip = $ip;
                echo "   Using first valid interface IP: " . $server_ip . "\n";
                break;
            }
        }
    }
}

if (!$server_ip) {
    $server_ip = '127.0.0.1';
    echo "   Fallback to localhost: " . $server_ip . "\n";
}

echo "\n4. Final Result: " . $server_ip . "\n";

if ($server_ip === '127.0.0.1') {
    echo "\n⚠️  WARNING: Using localhost IP! This will not work for external DNS.\n";
    echo "   To fix this, add your server's IP to /docker/whp/settings.json:\n";
    echo '   {"server_ip": "192.168.1.105"}' . "\n";
} else {
    echo "\n✅ Good: Using non-localhost IP for DNS records.\n";
}

echo "\n=== Test Complete ===\n";
?>