<?php
/**
 * HAProxy Manager - Active Implementation
 * 
 * This class handles HAProxy configuration management for the WHP control panel
 * including backend server management, load balancing, and SSL termination.
 * 
 * @version 1.0.0
 * @author WHP Development Team
 */

class haproxy_manager {
    
    private $db;
    private $haproxy_api_url;
    private $haproxy_api_key;
    private $settings;
    
    public function __construct() {
        // Initialize database connection
        require_once('/docker/whp/web/libs/mysqlmgmt.php');
        $MySQLMgmt = new mysqlmgmt();
        $this->db = $MySQLMgmt->getMySQLConnection();
        
        // Load settings
        $this->loadSettings();
        
        // Set HAProxy API configuration
        $this->haproxy_api_url = 'http://localhost:8000';
        $this->haproxy_api_key = $this->settings['haproxy_api_key'] ?? '';
    }
    
    /**
     * Load WHP settings from settings.json
     */
    private function loadSettings() {
        $settings_file = '/docker/whp/settings.json';
        if (file_exists($settings_file)) {
            $settings_content = file_get_contents($settings_file);
            $this->settings = json_decode($settings_content, true) ?: [];
        } else {
            $this->settings = [];
        }
    }
    
    /**
     * Make API request to HAProxy Manager
     * 
     * @param string $method HTTP method (GET, POST, DELETE)
     * @param string $endpoint API endpoint
     * @param array $data Request data
     * @return array API response
     */
    private function makeApiRequest($method, $endpoint, $data = []) {
        $url = $this->haproxy_api_url . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        
        // Add authorization header if API key is set
        $headers = ['Content-Type: application/json'];
        if (!empty($this->haproxy_api_key)) {
            $headers[] = 'Authorization: Bearer ' . $this->haproxy_api_key;
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        // Add request data for POST/PUT/DELETE requests
        if (in_array($method, ['POST', 'PUT', 'DELETE']) && !empty($data)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return [
                'success' => false,
                'error' => 'cURL Error: ' . $error
            ];
        }
        
        $decoded_response = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300) {
            return [
                'success' => true,
                'data' => $decoded_response,
                'http_code' => $http_code
            ];
        } else {
            return [
                'success' => false,
                'error' => $decoded_response['message'] ?? 'HTTP Error: ' . $http_code,
                'http_code' => $http_code
            ];
        }
    }
    
    /**
     * Add domain to HAProxy configuration
     * 
     * @param string $domain Domain name
     * @param array $servers Backend servers configuration
     * @return array Result array with success/error status
     */
    public function addDomain($domain, $servers = []) {
        // Generate backend name from domain
        $backend_name = $this->generateBackendName($domain);
        
        // Prepare API request data
        $request_data = [
            'domain' => $domain,
            'backend_name' => $backend_name,
            'template_override' => null,
            'servers' => $servers
        ];
        
        // Make API request
        $result = $this->makeApiRequest('POST', '/api/domain', $request_data);
        
        if ($result['success']) {
            // Store domain configuration in database
            $this->storeDomainConfiguration($domain, $backend_name, $servers);
        }
        
        return $result;
    }
    

    
    /**
     * Remove domain from HAProxy configuration
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function removeDomain($domain) {
        // Make API request to remove domain
        $result = $this->makeApiRequest('DELETE', '/api/domain', ['domain' => $domain]);
        
        if ($result['success']) {
            // Remove domain configuration from database
            $this->removeDomainConfiguration($domain);
        }
        
        return $result;
    }
    
    /**
     * Enable SSL for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function enableSSL($domain) {
        error_log("HAProxy Manager: Enabling SSL for domain: $domain");
        
        // Make API request to enable SSL
        $result = $this->makeApiRequest('POST', '/api/ssl', ['domain' => $domain]);
        
        error_log("HAProxy Manager: SSL enable result for $domain: " . ($result['success'] ? 'SUCCESS' : 'FAILED - ' . ($result['error'] ?? 'Unknown error')));
        
        if ($result['success']) {
            // Update SSL status in database
            $this->updateSSLStatus($domain, true);
        }
        
        return $result;
    }
    
    /**
     * Request SSL certificate for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function requestCertificate($domain) {
        return $this->makeApiRequest('POST', '/api/certificates/request', ['domain' => $domain]);
    }
    
    /**
     * Renew SSL certificate for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function renewCertificate($domain) {
        return $this->makeApiRequest('POST', '/api/certificates/renew', ['domain' => $domain]);
    }
    
    /**
     * Get SSL certificate status for a domain
     * 
     * @param string $domain Domain name (optional, if not provided gets all certificates)
     * @return array Result array with success/error status
     */
    public function getCertificateStatus($domain = null) {
        $data = [];
        if ($domain) {
            $data['domain'] = $domain;
        }
        return $this->makeApiRequest('GET', '/api/certificates/status', $data);
    }
    
    /**
     * Download SSL certificate for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function downloadCertificate($domain) {
        error_log("HAProxy Manager: Attempting to download certificate for domain: $domain");
        $result = $this->makeApiRequest('GET', '/api/certificates/' . urlencode($domain) . '/download');
        error_log("HAProxy Manager: Download certificate result for $domain: " . json_encode($result));
        return $result;
    }
    
    /**
     * Get SSL certificate key for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function getCertificateKey($domain) {
        return $this->makeApiRequest('GET', '/api/certificates/' . urlencode($domain) . '/key');
    }
    
    /**
     * Get SSL certificate cert for a domain
     * 
     * @param string $domain Domain name
     * @return array Result array with success/error status
     */
    public function getCertificateCert($domain) {
        return $this->makeApiRequest('GET', '/api/certificates/' . urlencode($domain) . '/cert');
    }
    
    /**
     * Get HAProxy health status
     * 
     * @return array Health status
     */
    public function getHealthStatus() {
        // Health endpoint is not under /api path
        $health_url = 'http://localhost:8000/health';
        
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $health_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $this->haproxy_api_key,
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_error($ch)) {
                curl_close($ch);
                return ['success' => false, 'error' => 'Connection error: ' . curl_error($ch)];
            }
            
            curl_close($ch);
            
            if ($http_code === 200) {
                $data = json_decode($response, true);
                return ['success' => true, 'data' => $data];
            } else {
                return ['success' => false, 'error' => 'HTTP ' . $http_code . ': ' . $response];
            }
        } catch (Exception $e) {
            return ['success' => false, 'error' => 'Health check failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Reload HAProxy configuration
     * 
     * @return array Result array with success/error status
     */
    public function reloadConfiguration() {
        return $this->makeApiRequest('GET', '/api/reload');
    }
    
    /**
     * Regenerate HAProxy configuration
     * 
     * @return array Result array with success/error status
     */
    public function regenerateConfiguration() {
        return $this->makeApiRequest('GET', '/api/regenerate');
    }
    
    /**
     * Generate backend name from domain
     * 
     * Converts domain names to HAProxy-compatible backend names by replacing
     * dots and special characters with underscores. The HAProxy service templates
     * automatically append '-backend' suffix, so this method returns only the
     * sanitized domain name (e.g., 'example.com' becomes 'example_com').
     * 
     * @param string $domain Domain name (e.g., 'example.com')
     * @return string Sanitized backend name (e.g., 'example_com')
     */
    private function generateBackendName($domain) {
        // Remove dots and special characters, convert to lowercase
        $backend_name = preg_replace('/[^a-zA-Z0-9_]/', '_', strtolower($domain));
        // Don't append '_backend' here as the HAProxy service appends '-backend' itself
        // Final HAProxy backend name will be: {sanitized_domain}-backend
        return $backend_name;
    }
    
    /**
     * Store domain configuration in database
     * 
     * @param string $domain Domain name
     * @param string $backend_name Backend name
     * @param array $servers Backend servers
     */
    private function storeDomainConfiguration($domain, $backend_name, $servers) {
        try {
            // Insert or update domain configuration
            $stmt = $this->db->prepare("
                INSERT INTO whp.haproxy_domains (domain_name, backend_name) 
                VALUES (?, ?) 
                ON DUPLICATE KEY UPDATE 
                backend_name = VALUES(backend_name),
                updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([$domain, $backend_name]);
            
            // Remove existing servers for this backend
            $stmt = $this->db->prepare("DELETE FROM whp.haproxy_backend_servers WHERE backend_name = ?");
            $stmt->execute([$backend_name]);
            
            // Insert new servers
            foreach ($servers as $server) {
                $stmt = $this->db->prepare("
                    INSERT INTO whp.haproxy_backend_servers 
                    (backend_name, server_name, server_address, server_port, server_options) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $backend_name,
                    $server['name'],
                    $server['address'],
                    $server['port'],
                    $server['options'] ?? 'check'
                ]);
            }
        } catch (PDOException $e) {
            error_log("HAProxy Manager: Failed to store domain configuration: " . $e->getMessage());
        }
    }
    
    /**
     * Remove domain configuration from database
     * 
     * @param string $domain Domain name
     */
    private function removeDomainConfiguration($domain) {
        try {
            // Get backend name
            $stmt = $this->db->prepare("SELECT backend_name FROM whp.haproxy_domains WHERE domain_name = ?");
            $stmt->execute([$domain]);
            $result = $stmt->fetch();
            
            if ($result) {
                $backend_name = $result['backend_name'];
                
                // Remove servers
                $stmt = $this->db->prepare("DELETE FROM whp.haproxy_backend_servers WHERE backend_name = ?");
                $stmt->execute([$backend_name]);
                
                // Remove domain
                $stmt = $this->db->prepare("DELETE FROM whp.haproxy_domains WHERE domain_name = ?");
                $stmt->execute([$domain]);
            }
        } catch (PDOException $e) {
            error_log("HAProxy Manager: Failed to remove domain configuration: " . $e->getMessage());
        }
    }
    
    /**
     * Update SSL status in database
     * 
     * @param string $domain Domain name
     * @param bool $enabled SSL enabled status
     */
    private function updateSSLStatus($domain, $enabled) {
        try {
            $stmt = $this->db->prepare("
                UPDATE whp.haproxy_domains 
                SET ssl_enabled = ?, 
                    ssl_last_renewal = CURRENT_TIMESTAMP,
                    updated_at = CURRENT_TIMESTAMP
                WHERE domain_name = ?
            ");
            $stmt->execute([$enabled ? 1 : 0, $domain]);
        } catch (PDOException $e) {
            error_log("HAProxy Manager: Failed to update SSL status: " . $e->getMessage());
        }
    }
    
    /**
     * Get domain configuration from database
     * 
     * @param string $domain Domain name
     * @return array Domain configuration
     */
    public function getDomainConfiguration($domain) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM whp.haproxy_domains 
                WHERE domain_name = ?
            ");
            $stmt->execute([$domain]);
            $domain_config = $stmt->fetch();
            
            if ($domain_config) {
                // Get backend servers
                $stmt = $this->db->prepare("
                    SELECT * FROM whp.haproxy_backend_servers 
                    WHERE backend_name = ?
                ");
                $stmt->execute([$domain_config['backend_name']]);
                $servers = $stmt->fetchAll();
                
                $domain_config['servers'] = $servers;
            }
            
            return $domain_config ?: [];
        } catch (PDOException $e) {
            error_log("HAProxy Manager: Failed to get domain configuration: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get all configured domains
     * 
     * @return array List of configured domains
     */
    public function getAllDomains() {
        try {
            $stmt = $this->db->query("SELECT * FROM whp.haproxy_domains ORDER BY domain_name");
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("HAProxy Manager: Failed to get all domains: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Configure HAProxy for a WHP site
     * 
     * @param int $site_id Site ID
     * @param array $site_data Site configuration data
     * @return array Result array with success/error status
     */
    public function configureSite($site_id, $site_data) {
        try {
            // Get site domains with FQDNs
            $stmt = $this->db->prepare("
                SELECT 
                    d.domain_name,
                    sd.fqdn
                FROM whp.site_domains sd
                JOIN whp.domains d ON sd.domain_id = d.id
                WHERE sd.site_id = ?
            ");
            $stmt->execute([$site_id]);
            $site_domains = $stmt->fetchAll();
            
            // Get site containers and primary domain, including listen_port_tls
            $stmt = $this->db->prepare("
                SELECT sc.container_id, sc.container_number, ct.listen_port, ct.listen_port_tls, COALESCE(sd.fqdn, d.domain_name) as primary_domain
                FROM whp.site_containers sc
                JOIN whp.sites s ON sc.site_id = s.id
                JOIN whp.domains d ON s.primary_domain_id = d.id
                LEFT JOIN whp.site_domains sd ON s.id = sd.site_id AND sd.is_primary = 1
                JOIN whp.container_types ct ON s.container_type_id = ct.id
                WHERE sc.site_id = ? AND sc.status = 'running'
            ");
            $stmt->execute([$site_id]);
            $containers = $stmt->fetchAll();
            
            // Configure each FQDN
            $results = [];
            foreach ($site_domains as $site_domain) {
                $fqdn = $site_domain['fqdn'] ?: $site_domain['domain_name']; // Use FQDN if available, fallback to domain_name
                $servers = [];
                
                // Add each container as a server
                foreach ($containers as $container) {
                    // Generate container name from primary domain and container number
                    $container_name = $container['primary_domain'] . '-' . str_pad($container['container_number'], 2, '0', STR_PAD_LEFT);
                    $options = 'check';
                    if (!empty($container['listen_port_tls']) && $container['listen_port_tls'] == 1) {
                        $options .= ' ssl verify none';
                    }
                    $servers[] = [
                        'name' => $container_name,
                        'address' => $container_name, // Use container name for client-net resolution
                        'port' => $container['listen_port'],
                        'options' => $options
                    ];
                }
                
                // Remove existing configuration first, then add new one
                $remove_result = $this->removeDomain($fqdn);
                if (!$remove_result['success']) {
                    error_log("HAProxy Manager: Failed to remove existing domain configuration for $fqdn: " . $remove_result['error']);
                }
                
                // Add FQDN to HAProxy with new server configuration
                $result = $this->addDomain($fqdn, $servers);
                $results[$fqdn] = $result;
            }
            
            // Regenerate HAProxy configuration to ensure changes take effect
            $regenerate_result = $this->regenerateConfiguration();
            if (!$regenerate_result['success']) {
                error_log("HAProxy Manager: Failed to regenerate configuration: " . $regenerate_result['error']);
            }
            
            return [
                'success' => true,
                'results' => $results,
                'regenerate_result' => $regenerate_result
            ];
        } catch (PDOException $e) {
            return [
                'success' => false,
                'error' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Remove HAProxy configuration for a WHP site
     * 
     * @param int $site_id Site ID
     * @return array Result array with success/error status
     */
    public function removeSite($site_id) {
        try {
            // Get site domains with FQDNs
            $stmt = $this->db->prepare("
                SELECT 
                    d.domain_name,
                    sd.fqdn
                FROM whp.site_domains sd
                JOIN whp.domains d ON sd.domain_id = d.id
                WHERE sd.site_id = ?
            ");
            $stmt->execute([$site_id]);
            $site_domains = $stmt->fetchAll();
            
            // Remove each FQDN from HAProxy
            $results = [];
            foreach ($site_domains as $site_domain) {
                $fqdn = $site_domain['fqdn'] ?: $site_domain['domain_name']; // Use FQDN if available, fallback to domain_name
                $result = $this->removeDomain($fqdn);
                $results[$fqdn] = $result;
            }
            
            return [
                'success' => true,
                'results' => $results
            ];
        } catch (PDOException $e) {
            return [
                'success' => false,
                'error' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
}
?> 