-- Create the required tables for WHP user-resources functionality
USE whp;

-- Create client_allowances table (updated to use username directly)
CREATE TABLE IF NOT EXISTS client_allowances (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(32) NOT NULL UNIQUE,
    max_cpu_allowance DECIMAL(4,2) DEFAULT 0.5,
    max_memory_allowance INT DEFAULT 512,
    max_domains INT DEFAULT 1,
    max_disk_space INT DEFAULT 1000,
    max_email_accounts INT DEFAULT 0,
    max_email_storage_mb INT DEFAULT 1000,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_username (username)
);

-- Create container_types table
CREATE TABLE IF NOT EXISTS container_types (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(64) NOT NULL UNIQUE,
    description TEXT,
    base_image VARCHAR(255) NOT NULL,
    min_cpu DECIMAL(3,2) DEFAULT 0.25,
    min_memory INT DEFAULT 256,
    startup_env TEXT COMMENT 'JSON object of startup environment variables',
    user_variables TEXT COMMENT 'JSON object of user-configurable variables',
    mount_options TEXT COMMENT 'JSON object of mount options and volumes',
    last_image_pull TIMESTAMP NULL DEFAULT NULL COMMENT 'Last time the base image was pulled',
    image_digest VARCHAR(255) NULL DEFAULT NULL COMMENT 'Current image digest/hash',
    listen_port INT DEFAULT 80 COMMENT 'Port that containers listen on for HAProxy backend configuration',
    listen_port_tls TINYINT(1) DEFAULT 0 COMMENT 'If set, this port uses TLS/SSL for HAProxy backend',
    active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_name (name),
    INDEX idx_active (active)
);

-- Create domains table (updated to use username instead of user_id)
CREATE TABLE IF NOT EXISTS domains (
    id INT AUTO_INCREMENT PRIMARY KEY,
    domain_name VARCHAR(255) NOT NULL UNIQUE,
    username VARCHAR(32) NOT NULL,
    container_type_id INT NOT NULL,
    cpu_allowance DECIMAL(4,2) DEFAULT 0.25,
    memory_allowance INT DEFAULT 256,
    active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (container_type_id) REFERENCES container_types(id) ON DELETE RESTRICT,
    INDEX idx_domain_name (domain_name),
    INDEX idx_username (username),
    INDEX idx_container_type (container_type_id),
    INDEX idx_active (active)
);

-- Create containers table
CREATE TABLE IF NOT EXISTS containers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    container_id VARCHAR(255) NOT NULL UNIQUE,
    domain_id INT NOT NULL,
    ip_address VARCHAR(45) DEFAULT NULL,
    port INT DEFAULT NULL,
    status VARCHAR(20) DEFAULT 'stopped',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (domain_id) REFERENCES domains(id) ON DELETE CASCADE,
    INDEX idx_container_id (container_id),
    INDEX idx_domain_id (domain_id),
    INDEX idx_status (status)
);

-- Insert actual container types from repo.anhonesthost.net
INSERT IGNORE INTO container_types (name, description, base_image, min_cpu, min_memory, startup_env, user_variables, mount_options, listen_port, listen_port_tls, active) VALUES 
('PHP 7.4', 'Apache with PHP 7.4', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php74', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('PHP 8.0', 'Apache with PHP 8.0', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php80', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('PHP 8.1', 'Apache with PHP 8.1', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php81', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('PHP 8.2', 'Apache with PHP 8.2', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php82', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('PHP 8.3', 'Apache with PHP 8.3', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php83', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('PHP 8.4', 'Apache with PHP 8.4', 'repo.anhonesthost.net/cloud-hosting-platform/cac:php84', 0.25, 256, 
'{"uid": "${WHP_UID}", "user": "${WHP_USER}", "domain": "${WHP_DOMAIN}"}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/home/${WHP_USER}", "type": "bind"}]}',
443, 1, 1),

('Static Apache', 'Default Apache container for static content only', 'repo.anhonesthost.net/cloud-hosting-platform/whp-default:latest', 0.25, 256, 
'{}',
'{}',
'{"volumes": [{"source": "${WHP_HOME}/${WHP_DOMAIN}", "target": "/usr/local/apache2/htdocs", "type": "bind"}]}',
80, 0, 1);

SELECT 'Schema updated for username-based user management' as status; 