#!/usr/bin/env php
<?php
/**
 * Backup Scheduler Script
 * This script processes scheduled backups
 * 
 * Usage: php backup-scheduler.php [--verbose|-v]
 *   --verbose, -v  Show detailed logging information
 * 
 * Run via cron: every 5 minutes - add to crontab: star-slash-5 * * * * /usr/bin/php /path/to/backup-scheduler.php
 * For verbose logging in cron: star-slash-5 * * * * /usr/bin/php /path/to/backup-scheduler.php --verbose >> /var/log/backup-scheduler.log 2>&1
 */

// Set up path to include WHP libraries
$whpPath = realpath(__DIR__ . '/../web-files');
if (!$whpPath) {
    die("Error: Could not find WHP web files directory\n");
}

// Add WHP lib path
ini_set('include_path', $whpPath . '/libs' . PATH_SEPARATOR . ini_get('include_path'));

// Load configuration (only if not already loaded)
if (!defined('MYSQL_PASS')) {
    require_once $whpPath . '/configs/config.php';
}

// Load backup system
require_once $whpPath . '/libs/BackupEngine.php';
require_once $whpPath . '/libs/BackupTarget.php';
require_once $whpPath . '/libs/BackupHistory.php';
if (!class_exists('mysqlmgmt')) {
    require_once $whpPath . '/libs/mysqlmgmt.php';
}

use WHPBackup\BackupEngine;
use WHPBackup\BackupTarget;
use WHPBackup\BackupHistory;

// Enable error reporting for CLI
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check for verbose flag
$verbose = in_array('--verbose', $argv) || in_array('-v', $argv);

// Function to log messages
function logMessage($message, $level = 'INFO') {
    global $verbose;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$level] $message\n";
    $logFile = '/var/log/whp-backup-scheduler.log';
    
    // Always log errors, warnings, and summaries to file
    if ($level === 'ERROR' || $level === 'WARNING' || $level === 'SUMMARY') {
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
    
    // Log info messages to file if verbose mode is enabled
    if ($verbose && $level === 'INFO') {
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
    
    // Output to stdout if running interactively (not from cron)
    if (php_sapi_name() === 'cli' && function_exists('posix_isatty') && posix_isatty(STDOUT)) {
        // Always show errors, warnings, and summaries
        if ($level === 'ERROR' || $level === 'WARNING' || $level === 'SUMMARY') {
            echo $logMessage;
            return;
        }
        
        // Only show info messages if verbose mode is enabled
        if ($verbose) {
            echo $logMessage;
        }
    }
}

// Function to update schedule next run time
function updateScheduleNextRun($db, $scheduleId, $scheduleType, $scheduleTime, $scheduleDay) {
    $now = new DateTime();
    $nextRun = new DateTime();
    
    switch ($scheduleType) {
        case 'hourly':
            $nextRun->setTime($nextRun->format('H'), 0, 0);
            $nextRun->add(new DateInterval('PT1H'));
            break;
            
        case 'daily':
            $time = explode(':', $scheduleTime);
            $nextRun->setTime($time[0], $time[1], 0);
            if ($nextRun <= $now) {
                $nextRun->add(new DateInterval('P1D'));
            }
            break;
            
        case 'weekly':
            $time = explode(':', $scheduleTime);
            $nextRun->setTime($time[0], $time[1], 0);
            $nextRun->modify("next " . getDayName($scheduleDay));
            break;
            
        case 'monthly':
            $time = explode(':', $scheduleTime);
            $nextRun->setTime($time[0], $time[1], 0);
            $nextRun->setDate($nextRun->format('Y'), $nextRun->format('m'), $scheduleDay);
            if ($nextRun <= $now) {
                $nextRun->add(new DateInterval('P1M'));
            }
            break;
    }
    
    $stmt = $db->prepare("UPDATE backup_schedules SET next_run_at = ?, last_run_at = NOW() WHERE id = ?");
    $stmt->execute([$nextRun->format('Y-m-d H:i:s'), $scheduleId]);
}

// Function to get day name from number
function getDayName($dayNum) {
    $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
    return $days[$dayNum] ?? 'Sunday';
}

try {
    logMessage("Starting backup scheduler");
    
    // Get database connection
    $mysql = new mysqlmgmt();
    $db = $mysql->getMySQLConnection();
    $db->exec("USE whp");
    
    $engine = new BackupEngine();
    $now = new DateTime();
    
    // Counters for summary
    $scheduledProcessed = 0;
    $scheduledFailed = 0;
    $queuedProcessed = 0;
    $queuedFailed = 0;
    
    // Get schedules that are due to run
    $stmt = $db->prepare("
        SELECT bs.*, bt.name as target_name 
        FROM backup_schedules bs 
        JOIN backup_targets bt ON bs.target_id = bt.id 
        WHERE bs.is_active = 1 
        AND (bs.next_run_at IS NULL OR bs.next_run_at <= NOW())
        ORDER BY bs.next_run_at ASC
    ");
    $stmt->execute();
    
    $schedules = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($schedules)) {
        logMessage("No schedules due to run");
    } else {
    
    foreach ($schedules as $schedule) {
        $scheduleId = $schedule['id'];
        $user = $schedule['user'];
        $targetId = $schedule['target_id'];
        $backupType = $schedule['backup_type'];
        $resourceName = $schedule['resource_name'];
        $targetName = $schedule['target_name'];
        
        logMessage("Processing schedule: $scheduleId for user: $user, type: $backupType");
        
        try {
            // Check if target is still accessible
            $target = new BackupTarget($db, $targetId);
            if (!$target->canAccess($user)) {
                logMessage("User $user no longer has access to target $targetName", 'WARNING');
                continue;
            }
            
            $result = null;
            
            switch ($backupType) {
                case 'site':
                    if (empty($resourceName)) {
                        // If no specific site is specified, get the first available site
                        $sites = $engine->getUserSites($user);
                        if (empty($sites)) {
                            logMessage("No sites found for user: $user", 'WARNING');
                            $scheduledFailed++;
                            continue 2;
                        }
                        $resourceName = $sites[0];
                        logMessage("No specific site specified, using: $resourceName");
                    } elseif ($resourceName === '*') {
                        // Backup all sites
                        $sites = $engine->getUserSites($user);
                        if (empty($sites)) {
                            logMessage("No sites found for user: $user", 'WARNING');
                            $scheduledFailed++;
                            continue 2;
                        }
                        
                        $allSuccess = true;
                        foreach ($sites as $site) {
                            try {
                                // Check if this is a WordPress site
                                $stmt = $db->prepare("SELECT id, database_name FROM wordpress_sites WHERE user = ? AND domain = ? AND status = 'active' LIMIT 1");
                                $stmt->execute([$user, $site]);
                                $wpSite = $stmt->fetch(PDO::FETCH_ASSOC);
                                
                                if ($wpSite) {
                                    logMessage("Site $site is a WordPress site, creating WordPress backup");
                                    $siteResult = $engine->createWordPressBackup($wpSite['id'], null, $targetId);
                                } else {
                                    $siteResult = $engine->createSiteBackup($user, $site, $targetId);
                                }
                                
                                if (!$siteResult || !$siteResult['success']) {
                                    $allSuccess = false;
                                }
                            } catch (Exception $e) {
                                logMessage("Failed to backup site $site: " . $e->getMessage(), 'ERROR');
                                $allSuccess = false;
                            }
                        }
                        
                        $result = ['success' => $allSuccess, 'backup_id' => 'multiple'];
                        break;
                    }
                    
                    // Single site backup
                    // Check if this is a WordPress site
                    $stmt = $db->prepare("SELECT id, database_name FROM wordpress_sites WHERE user = ? AND domain = ? AND status = 'active' LIMIT 1");
                    $stmt->execute([$user, $resourceName]);
                    $wpSite = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($wpSite) {
                        // This is a WordPress site, create a WordPress backup
                        logMessage("Site $resourceName is a WordPress site, creating WordPress backup");
                        $result = $engine->createWordPressBackup($wpSite['id'], null, $targetId);
                    } else {
                        // Regular site backup
                        $result = $engine->createSiteBackup($user, $resourceName, $targetId);
                    }
                    break;
                    
                case 'userfiles':
                    $result = $engine->createUserfilesBackup($user, $targetId);
                    break;
                    
                case 'database':
                    if (empty($resourceName)) {
                        // If no specific database is specified, get the first available database
                        $databases = $engine->getUserDatabases($user);
                        if (empty($databases)) {
                            logMessage("No databases found for user: $user", 'WARNING');
                            $scheduledFailed++;
                            continue 2;
                        }
                        $resourceName = $databases[0];
                        logMessage("No specific database specified, using: $resourceName");
                    } elseif ($resourceName === '*') {
                        // Backup all databases
                        $databases = $engine->getUserDatabases($user);
                        if (empty($databases)) {
                            logMessage("No databases found for user: $user", 'WARNING');
                            $scheduledFailed++;
                            continue 2;
                        }
                        
                        $allSuccess = true;
                        foreach ($databases as $database) {
                            try {
                                $dbResult = $engine->createDatabaseBackup($user, $database, $targetId);
                                if (!$dbResult || !$dbResult['success']) {
                                    $allSuccess = false;
                                }
                            } catch (Exception $e) {
                                logMessage("Failed to backup database $database: " . $e->getMessage(), 'ERROR');
                                $allSuccess = false;
                            }
                        }
                        
                        $result = ['success' => $allSuccess, 'backup_id' => 'multiple'];
                    } else {
                        // Single database backup
                        $result = $engine->createDatabaseBackup($user, $resourceName, $targetId);
                    }
                    break;
                    
                case 'all':
                    // Create backups for all user resources
                    $sites = $engine->getUserSites($user);
                    $databases = $engine->getUserDatabases($user);
                    
                    // Backup all sites
                    foreach ($sites as $site) {
                        $engine->createSiteBackup($user, $site, $targetId);
                    }
                    
                    // Backup userfiles
                    $engine->createUserfilesBackup($user, $targetId);
                    
                    // Backup all databases
                    foreach ($databases as $database) {
                        $engine->createDatabaseBackup($user, $database, $targetId);
                    }
                    
                    $result = ['success' => true, 'backup_id' => 'multiple'];
                    break;
                    
                default:
                    logMessage("Unknown backup type: $backupType", 'ERROR');
                    continue 2;
            }
            
            if ($result && $result['success']) {
                logMessage("Scheduled backup completed successfully for user: $user");
                $scheduledProcessed++;
            } else {
                // Get detailed error message
                $errorMessage = "Scheduled backup failed for user: $user";
                if ($result && isset($result['error'])) {
                    $errorMessage .= " - " . $result['error'];
                } elseif ($result && isset($result['message'])) {
                    $errorMessage .= " - " . $result['message'];
                } elseif (!$result) {
                    $errorMessage .= " - Backup operation returned null result";
                }
                logMessage($errorMessage, 'ERROR');
                $scheduledFailed++;
            }
            
        } catch (Exception $e) {
            logMessage("Error processing schedule $scheduleId: " . $e->getMessage(), 'ERROR');
            $scheduledFailed++;
        }
        
        // Update next run time
        updateScheduleNextRun(
            $db, 
            $scheduleId, 
            $schedule['schedule_type'], 
            $schedule['schedule_time'], 
            $schedule['schedule_day']
        );
    }
    }
    
    // Process backup queue
    logMessage("Processing backup queue");
    
    $stmt = $db->prepare("
        SELECT bq.*, bt.name as target_name 
        FROM backup_queue bq 
        JOIN backup_targets bt ON bq.target_id = bt.id 
        WHERE bq.status = 'queued'
        AND bq.attempts < 3
        ORDER BY 
            CASE bq.priority 
                WHEN 'high' THEN 1 
                WHEN 'normal' THEN 2 
                WHEN 'low' THEN 3 
            END,
            bq.created_at ASC
        LIMIT 10
    ");
    $stmt->execute();
    
    $queuedBackups = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($queuedBackups as $queueItem) {
        $queueId = $queueItem['id'];
        $user = $queueItem['user'];
        $targetId = $queueItem['target_id'];
        $backupType = $queueItem['backup_type'];
        $backupName = $queueItem['backup_name'];
        $targetName = $queueItem['target_name'];
        
        logMessage("Processing queued backup: $queueId for user: $user, type: $backupType, name: $backupName");
        
        try {
            // Mark as processing
            $stmt = $db->prepare("UPDATE backup_queue SET status = 'processing', started_at = NOW(), attempts = attempts + 1 WHERE id = ?");
            $stmt->execute([$queueId]);
            
            // Check if target is still accessible
            $target = new BackupTarget($db, $targetId);
            if (!$target->canAccess($user)) {
                logMessage("User $user no longer has access to target $targetName", 'WARNING');
                
                // Mark as failed
                $stmt = $db->prepare("UPDATE backup_queue SET status = 'failed', completed_at = NOW(), error_message = ? WHERE id = ?");
                $stmt->execute(["Access denied to backup target", $queueId]);
                $queuedFailed++;
                continue;
            }
            
            $result = null;
            
            switch ($backupType) {
                case 'site':
                    // Check if this is a WordPress site
                    $stmt = $db->prepare("SELECT id, database_name FROM wordpress_sites WHERE user = ? AND domain = ? AND status = 'active' LIMIT 1");
                    $stmt->execute([$user, $backupName]);
                    $wpSite = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($wpSite) {
                        // This is a WordPress site, create a WordPress backup
                        logMessage("Site $backupName is a WordPress site, creating WordPress backup");
                        $result = $engine->createWordPressBackup($wpSite['id'], null, $targetId);
                    } else {
                        // Regular site backup
                        $result = $engine->createSiteBackup($user, $backupName, $targetId);
                    }
                    break;
                    
                case 'userfiles':
                    $result = $engine->createUserfilesBackup($user, $targetId);
                    break;
                    
                case 'database':
                    $result = $engine->createDatabaseBackup($user, $backupName, $targetId);
                    break;
                    
                default:
                    throw new Exception("Unknown backup type: $backupType");
            }
            
            if ($result && $result['success']) {
                logMessage("Queued backup completed successfully for user: $user");
                
                // Mark as completed
                $stmt = $db->prepare("UPDATE backup_queue SET status = 'completed', completed_at = NOW() WHERE id = ?");
                $stmt->execute([$queueId]);
                $queuedProcessed++;
            } else {
                // Get detailed error message
                $errorMessage = "Backup operation returned failure";
                if ($result && isset($result['error'])) {
                    $errorMessage = $result['error'];
                } elseif ($result && isset($result['message'])) {
                    $errorMessage = $result['message'];
                } elseif (!$result) {
                    $errorMessage = "Backup operation returned null result";
                }
                throw new Exception($errorMessage);
            }
            
        } catch (Exception $e) {
            logMessage("Error processing queued backup $queueId: " . $e->getMessage(), 'ERROR');
            
            // Mark as failed
            $stmt = $db->prepare("UPDATE backup_queue SET status = 'failed', completed_at = NOW(), error_message = ? WHERE id = ?");
            $stmt->execute([$e->getMessage(), $queueId]);
            $queuedFailed++;
        }
    }
    
    // Always show summary
    $totalProcessed = $scheduledProcessed + $queuedProcessed;
    $totalFailed = $scheduledFailed + $queuedFailed;
    
    if ($totalProcessed > 0 || $totalFailed > 0) {
        logMessage("Completed: Scheduled($scheduledProcessed), Queued($queuedProcessed), Failed($totalFailed)", 'SUMMARY');
    } else {
        logMessage("No backups processed", 'INFO');
    }
    
} catch (Exception $e) {
    logMessage("Backup scheduler failed: " . $e->getMessage(), 'ERROR');
    exit(1);
}