-- Sites Schema for WHP
-- This allows users to configure domains with containers and manage multiple containers per site

USE whp;

-- Create sites table to manage site configurations
CREATE TABLE IF NOT EXISTS sites (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_name VARCHAR(255) NOT NULL,
    primary_domain_id INT NOT NULL,
    username VARCHAR(32) NOT NULL,
    container_type_id INT NOT NULL,
    container_count INT DEFAULT 1,
    cpu_per_container DECIMAL(4,2) DEFAULT 0.25,
    memory_per_container INT DEFAULT 256,
    user_variables JSON COMMENT 'User-configured variables for the container type',
    active BOOLEAN DEFAULT TRUE,
    auto_recreate_on_update TINYINT(1) DEFAULT 0 COMMENT 'Automatically recreate containers when new image versions are available',
    -- SSL/HAProxy Configuration (Future Implementation)
    ssl_enabled TINYINT(1) DEFAULT 0 COMMENT 'SSL certificate enabled (0=No, 1=Let\'s Encrypt, 2=Custom)',
    ssl_certificate_path VARCHAR(255) DEFAULT NULL COMMENT 'Path to SSL certificate file',
    ssl_key_path VARCHAR(255) DEFAULT NULL COMMENT 'Path to SSL private key file',
    ssl_force_https TINYINT(1) DEFAULT 0 COMMENT 'Force HTTPS redirects',
    ssl_hsts_enabled TINYINT(1) DEFAULT 0 COMMENT 'Enable HSTS header',
    ssl_last_renewal TIMESTAMP NULL COMMENT 'Last SSL certificate renewal date',
    haproxy_backend_name VARCHAR(255) DEFAULT NULL COMMENT 'HAProxy backend name for this site',
    haproxy_balance_method VARCHAR(50) DEFAULT 'roundrobin' COMMENT 'HAProxy load balancing method',
    haproxy_health_check VARCHAR(255) DEFAULT NULL COMMENT 'HAProxy health check configuration',
    haproxy_ssl_redirect TINYINT(1) DEFAULT 0 COMMENT 'HAProxy SSL redirect configuration',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (primary_domain_id) REFERENCES domains(id) ON DELETE CASCADE,
    FOREIGN KEY (container_type_id) REFERENCES container_types(id) ON DELETE RESTRICT,
    INDEX idx_site_name (site_name),
    INDEX idx_username (username),
    INDEX idx_primary_domain (primary_domain_id),
    INDEX idx_container_type (container_type_id),
    INDEX idx_active (active),
    INDEX idx_ssl_enabled (ssl_enabled),
    INDEX idx_haproxy_backend (haproxy_backend_name),
    INDEX idx_auto_recreate (auto_recreate_on_update)
);

-- Create site_domains table to link additional domains/subdomains to sites
CREATE TABLE IF NOT EXISTS site_domains (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    domain_id INT NOT NULL,
    fqdn VARCHAR(255) DEFAULT NULL,
    is_primary BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    FOREIGN KEY (domain_id) REFERENCES domains(id) ON DELETE CASCADE,
    UNIQUE KEY unique_site_domain (site_id, domain_id, fqdn),
    INDEX idx_site_id (site_id),
    INDEX idx_domain_id (domain_id),
    INDEX idx_is_primary (is_primary)
);

-- Create site_containers table to track individual containers for each site
CREATE TABLE IF NOT EXISTS site_containers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    container_id VARCHAR(255) NOT NULL UNIQUE,
    container_number INT NOT NULL COMMENT 'Container number within the site (1, 2, 3, etc.)',
    ip_address VARCHAR(45) DEFAULT NULL,
    port INT DEFAULT NULL,
    status VARCHAR(20) DEFAULT 'stopped',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    UNIQUE KEY unique_site_container_number (site_id, container_number),
    INDEX idx_site_id (site_id),
    INDEX idx_container_id (container_id),
    INDEX idx_status (status)
);

-- SSL Certificate Management Table (Future Implementation)
-- This table will store SSL certificate information for sites
CREATE TABLE IF NOT EXISTS ssl_certificates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    certificate_type ENUM('letsencrypt', 'custom', 'self_signed') NOT NULL,
    domain_name VARCHAR(255) NOT NULL,
    certificate_path VARCHAR(255) NOT NULL,
    private_key_path VARCHAR(255) NOT NULL,
    certificate_chain_path VARCHAR(255) DEFAULT NULL,
    issued_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_date TIMESTAMP NOT NULL,
    auto_renew BOOLEAN DEFAULT TRUE,
    status ENUM('active', 'expired', 'revoked', 'pending') DEFAULT 'pending',
    last_renewal_attempt TIMESTAMP NULL,
    renewal_failure_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    INDEX idx_site_id (site_id),
    INDEX idx_domain_name (domain_name),
    INDEX idx_expires_date (expires_date),
    INDEX idx_status (status),
    INDEX idx_auto_renew (auto_renew, expires_date)
);

-- HAProxy Backend Configuration Table (Future Implementation)
-- This table will store HAProxy backend configuration for sites
CREATE TABLE IF NOT EXISTS haproxy_backends (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    backend_name VARCHAR(255) NOT NULL UNIQUE,
    balance_method ENUM('roundrobin', 'static-rr', 'leastconn', 'source', 'uri') DEFAULT 'roundrobin',
    health_check_enabled BOOLEAN DEFAULT TRUE,
    health_check_interval INT DEFAULT 10000 COMMENT 'Health check interval in milliseconds',
    health_check_timeout INT DEFAULT 5000 COMMENT 'Health check timeout in milliseconds',
    health_check_retries INT DEFAULT 3,
    ssl_redirect BOOLEAN DEFAULT FALSE,
    custom_config TEXT COMMENT 'Custom HAProxy configuration for this backend',
    active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    INDEX idx_site_id (site_id),
    INDEX idx_backend_name (backend_name),
    INDEX idx_active (active)
);

-- HAProxy Backend Servers Table (Future Implementation)
-- This table will store individual server configurations for HAProxy backends
CREATE TABLE IF NOT EXISTS haproxy_backend_servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    backend_id INT NOT NULL,
    server_name VARCHAR(255) NOT NULL,
    server_address VARCHAR(255) NOT NULL,
    server_port INT NOT NULL,
    weight INT DEFAULT 1,
    backup BOOLEAN DEFAULT FALSE,
    check_enabled BOOLEAN DEFAULT TRUE,
    status ENUM('active', 'disabled', 'maintenance') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (backend_id) REFERENCES haproxy_backends(id) ON DELETE CASCADE,
    INDEX idx_backend_id (backend_id),
    INDEX idx_server_name (server_name),
    INDEX idx_status (status)
);

-- Add indexes for better performance
CREATE INDEX IF NOT EXISTS idx_sites_username_active ON sites (username, active);
CREATE INDEX IF NOT EXISTS idx_site_containers_site_status ON site_containers (site_id, status);
CREATE INDEX IF NOT EXISTS idx_site_domains_site_primary ON site_domains (site_id, is_primary);

-- Insert sample data for testing (optional)
-- INSERT IGNORE INTO sites (site_name, primary_domain_id, username, container_type_id, container_count, cpu_per_container, memory_per_container) VALUES
-- ('My Website', 1, 'testuser', 1, 2, 0.5, 512);

SELECT 'Sites schema created successfully' as status; 