#!/usr/bin/env php
<?php

// Background worker for processing backup jobs
// This script should be run as a daemon or via cron

// Load configuration first
require_once(__DIR__ . '/../configs/config.php');
require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once(__DIR__ . '/../libs/BackupJob.php');
require_once(__DIR__ . '/../libs/BackupHistory.php');

use WHPBackup\BackupJob;
use WHPBackup\BackupHistory;

function log_message($message) {
    $timestamp = date('Y-m-d H:i:s');
    error_log("[$timestamp] BackupWorker: $message");
}

function process_pending_backups() {
    try {
        $mysql = new mysqlmgmt();
        $db = $mysql->getMySQLConnection();
        $db->exec("USE whp");
        
        // Get pending backups
        $stmt = $db->prepare("SELECT id FROM backup_history WHERE status = 'pending' ORDER BY started_at ASC LIMIT 5");
        $stmt->execute();
        $backups = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($backups)) {
            log_message("No pending backups found");
            return;
        }
        
        log_message("Found " . count($backups) . " pending backups");
        
        $job = new BackupJob();
        
        foreach ($backups as $backup) {
            try {
                log_message("Processing backup ID: " . $backup['id']);
                $job->processBackup($backup['id']);
                log_message("Backup ID " . $backup['id'] . " completed successfully");
            } catch (Exception $e) {
                log_message("Backup ID " . $backup['id'] . " failed: " . $e->getMessage());
                log_message("Stack trace: " . $e->getTraceAsString());
            }
        }
        
    } catch (Exception $e) {
        log_message("Error in backup worker: " . $e->getMessage());
    }
}

// Main execution
if (php_sapi_name() === 'cli') {
    log_message("Backup worker started");
    
    // Process pending backups
    process_pending_backups();
    
    log_message("Backup worker finished");
} else {
    echo "This script can only be run from command line\n";
    exit(1);
}