<?php

namespace WHPBackup;

use Exception;

class BackupStorage {
    private $config;
    private $s3Client;
    
    public function __construct($config) {
        $this->config = $config;
        $this->initializeS3Client();
    }
    
    private function initializeS3Client() {
        $this->s3Client = new S3Client($this->config);
    }
    
    public function uploadFile($localPath, $remotePath) {
        try {
            // Construct full path with prefix
            if (!empty($this->config['path_prefix'])) {
                $fullPath = rtrim($this->config['path_prefix'], '/') . '/' . ltrim($remotePath, '/');
            } else {
                $fullPath = ltrim($remotePath, '/');
            }
            
            
            $result = $this->s3Client->putObject([
                'Bucket' => $this->config['bucket'],
                'Key' => $fullPath,
                'SourceFile' => $localPath
            ]);
            
            return [
                'success' => true,
                'path' => $fullPath,
                'size' => filesize($localPath),
                'etag' => $result['ETag'] ?? null
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function downloadFile($remotePath, $localPath) {
        try {
            $fullPath = $this->config['path_prefix'] . '/' . ltrim($remotePath, '/');
            
            $result = $this->s3Client->getObject([
                'Bucket' => $this->config['bucket'],
                'Key' => $fullPath,
                'SaveAs' => $localPath
            ]);
            
            return [
                'success' => true,
                'path' => $localPath,
                'size' => filesize($localPath)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function streamFile($remotePath, $outputCallback = null) {
        try {
            $fullPath = $this->config['path_prefix'] . '/' . ltrim($remotePath, '/');
            
            $result = $this->s3Client->streamObject([
                'Bucket' => $this->config['bucket'],
                'Key' => $fullPath,
                'Callback' => $outputCallback
            ]);
            
            return [
                'success' => true,
                'size' => $result['size']
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function deleteFile($remotePath) {
        try {
            $fullPath = $this->config['path_prefix'] . '/' . ltrim($remotePath, '/');
            
            $this->s3Client->deleteObject([
                'Bucket' => $this->config['bucket'],
                'Key' => $fullPath
            ]);
            
            return [
                'success' => true,
                'path' => $fullPath
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function listFiles($prefix = '') {
        try {
            $fullPrefix = $this->config['path_prefix'] . '/' . ltrim($prefix, '/');
            
            $result = $this->s3Client->listObjects([
                'Bucket' => $this->config['bucket'],
                'Prefix' => $fullPrefix
            ]);
            
            $files = [];
            if (isset($result['Contents'])) {
                foreach ($result['Contents'] as $object) {
                    $files[] = [
                        'key' => $object['Key'],
                        'size' => $object['Size'],
                        'modified' => $object['LastModified']
                    ];
                }
            }
            
            return [
                'success' => true,
                'files' => $files
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function getFileInfo($remotePath) {
        try {
            $fullPath = $this->config['path_prefix'] . '/' . ltrim($remotePath, '/');
            
            $result = $this->s3Client->headObject([
                'Bucket' => $this->config['bucket'],
                'Key' => $fullPath
            ]);
            
            return [
                'success' => true,
                'exists' => true,
                'size' => $result['ContentLength'],
                'modified' => $result['LastModified'],
                'etag' => $result['ETag']
            ];
            
        } catch (Exception $e) {
            if (strpos($e->getMessage(), '404') !== false) {
                return [
                    'success' => true,
                    'exists' => false
                ];
            }
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function testConnection() {
        try {
            $this->s3Client->listObjects([
                'Bucket' => $this->config['bucket'],
                'MaxKeys' => 1
            ]);
            
            return [
                'success' => true,
                'message' => 'Connection successful'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

/**
 * Simple S3-compatible client implementation
 */
class S3Client {
    private $config;
    
    public function __construct($config) {
        $this->config = $config;
    }
    
    public function putObject($params) {
        $url = $this->buildUrl($params['Bucket'], $params['Key']);
        
        // For S3, we can use UNSIGNED-PAYLOAD for streaming uploads
        // This avoids having to load the entire file into memory
        $payloadHash = 'UNSIGNED-PAYLOAD';
        
        $fileSize = filesize($params['SourceFile']);
        $headers = $this->buildHeaders('PUT', $params['Bucket'], $params['Key'], '', 'application/octet-stream', $payloadHash, $fileSize);
        
        $fp = fopen($params['SourceFile'], 'r');
        if (!$fp) {
            throw new Exception("Could not open file: " . $params['SourceFile']);
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_PUT, true);
        curl_setopt($ch, CURLOPT_INFILE, $fp);
        curl_setopt($ch, CURLOPT_INFILESIZE, filesize($params['SourceFile']));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        
        // Add error handling for curl errors
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        fclose($fp);
        
        if ($curlError) {
            throw new Exception("CURL Error: " . $curlError);
        }
        
        if ($httpCode >= 400) {
            // Try to extract error message from response
            $errorMsg = "Upload failed with HTTP code: $httpCode";
            if (!empty($response)) {
                // Try to parse XML error response
                $xml = @simplexml_load_string($response);
                if ($xml && isset($xml->Message)) {
                    $errorMsg .= " - " . (string)$xml->Message;
                } else {
                    $errorMsg .= " - Response: " . substr($response, 0, 500);
                }
            }
            throw new Exception($errorMsg);
        }
        
        // Extract ETag from response headers
        preg_match('/ETag: "([^"]+)"/', $response, $matches);
        $etag = $matches[1] ?? null;
        
        return ['ETag' => $etag];
    }
    
    public function getObject($params) {
        $url = $this->buildUrl($params['Bucket'], $params['Key']);
        $headers = $this->buildHeaders('GET', $params['Bucket'], $params['Key'], '');
        
        $fp = fopen($params['SaveAs'], 'w');
        if (!$fp) {
            throw new Exception("Could not create file: " . $params['SaveAs']);
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_FILE, $fp);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        fclose($fp);
        
        if ($httpCode >= 400) {
            unlink($params['SaveAs']);
            throw new Exception("Download failed with HTTP code: $httpCode");
        }
        
        return ['success' => true];
    }
    
    public function streamObject($params) {
        $url = $this->buildUrl($params['Bucket'], $params['Key']);
        $headers = $this->buildHeaders('GET', $params['Bucket'], $params['Key'], '');
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_HEADER, false); // Don't include headers in output
        curl_setopt($ch, CURLOPT_WRITEFUNCTION, function($ch, $data) use ($params) {
            if (isset($params['Callback']) && is_callable($params['Callback'])) {
                $params['Callback']($data);
            } else {
                echo $data;
                flush(); // Ensure data is sent immediately
            }
            return strlen($data);
        });
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, true); // Handle binary data properly
        
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentLength = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        curl_close($ch);
        
        if ($httpCode >= 400) {
            throw new Exception("Stream failed with HTTP code: $httpCode");
        }
        
        return ['success' => true, 'size' => $contentLength];
    }
    
    public function deleteObject($params) {
        $url = $this->buildUrl($params['Bucket'], $params['Key']);
        $headers = $this->buildHeaders('DELETE', $params['Bucket'], $params['Key'], '');
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 400) {
            throw new Exception("Delete failed with HTTP code: $httpCode");
        }
        
        return ['success' => true];
    }
    
    public function listObjects($params) {
        $url = $this->buildUrl($params['Bucket'], '');
        $query = [];
        
        if (isset($params['Prefix'])) {
            $query['prefix'] = $params['Prefix'];
        }
        if (isset($params['MaxKeys'])) {
            $query['max-keys'] = $params['MaxKeys'];
        }
        
        if (!empty($query)) {
            $url .= '?' . http_build_query($query);
        }
        
        $queryString = '';
        if (!empty($query)) {
            $queryString = http_build_query($query);
        }
        
        $headers = $this->buildHeaders('GET', $params['Bucket'], '', $queryString);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 400) {
            throw new Exception("List failed with HTTP code: $httpCode");
        }
        
        // Parse XML response
        $xml = simplexml_load_string($response);
        $contents = [];
        
        if ($xml && isset($xml->Contents)) {
            foreach ($xml->Contents as $object) {
                $contents[] = [
                    'Key' => (string)$object->Key,
                    'Size' => (int)$object->Size,
                    'LastModified' => (string)$object->LastModified
                ];
            }
        }
        
        return ['Contents' => $contents];
    }
    
    public function headObject($params) {
        $url = $this->buildUrl($params['Bucket'], $params['Key']);
        $headers = $this->buildHeaders('HEAD', $params['Bucket'], $params['Key'], '');
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_NOBODY, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 400) {
            throw new Exception("Head request failed with HTTP code: $httpCode");
        }
        
        // Parse headers
        $headers = [];
        $lines = explode("\n", $response);
        foreach ($lines as $line) {
            if (strpos($line, ':') !== false) {
                list($key, $value) = explode(':', $line, 2);
                $headers[trim($key)] = trim($value);
            }
        }
        
        return [
            'ContentLength' => $headers['Content-Length'] ?? 0,
            'LastModified' => $headers['Last-Modified'] ?? '',
            'ETag' => trim($headers['ETag'] ?? '', '"')
        ];
    }
    
    private function buildUrl($bucket, $key) {
        $endpoint = rtrim($this->config['endpoint'], '/');
        
        // Handle different S3 service URL formats
        // Path-style: https://s3.amazonaws.com/bucket/key
        // Virtual-hosted-style: https://bucket.s3.amazonaws.com/key
        
        // For now, use path-style which works with most S3-compatible services
        $url = $endpoint . '/' . $bucket;
        if (!empty($key)) {
            $url .= '/' . ltrim($key, '/');
        }
        
        return $url;
    }
    
    private function buildHeaders($method, $bucket, $key, $queryString = '', $contentType = 'application/octet-stream', $payloadHash = 'UNSIGNED-PAYLOAD', $contentLength = null) {
        $timestamp = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');
        
        // AWS Signature Version 4
        $region = $this->config['region'] ?? 'us-east-1';
        $service = 's3';
        $credentialScope = $dateStamp . '/' . $region . '/' . $service . '/aws4_request';
        
        $host = parse_url($this->config['endpoint'], PHP_URL_HOST);
        $port = parse_url($this->config['endpoint'], PHP_URL_PORT);
        
        // Include port in host header if it's not standard (80/443)
        if ($port && $port != 80 && $port != 443) {
            $host = $host . ':' . $port;
        }
        
        $headers = [
            'Host: ' . $host,
            'X-Amz-Date: ' . $timestamp,
            'X-Amz-Content-Sha256: ' . $payloadHash,
            'Content-Type: ' . $contentType
        ];
        
        // Add Content-Length for PUT requests
        if ($contentLength !== null) {
            $headers[] = 'Content-Length: ' . $contentLength;
        }
        
        $headers[] = 'Authorization: ' . $this->signRequestV4($method, $bucket, $key, $timestamp, $dateStamp, $region, $service, $credentialScope, $queryString, $contentType, $payloadHash, $contentLength);
        
        return $headers;
    }
    
    private function signRequestV4($method, $bucket, $key, $timestamp, $dateStamp, $region, $service, $credentialScope, $queryString = '', $contentType = 'application/octet-stream', $payloadHash = 'UNSIGNED-PAYLOAD', $contentLength = null) {
        // AWS Signature Version 4
        $accessKey = $this->config['access_key'];
        $secretKey = $this->config['secret_key'];
        
        // Create canonical URI - must be URL encoded
        // For S3, the canonical URI should include the bucket and the object key
        // The key should be URL-encoded, but keep the slashes
        $parts = [$bucket];
        if (!empty($key)) {
            $keyParts = explode('/', $key);
            foreach ($keyParts as $part) {
                if ($part !== '') {
                    $parts[] = rawurlencode($part);
                }
            }
        }
        $canonicalUri = '/' . implode('/', $parts);
        
        $canonicalQueryString = $queryString;
        
        // Create canonical headers - must be lowercase and sorted
        $host = parse_url($this->config['endpoint'], PHP_URL_HOST);
        $port = parse_url($this->config['endpoint'], PHP_URL_PORT);
        
        // Include port in host header if it's not standard (80/443)
        if ($port && $port != 80 && $port != 443) {
            $host = $host . ':' . $port;
        }
        
        $canonicalHeaders = "content-type:" . $contentType . "\n" .
                           "host:" . $host . "\n" .
                           "x-amz-content-sha256:" . $payloadHash . "\n" .
                           "x-amz-date:" . $timestamp . "\n";
        $signedHeaders = 'content-type;host;x-amz-content-sha256;x-amz-date';
        
        $canonicalRequest = $method . "\n" .
                           $canonicalUri . "\n" .
                           $canonicalQueryString . "\n" .
                           $canonicalHeaders . "\n" .
                           $signedHeaders . "\n" .
                           $payloadHash;
        
        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $stringToSign = $algorithm . "\n" .
                       $timestamp . "\n" .
                       $credentialScope . "\n" .
                       hash('sha256', $canonicalRequest);
        
        
        // Create signing key
        $signingKey = $this->getSigningKey($secretKey, $dateStamp, $region, $service);
        
        // Create signature
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);
        
        // Create authorization header
        $authorization = $algorithm . ' ' .
                        'Credential=' . $accessKey . '/' . $credentialScope . ', ' .
                        'SignedHeaders=' . $signedHeaders . ', ' .
                        'Signature=' . $signature;
        
        
        return $authorization;
    }
    
    private function getSigningKey($secretKey, $dateStamp, $region, $service) {
        $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $secretKey, true);
        $kRegion = hash_hmac('sha256', $region, $kDate, true);
        $kService = hash_hmac('sha256', $service, $kRegion, true);
        $kSigning = hash_hmac('sha256', 'aws4_request', $kService, true);
        
        return $kSigning;
    }
}