<?php
// API endpoint for site creation with container readiness checking
header('Content-Type: application/json');

// Check API permission
check_api_permission('sites');

require_once '../libs/mysqlmgmt.php';
require_once '../libs/docker-api.php';
require_once '../libs/pdns-integration.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

if (empty($_POST['action']) || $_POST['action'] !== 'create_site') {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid action']);
    exit;
}

$is_root = (defined('AUTH_USER') && AUTH_USER === 'root');

// Initialize database connection
$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();

if (!$db) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database connection failed']);
    exit;
}

// Initialize Docker API and PowerDNS
$DockerAPI = new docker_api();
$PowerDNS = new PowerDNSIntegration($db);

try {
    // Site creation validation
    if (empty($_POST['site_name']) || empty($_POST['primary_domain_id']) || empty($_POST['container_type_id'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Missing required fields']);
        exit;
    }

    $site_name = trim($_POST['site_name']);
    $primary_fqdn = trim($_POST['primary_domain_id']);
    $container_type_id = intval($_POST['container_type_id']);
    $container_count = max(1, min(10, intval($_POST['container_count'] ?? 1)));
    $cpu_per_container = floatval($_POST['cpu_per_container'] ?? 0.25);
    $memory_per_container = intval($_POST['memory_per_container'] ?? 256);
    $auto_recreate_on_update = !empty($_POST['auto_recreate_on_update']) ? 1 : 0;
    
    // Collect user variables
    $user_variables = [];
    if (!empty($_POST['user_variables'])) {
        $user_variables = json_decode($_POST['user_variables'], true) ?: [];
    }

    // Domain ownership and DNS validation
    // Get all active domains to find the parent domain for the FQDN
    $stmt = $db->prepare("SELECT id, domain_name, username FROM whp.domains WHERE active = 1 AND (username = ? OR ? = 1)");
    $stmt->execute([AUTH_USER, $is_root ? 1 : 0]);
    $all_domains = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Find parent domain for primary FQDN
    $primary_domain_row = null;
    foreach ($all_domains as $row) {
        if ($primary_fqdn === $row['domain_name'] || (str_ends_with($primary_fqdn, '.' . $row['domain_name']))) {
            $primary_domain_row = $row;
            break;
        }
    }

    if (!$primary_domain_row) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Domain not found or access denied']);
        exit;
    }

    // Validate FQDN is a valid A or CNAME for this domain
    $valid_fqdn = false;
    $fqdn_records = $PowerDNS->getRecords($primary_domain_row['domain_name']);
    foreach ($fqdn_records as $rec) {
        if (strcasecmp($rec['name'], $primary_fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
            $valid_fqdn = true;
            break;
        }
    }
    if (!$valid_fqdn) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'DNS validation failed: No A or CNAME record found for ' . $primary_fqdn]);
        exit;
    }

    // Container type validation
    $stmt = $db->prepare("SELECT * FROM whp.container_types WHERE id = ? AND active = 1");
    $stmt->execute([$container_type_id]);
    $container_type = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$container_type) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid container type']);
        exit;
    }

    // Resource validation
    if ($cpu_per_container < $container_type['min_cpu'] || $memory_per_container < $container_type['min_memory']) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Resources below container type minimums']);
        exit;
    }

    // File system preparation using actual FQDN for subdomain support
    $domain_dir = '/docker/users/' . AUTH_USER . '/' . $primary_fqdn;
    if (!file_exists($domain_dir)) {
        if (!mkdir($domain_dir, 0755, true)) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Failed to create directory']);
            exit;
        }
        
        // Set proper ownership
        $uid = trim(shell_exec("id -u " . escapeshellarg(AUTH_USER)));
        $gid = trim(shell_exec("id -g " . escapeshellarg(AUTH_USER)));
        if (is_numeric($uid) && is_numeric($gid)) {
            chown($domain_dir, intval($uid));
            chgrp($domain_dir, intval($gid));
        }
    }

    // Database site record creation
    $stmt = $db->prepare("
        INSERT INTO whp.sites (site_name, primary_domain_id, username, container_type_id, container_count, 
                              cpu_per_container, memory_per_container, user_variables, auto_recreate_on_update) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $site_name,
        $primary_domain_row['id'],
        AUTH_USER,
        $container_type_id,
        $container_count,
        $cpu_per_container,
        $memory_per_container,
        json_encode($user_variables),
        $auto_recreate_on_update
    ]);
    
    $site_id = $db->lastInsertId();

    // Link primary domain to site
    $stmt = $db->prepare("INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) VALUES (?, ?, ?, 1)");
    $stmt->execute([$site_id, $primary_domain_row['id'], $primary_fqdn]);

    // Process additional domains
    $additional_domains_added = [];
    if (!empty($_POST['additional_domains']) && is_array($_POST['additional_domains'])) {
        foreach ($_POST['additional_domains'] as $additional_fqdn) {
            $additional_fqdn = trim($additional_fqdn);
            if (empty($additional_fqdn)) continue;

            // Find parent domain using already-loaded domains
            $parent_row = null;
            foreach ($all_domains as $row) {
                if ($additional_fqdn === $row['domain_name'] || (str_ends_with($additional_fqdn, '.' . $row['domain_name']))) {
                    $parent_row = $row;
                    break;
                }
            }

            if ($parent_row) {
                // Validate additional FQDN has valid A or CNAME record
                $valid_fqdn = false;
                $fqdn_records = $PowerDNS->getRecords($parent_row['domain_name']);
                foreach ($fqdn_records as $rec) {
                    if (strcasecmp($rec['name'], $additional_fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                        $valid_fqdn = true;
                        break;
                    }
                }
                
                if ($valid_fqdn) {
                // Check if not already linked
                $stmt = $db->prepare("SELECT id FROM whp.site_domains WHERE site_id = ? AND domain_id = ? AND fqdn = ?");
                $stmt->execute([$site_id, $parent_row['id'], $additional_fqdn]);
                
                if (!$stmt->fetch()) {
                    $stmt = $db->prepare("INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) VALUES (?, ?, ?, 0)");
                    $stmt->execute([$site_id, $parent_row['id'], $additional_fqdn]);
                    $additional_domains_added[] = $additional_fqdn;
                }
                }
            }
        }
    }

    // Container creation
    $container_creation_success = true;
    $container_errors = [];
    $created_containers = [];

    // Get container type details
    $stmt = $db->prepare("SELECT * FROM whp.container_types WHERE id = ?");
    $stmt->execute([$container_type_id]);
    $container_type_details = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get user UID
    $uid = trim(shell_exec("id -u " . escapeshellarg(AUTH_USER)));
    if (!is_numeric($uid)) {
        throw new Exception('Invalid user UID');
    }

    for ($i = 1; $i <= $container_count; $i++) {
        $container_name = $primary_fqdn . '-' . str_pad($i, 2, '0', STR_PAD_LEFT);
        
        // Environment variables
        $env_vars = [
            'WHP_USER=' . AUTH_USER,
            'WHP_UID=' . $uid,
            'WHP_DOMAIN=' . $primary_fqdn,
            'WHP_CONTAINER_NUMBER=' . $i
        ];

        // Add user variables
        foreach ($user_variables as $var_name => $var_value) {
            $env_vars[] = $var_name . '=' . $var_value;
        }

        // Process startup environment variables
        if (!empty($container_type_details['startup_env'])) {
            $startup_env = json_decode($container_type_details['startup_env'], true);
            if ($startup_env) {
                foreach ($startup_env as $var_name => $var_value) {
                    $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, AUTH_USER, $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                    $env_vars[] = $var_name . '=' . $var_value;
                }
            }
        }

        // Volumes using actual FQDN path
        $volumes = [
            '/docker/users/' . AUTH_USER . '/' . $primary_fqdn . ':/var/www/html'
        ];

        // Process mount options
        if (!empty($container_type_details['mount_options'])) {
            $mount_options = json_decode($container_type_details['mount_options'], true);
            if ($mount_options && isset($mount_options['volumes'])) {
                foreach ($mount_options['volumes'] as $volume) {
                    $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], AUTH_USER, $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                    $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], AUTH_USER, $primary_fqdn, $uid, $i, $container_type_details['listen_port'] ?? 80);
                    $volumes[] = $source . ':' . $target;
                }
            }
        }

        // Container configuration
        $container_config = [
            'image' => $container_type_details['base_image'],
            'name' => $container_name,
            'labels' => [
                'whp.user' => AUTH_USER,
                'whp.domain' => $primary_fqdn,
                'whp.site_id' => strval($site_id),
                'whp.container_number' => strval($i)
            ],
            'env' => $env_vars,
            'volumes' => $volumes,
            'cpu' => $cpu_per_container,
            'memory' => $memory_per_container,
            'network' => 'client-net'
        ];

        $result = $DockerAPI->create_container($container_config);

        if ($result['success']) {
            $stmt = $db->prepare("INSERT INTO whp.site_containers (site_id, container_id, container_number, status) VALUES (?, ?, ?, 'active')");
            $stmt->execute([$site_id, $result['container_id'], $i]);
            $created_containers[] = $result['container_id'];
        } else {
            $container_creation_success = false;
            $container_errors[] = "Container $i: " . $result['error'];
        }
    }

    // Configure HAProxy if containers were created successfully
    if ($container_creation_success) {
        require_once('/docker/whp/web/libs/haproxy_manager.php');
        $haproxy_manager = new haproxy_manager();
        $haproxy_result = $haproxy_manager->configureSite($site_id, [
            'username' => AUTH_USER,
            'primary_domain' => $primary_domain_row['domain_name'],
            'site_name' => $site_name,
            'container_count' => $container_count
        ]);

        if (!$haproxy_result['success']) {
            $container_errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
        }
    }

    // Prepare response
    $response = [
        'success' => $container_creation_success,
        'site_id' => $site_id,
        'container_count' => $container_count,
        'created_containers' => $created_containers
    ];

    if ($container_creation_success) {
        $domain_info = '';
        if (!empty($additional_domains_added)) {
            $domain_info = ' with additional domains: ' . implode(', ', $additional_domains_added);
        }
        $response['message'] = 'Site created successfully with ' . $container_count . ' container(s) started and HAProxy configured: ' . $site_name . $domain_info;
    } else {
        $response['success'] = false;
        $response['message'] = 'Site created but some containers failed to start';
        $response['errors'] = $container_errors;
    }

    echo json_encode($response);

} catch (Exception $e) {
    error_log("Site creation error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Site creation failed: ' . $e->getMessage()
    ]);
}
?>