#!/usr/bin/env php
<?php
/**
 * Backup Scheduler - Processes scheduled backups
 * This script should be run via cron every minute
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include necessary files
require_once(__DIR__ . '/../configs/config.php');
require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once(__DIR__ . '/../libs/BackupSchedule.php');
require_once(__DIR__ . '/../libs/BackupEngine.php');
require_once(__DIR__ . '/../libs/BackupHistory.php');

use WHPBackup\BackupSchedule;
use WHPBackup\BackupEngine;
use WHPBackup\BackupHistory;

echo "Starting backup scheduler at " . date('Y-m-d H:i:s') . "\n";

try {
    // Get due schedules
    $schedules = BackupSchedule::getDueSchedules();
    
    if (empty($schedules)) {
        echo "No schedules due for processing.\n";
        exit(0);
    }
    
    echo "Found " . count($schedules) . " schedules due for processing.\n";
    
    $engine = new BackupEngine();
    
    foreach ($schedules as $scheduleData) {
        echo "Processing schedule ID: " . $scheduleData['id'] . "\n";
        
        try {
            $schedule = new BackupSchedule(null, $scheduleData['id']);
            
            // Create backup based on type
            $backupCount = 0;
            switch ($scheduleData['backup_type']) {
                case 'site':
                    if ($scheduleData['resource_name']) {
                        $result = $engine->createSiteBackup($scheduleData['user'], $scheduleData['resource_name'], $scheduleData['target_id']);
                        $backupCount = 1;
                    } else {
                        // Backup all sites for user
                        $sites = $engine->getUserSites($scheduleData['user']);
                        foreach ($sites as $site) {
                            $result = $engine->createSiteBackup($scheduleData['user'], $site, $scheduleData['target_id']);
                            $backupCount++;
                        }
                    }
                    break;
                    
                case 'userfiles':
                    $result = $engine->createUserfilesBackup($scheduleData['user'], $scheduleData['target_id']);
                    $backupCount = 1;
                    break;
                    
                case 'database':
                    if ($scheduleData['resource_name']) {
                        $result = $engine->createDatabaseBackup($scheduleData['user'], $scheduleData['resource_name'], $scheduleData['target_id']);
                        $backupCount = 1;
                    } else {
                        // Backup all databases for user
                        $databases = $engine->getUserDatabases($scheduleData['user']);
                        foreach ($databases as $database) {
                            $result = $engine->createDatabaseBackup($scheduleData['user'], $database, $scheduleData['target_id']);
                            $backupCount++;
                        }
                    }
                    break;
                    
                default:
                    throw new Exception("Unknown backup type: " . $scheduleData['backup_type']);
            }
            
            // Update schedule last run time
            $schedule->updateLastRun();
            
            // Clean up old backups based on retention
            $engine->cleanupScheduledBackups($scheduleData['target_id'], $scheduleData['user'], $scheduleData['backup_type'], $scheduleData['max_retention']);
            
            echo "Successfully processed schedule ID: " . $scheduleData['id'] . " (created $backupCount backups)\n";
            
        } catch (Exception $e) {
            echo "Error processing schedule ID " . $scheduleData['id'] . ": " . $e->getMessage() . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "Fatal error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "Backup scheduler completed at " . date('Y-m-d H:i:s') . "\n";