#!/usr/bin/env bash

# This script sets up PowerDNS server and admin interface for the Web Hosting Panel
# It should be run after the main setup.sh script

# Create directories for PowerDNS
mkdir -p /docker/powerdns/{data,config}

# Generate a random API key for PowerDNS
PDNS_API_KEY=$(tr -dc 'A-Za-z0-9' </dev/urandom | head -c 32)
SECRET_KEY=$(tr -dc 'A-Za-z0-9!@#$%^&*(-_=+)' </dev/urandom | head -c 50)

# Get MySQL password from root configuration
MYSQL_PASSWORD=$(grep password /root/.my.cnf | cut -d= -f2-)

# Create PowerDNS configuration files
cat > /docker/powerdns/config/pdns.conf << EOF
launch=gmysql
gmysql-host=mysql
gmysql-port=3306
gmysql-dbname=powerdns
gmysql-user=pdns
gmysql-password=${MYSQL_PASSWORD}
gmysql-dnssec=yes
api=yes
api-key=${PDNS_API_KEY}
webserver=yes
webserver-address=0.0.0.0
webserver-port=8081
webserver-allow-from=0.0.0.0/0
EOF

# Set up PowerDNS database
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "CREATE DATABASE IF NOT EXISTS powerdns;"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "CREATE USER IF NOT EXISTS 'pdns'@'%' IDENTIFIED BY '${MYSQL_PASSWORD}';"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "GRANT ALL PRIVILEGES ON powerdns.* TO 'pdns'@'%';"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "FLUSH PRIVILEGES;"

# Download and import PowerDNS schema
wget -q -O /tmp/schema.mysql.sql https://raw.githubusercontent.com/PowerDNS/pdns/master/modules/gmysqlbackend/schema.mysql.sql
mysql -h 127.0.0.1 -u pdns -p"${MYSQL_PASSWORD}" powerdns < /tmp/schema.mysql.sql
rm /tmp/schema.mysql.sql

# Start PowerDNS server
docker run -d --name powerdns \
  --network client-net \
  -p 53:53/tcp -p 53:53/udp \
  -p 8081:8081 \
  -v /docker/powerdns/config/pdns.conf:/etc/pdns/pdns.conf:ro \
  --restart unless-stopped \
  pschiffe/pdns-mysql:latest

# Set up PowerDNS-Admin database
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "CREATE DATABASE IF NOT EXISTS pdns_admin;"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "CREATE USER IF NOT EXISTS 'pdns_admin'@'%' IDENTIFIED BY '${MYSQL_PASSWORD}';"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "GRANT ALL PRIVILEGES ON pdns_admin.* TO 'pdns_admin'@'%';"
mysql -h 127.0.0.1 -u root -p"${MYSQL_PASSWORD}" -e "FLUSH PRIVILEGES;"

# Start PowerDNS-Admin
docker run -d --name pdns-admin \
  --network client-net \
  -p 9393:80 \
  -e PDNS_API_URL=http://powerdns:8081 \
  -e PDNS_API_KEY=${PDNS_API_KEY} \
  -e PDNS_PROTO=http \
  -e PDNS_PORT=8081 \
  -e PDNS_HOST=powerdns \
  -e SECRET_KEY=${SECRET_KEY} \
  -e MYSQL_HOST=mysql \
  -e MYSQL_PORT=3306 \
  -e MYSQL_USER=pdns_admin \
  -e MYSQL_PASSWORD=${MYSQL_PASSWORD} \
  -e MYSQL_DATABASE=pdns_admin \
  --restart unless-stopped \
  ngoduykhanh/powerdns-admin:latest

echo "PowerDNS and PowerDNS-Admin have been set up successfully"
echo "PowerDNS API Key: ${PDNS_API_KEY}"
echo "PowerDNS Admin URL: http://your-server-ip:9393"
echo "Default PowerDNS Admin login: admin/admin"
echo "Please change the default password after first login"

# Update firewall rules if not already done
firewall-cmd --permanent --add-port 53/tcp
firewall-cmd --permanent --add-port 53/udp
firewall-cmd --permanent --add-port 8081/tcp
firewall-cmd --permanent --add-port 9393/tcp
firewall-cmd --reload