<?php
// Check if user is authenticated
if (!defined('AUTH_USER')) {
    header('Location: login.php');
    exit();
}

require_once 'libs/BackupTarget.php';
require_once 'libs/BackupHistory.php';
require_once 'libs/BackupEngine.php';
require_once 'libs/BackupSchedule.php';

use WHPBackup\BackupTarget;
use WHPBackup\BackupHistory;
use WHPBackup\BackupEngine;
use WHPBackup\BackupSchedule;

$engine = new BackupEngine();
$targets = BackupTarget::listTargets(null, AUTH_USER);
$isRoot = AUTH_USER === 'root';

// Get user's sites and databases
if ($isRoot) {
    // Root can see all sites and databases
    $sites = [];
    $databases = [];
    // Get all users
    $allUsersList = [];
    $userDir = '/docker/users';
    if (is_dir($userDir)) {
        $directories = glob($userDir . '/*', GLOB_ONLYDIR);
        foreach ($directories as $dir) {
            $allUsersList[] = basename($dir);
        }
    }
    // Get sites for all users
    foreach ($allUsersList as $user) {
        $userSites = $engine->getUserSites($user);
        foreach ($userSites as $site) {
            $sites[] = $user . '/' . $site;
        }
    }
    // Get databases for all users
    foreach ($allUsersList as $user) {
        $userDatabases = $engine->getUserDatabases($user);
        foreach ($userDatabases as $database) {
            $databases[] = $database;
        }
    }
} else {
    // Regular users only see their own
    $sites = $engine->getUserSites(AUTH_USER);
    $databases = $engine->getUserDatabases(AUTH_USER);
}

// Get all users if root
$allUsers = [];
if ($isRoot) {
    $userDir = '/docker/users';
    if (is_dir($userDir)) {
        $directories = glob($userDir . '/*', GLOB_ONLYDIR);
        foreach ($directories as $dir) {
            $allUsers[] = basename($dir);
        }
    }
}

// Get schedules
$scheduleFilters = ['user' => AUTH_USER, 'limit' => 20];
if ($isRoot) {
    unset($scheduleFilters['user']); // Root can see all schedules
}
$schedules = BackupSchedule::listSchedules(null, $scheduleFilters);

// Get recent backups
$backupFilters = ['user' => AUTH_USER, 'exclude_deleted' => true, 'limit' => 20];
if ($isRoot) {
    unset($backupFilters['user']); // Root can see all backups
}
$recentBackups = BackupHistory::listBackups(null, $backupFilters);

// Get backup statistics
$completedFilters = $isRoot ? ['status' => 'completed'] : ['user' => AUTH_USER, 'status' => 'completed'];
$stats = [
    'total_backups' => BackupHistory::countBackups(null, $completedFilters),
    'total_size' => BackupHistory::getTotalSize(null, $isRoot ? [] : ['user' => AUTH_USER])
];
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-end align-items-center mb-4">
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#targetModal" onclick="clearTargetForm()">
                    <i class="fa fa-plus"></i> Add Backup Target
                </button>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h5>Total Backups</h5>
                            <h3><?php echo number_format($stats['total_backups']); ?></h3>
                        </div>
                        <i class="fa fa-archive fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h5>Total Size</h5>
                            <h3><?php echo formatBytes($stats['total_size']); ?></h3>
                        </div>
                        <i class="fa fa-hdd-o fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h5>Sites</h5>
                            <h3><?php echo count($sites); ?></h3>
                        </div>
                        <i class="fa fa-globe fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-warning text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h5>Databases</h5>
                            <h3><?php echo count($databases); ?></h3>
                        </div>
                        <i class="fa fa-database fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Backup Section -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fa fa-plus-circle"></i> Create New Backup</h5>
                </div>
                <div class="card-body">
                    <form id="backupForm">
                        <div class="row">
                            <?php if ($isRoot): ?>
                            <div class="col-md-3">
                                <label for="backupUser" class="form-label">User</label>
                                <select class="form-select" id="backupUser" required>
                                    <option value="">Select User</option>
                                    <?php foreach ($allUsers as $user): ?>
                                    <option value="<?php echo htmlspecialchars($user); ?>"><?php echo htmlspecialchars($user); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <?php endif; ?>
                            <div class="col-md-3">
                                <label for="backupType" class="form-label">Backup Type</label>
                                <select class="form-select" id="backupType" required>
                                    <option value="">Select Type</option>
                                    <option value="site">Site</option>
                                    <option value="userfiles">User Files</option>
                                    <option value="database">Database</option>
                                </select>
                            </div>
                            <div class="col-md-3" id="siteSelect" style="display: none;">
                                <label for="siteSelection" class="form-label">Select Site</label>
                                <select class="form-select" id="siteSelection">
                                    <option value="">Select Site</option>
                                    <?php foreach ($sites as $site): ?>
                                    <option value="<?php echo htmlspecialchars($site); ?>"><?php echo htmlspecialchars($site); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3" id="dbSelect" style="display: none;">
                                <label for="dbSelection" class="form-label">Select Database</label>
                                <select class="form-select" id="dbSelection">
                                    <option value="">Select Database</option>
                                    <?php foreach ($databases as $db): ?>
                                    <option value="<?php echo htmlspecialchars($db); ?>"><?php echo htmlspecialchars($db); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="targetSelection" class="form-label">Backup Target</label>
                                <select class="form-select" id="targetSelection" required>
                                    <option value="">Select Target</option>
                                    <?php foreach ($targets as $target): ?>
                                    <option value="<?php echo $target['id']; ?>">
                                        <?php echo htmlspecialchars($target['name']); ?>
                                        <?php if ($target['is_global']): ?>(Global)<?php endif; ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-12 mt-3">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fa fa-play"></i> Start Backup
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Backup Targets Section -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fa fa-bullseye"></i> Backup Targets</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Type</th>
                                    <th>Endpoint</th>
                                    <th>Bucket</th>
                                    <th>Retention</th>
                                    <th>Max Backups</th>
                                    <th>Global</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($targets as $target): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($target['name']); ?></td>
                                    <td><span class="badge bg-secondary"><?php echo strtoupper($target['type']); ?></span></td>
                                    <td><?php echo htmlspecialchars($target['endpoint']); ?></td>
                                    <td><?php echo htmlspecialchars($target['bucket']); ?></td>
                                    <td><?php echo $target['retention_days']; ?> days</td>
                                    <td><?php echo $target['max_backups']; ?></td>
                                    <td>
                                        <?php if ($target['is_global']): ?>
                                        <span class="badge bg-success">Yes</span>
                                        <?php else: ?>
                                        <span class="badge bg-secondary">No</span>
                                        <?php endif; ?>
                                        <?php if (!$target['is_global'] && $target['owner']): ?>
                                        <small class="text-muted d-block">(<?php echo htmlspecialchars($target['owner']); ?>)</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-primary" onclick="testTarget(<?php echo $target['id']; ?>)">
                                            <i class="fa fa-check"></i> Test
                                        </button>
                                        <?php if ($isRoot || $target['owner'] === AUTH_USER): ?>
                                        <button class="btn btn-sm btn-outline-warning" onclick="editTarget(<?php echo $target['id']; ?>)">
                                            <i class="fa fa-edit"></i> Edit
                                        </button>
                                        <button class="btn btn-sm btn-outline-danger" onclick="deleteTarget(<?php echo $target['id']; ?>)">
                                            <i class="fa fa-trash"></i> Delete
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scheduled Backups Section -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h5><i class="fa fa-clock-o"></i> Scheduled Backups</h5>
                            <small class="text-muted">
                                <i class="fa fa-info-circle"></i> Next run times shown in your local timezone
                            </small>
                        </div>
                        <div>
                            <?php if ($isRoot): ?>
                            <button class="btn btn-outline-info btn-sm me-2" onclick="runSchedulerNow()">
                                <i class="fa fa-refresh"></i> Run Scheduler Now
                            </button>
                            <?php endif; ?>
                            <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#scheduleModal" onclick="clearScheduleForm()">
                                <i class="fa fa-plus"></i> Add Schedule
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Type</th>
                                    <th>Resource</th>
                                    <?php if ($isRoot): ?><th>User</th><?php endif; ?>
                                    <th>Target</th>
                                    <th>Schedule</th>
                                    <th>Next Run</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($schedules as $schedule): ?>
                                <tr>
                                    <td>
                                        <span class="badge bg-<?php echo getBackupTypeColor($schedule['backup_type']); ?>">
                                            <?php echo ucfirst($schedule['backup_type']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($schedule['resource_name'] ?? 'All'); ?></td>
                                    <?php if ($isRoot): ?>
                                    <td><?php echo htmlspecialchars($schedule['user']); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($schedule['target_name']); ?></td>
                                    <td>
                                        <?php 
                                        $scheduleText = ucfirst($schedule['schedule_type']);
                                        if ($schedule['schedule_type'] === 'weekly' && $schedule['schedule_day'] !== null) {
                                            $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                                            $scheduleText .= ' (' . $days[$schedule['schedule_day']] . ')';
                                        } elseif ($schedule['schedule_type'] === 'monthly' && $schedule['schedule_day'] !== null) {
                                            $scheduleText .= ' (' . $schedule['schedule_day'] . ')';
                                        }
                                        echo $scheduleText . ' at ' . date('H:i', strtotime($schedule['schedule_time']));
                                        ?>
                                    </td>
                                    <td>
                                        <?php if ($schedule['next_run_at']): ?>
                                        <span class="local-time" data-utc="<?php echo date('c', strtotime($schedule['next_run_at'])); ?>">
                                            <?php echo date('Y-m-d H:i', strtotime($schedule['next_run_at'])); ?>
                                        </span>
                                        <?php else: ?>
                                        N/A
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?php echo $schedule['is_active'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $schedule['is_active'] ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($isRoot || $schedule['user'] === AUTH_USER): ?>
                                        <button class="btn btn-sm btn-outline-success" onclick="testSchedule(<?php echo $schedule['id']; ?>)">
                                            <i class="fa fa-play"></i> Test
                                        </button>
                                        <button class="btn btn-sm btn-outline-warning" onclick="editSchedule(<?php echo $schedule['id']; ?>)">
                                            <i class="fa fa-edit"></i> Edit
                                        </button>
                                        <button class="btn btn-sm btn-outline-danger" onclick="deleteSchedule(<?php echo $schedule['id']; ?>)">
                                            <i class="fa fa-trash"></i> Delete
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Backups Section -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5><i class="fa fa-history"></i> Recent Backups</h5>
                        <small class="text-muted">
                            <i class="fa fa-clock-o"></i> Times shown in your local timezone
                        </small>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Type</th>
                                    <th>User</th>
                                    <th>Target</th>
                                    <th>Size</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="backupsList">
                                <?php foreach ($recentBackups as $backup): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($backup['backup_name']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo getBackupTypeColor($backup['backup_type']); ?>">
                                            <?php echo ucfirst($backup['backup_type']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($backup['user']); ?></td>
                                    <td><?php echo htmlspecialchars($backup['target_name']); ?></td>
                                    <td><?php echo formatBytes($backup['backup_size']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo getStatusColor($backup['status']); ?>">
                                            <?php echo ucfirst($backup['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="local-time" data-utc="<?php echo date('c', strtotime($backup['started_at'])); ?>">
                                            <?php echo date('Y-m-d H:i:s', strtotime($backup['started_at'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($backup['status'] === 'completed'): ?>
                                        <button class="btn btn-sm btn-outline-success" onclick="downloadBackup(<?php echo $backup['id']; ?>)">
                                            <i class="fa fa-download"></i> Download
                                        </button>
                                        <?php endif; ?>
                                        <?php if ($isRoot || $backup['user'] === AUTH_USER): ?>
                                        <button class="btn btn-sm btn-outline-danger" onclick="deleteBackup(<?php echo $backup['id']; ?>)">
                                            <i class="fa fa-trash"></i> Delete
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Backup Target Modal -->
<div class="modal fade" id="targetModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add/Edit Backup Target</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="targetForm">
                <div class="modal-body">
                    <input type="hidden" id="targetId" name="targetId">
                    <div class="row">
                        <div class="col-md-6">
                            <label for="targetName" class="form-label">Name</label>
                            <input type="text" class="form-control" id="targetName" name="name" required>
                        </div>
                        <div class="col-md-6">
                            <label for="targetType" class="form-label">Type</label>
                            <select class="form-select" id="targetType" name="type" required>
                                <option value="s3">S3 Compatible</option>
                            </select>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="targetEndpoint" class="form-label">Endpoint</label>
                            <input type="url" class="form-control" id="targetEndpoint" name="endpoint" required>
                        </div>
                        <div class="col-md-6">
                            <label for="targetBucket" class="form-label">Bucket</label>
                            <input type="text" class="form-control" id="targetBucket" name="bucket" required>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="targetAccessKey" class="form-label">Access Key</label>
                            <input type="text" class="form-control" id="targetAccessKey" name="access_key" required>
                        </div>
                        <div class="col-md-6">
                            <label for="targetSecretKey" class="form-label">Secret Key</label>
                            <input type="password" class="form-control" id="targetSecretKey" name="secret_key" required>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="targetRegion" class="form-label">Region</label>
                            <input type="text" class="form-control" id="targetRegion" name="region" value="us-east-1">
                        </div>
                        <div class="col-md-6">
                            <label for="targetPathPrefix" class="form-label">Path Prefix</label>
                            <input type="text" class="form-control" id="targetPathPrefix" name="path_prefix" placeholder="whp-backups">
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="targetRetention" class="form-label">Retention Days</label>
                            <input type="number" class="form-control" id="targetRetention" name="retention_days" value="30" min="1">
                        </div>
                        <div class="col-md-6">
                            <label for="targetMaxBackups" class="form-label">Max Backups</label>
                            <input type="number" class="form-control" id="targetMaxBackups" name="max_backups" value="10" min="1">
                        </div>
                    </div>
                    <?php if ($isRoot): ?>
                    <div class="row mt-3">
                        <div class="col-md-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="targetGlobal" name="is_global">
                                <label class="form-check-label" for="targetGlobal">
                                    Global target (available to all users)
                                </label>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Target</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Schedule Modal -->
<div class="modal fade" id="scheduleModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add/Edit Backup Schedule</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="scheduleForm">
                <div class="modal-body">
                    <input type="hidden" id="scheduleId" name="schedule_id">
                    <div class="row">
                        <?php if ($isRoot): ?>
                        <div class="col-md-6">
                            <label for="scheduleUser" class="form-label">User</label>
                            <select class="form-select" id="scheduleUser" name="schedule_user" required>
                                <option value="">Select User</option>
                                <?php foreach ($allUsers as $user): ?>
                                <option value="<?php echo htmlspecialchars($user); ?>"><?php echo htmlspecialchars($user); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        <div class="col-md-6">
                            <label for="scheduleBackupType" class="form-label">Backup Type</label>
                            <select class="form-select" id="scheduleBackupType" name="backup_type" required>
                                <option value="">Select Type</option>
                                <option value="site">Site</option>
                                <option value="userfiles">User Files</option>
                                <option value="database">Database</option>
                            </select>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="scheduleResource" class="form-label">Resource Name</label>
                            <input type="text" class="form-control" id="scheduleResource" name="resource_name" placeholder="Leave blank for all">
                            <small class="form-text text-muted">For sites: domain name, for databases: database name</small>
                        </div>
                        <div class="col-md-6">
                            <label for="scheduleTarget" class="form-label">Backup Target</label>
                            <select class="form-select" id="scheduleTarget" name="target_id" required>
                                <option value="">Select Target</option>
                                <?php foreach ($targets as $target): ?>
                                <option value="<?php echo $target['id']; ?>">
                                    <?php echo htmlspecialchars($target['name']); ?>
                                    <?php if ($target['is_global']): ?>(Global)<?php endif; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-4">
                            <label for="scheduleType" class="form-label">Schedule Type</label>
                            <select class="form-select" id="scheduleType" name="schedule_type" required>
                                <option value="">Select Type</option>
                                <option value="daily">Daily</option>
                                <option value="weekly">Weekly</option>
                                <option value="monthly">Monthly</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="scheduleTime" class="form-label">Time</label>
                            <input type="time" class="form-control" id="scheduleTime" name="schedule_time" value="02:00">
                        </div>
                        <div class="col-md-4" id="scheduleDayContainer" style="display: none;">
                            <label for="scheduleDay" class="form-label">Day</label>
                            <select class="form-select" id="scheduleDay" name="schedule_day">
                                <option value="">Select Day</option>
                            </select>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-md-6">
                            <label for="scheduleRetention" class="form-label">Max Retention</label>
                            <input type="number" class="form-control" id="scheduleRetention" name="max_retention" value="30" min="1">
                            <small class="form-text text-muted">Maximum number of backups to keep</small>
                        </div>
                        <div class="col-md-6">
                            <div class="form-check mt-4">
                                <input class="form-check-input" type="checkbox" id="scheduleActive" name="is_active" checked>
                                <label class="form-check-label" for="scheduleActive">
                                    Active
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Schedule</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Show/hide selection fields based on backup type
document.getElementById('backupType').addEventListener('change', function() {
    const siteSelect = document.getElementById('siteSelect');
    const dbSelect = document.getElementById('dbSelect');
    
    siteSelect.style.display = 'none';
    dbSelect.style.display = 'none';
    
    if (this.value === 'site') {
        siteSelect.style.display = 'block';
    } else if (this.value === 'database') {
        dbSelect.style.display = 'block';
    }
});

// Handle backup form submission
document.getElementById('backupForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'create_backup');
    formData.append('backup_type', document.getElementById('backupType').value);
    formData.append('target_id', document.getElementById('targetSelection').value);
    formData.append('nonce', '<?php echo $_SESSION['nonce']; ?>');
    
    <?php if ($isRoot): ?>
    if (document.getElementById('backupUser')) {
        formData.append('backup_user', document.getElementById('backupUser').value);
    }
    <?php endif; ?>
    
    if (document.getElementById('backupType').value === 'site') {
        formData.append('site', document.getElementById('siteSelection').value);
    } else if (document.getElementById('backupType').value === 'database') {
        formData.append('database', document.getElementById('dbSelection').value);
    }
    
    fetch('api/backup-actions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Backup started successfully!', 'success');
            setTimeout(() => location.reload(), 2000);
        } else {
            showAlert('Error: ' + data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error: ' + error.message, 'danger');
    });
});

// Handle target form submission
document.getElementById('targetForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', document.getElementById('targetId').value ? 'update_target' : 'create_target');
    formData.append('nonce', '<?php echo $_SESSION['nonce']; ?>');
    
    fetch('api/backup-actions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Target saved successfully!', 'success');
            bootstrap.Modal.getInstance(document.getElementById('targetModal')).hide();
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Error: ' + data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error: ' + error.message, 'danger');
    });
});

// Utility functions
function testTarget(targetId) {
    fetch('api/backup-actions.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=test_target&target_id=${targetId}&csrf_token=<?php echo get_csrf_token(); ?>`
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Connection test successful!', 'success');
        } else {
            showAlert('Connection test failed: ' + data.message, 'danger');
        }
    });
}

function downloadBackup(backupId) {
    window.location.href = 'api/backup-actions.php?action=download&backup_id=' + backupId + '&nonce=<?php echo $_SESSION['nonce']; ?>';
}

function deleteBackup(backupId) {
    if (confirm('Are you sure you want to delete this backup?')) {
        fetch('api/backup-actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=delete_backup&backup_id=${backupId}&nonce=<?php echo $_SESSION['nonce']; ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Backup deleted successfully!', 'success');
                setTimeout(() => location.reload(), 1000);
            } else {
                showAlert('Error: ' + data.message, 'danger');
            }
        });
    }
}

function editTarget(targetId) {
    // Fetch target details and populate the form
    fetch('api/backup-actions.php?action=get_target&target_id=' + targetId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const target = data.target;
                
                // Populate form fields
                document.getElementById('targetId').value = target.id;
                document.getElementById('targetName').value = target.name;
                document.getElementById('targetType').value = target.type;
                document.getElementById('targetEndpoint').value = target.endpoint;
                document.getElementById('targetBucket').value = target.bucket;
                document.getElementById('targetAccessKey').value = target.access_key;
                document.getElementById('targetSecretKey').value = target.secret_key;
                document.getElementById('targetRegion').value = target.region;
                document.getElementById('targetPathPrefix').value = target.path_prefix;
                document.getElementById('targetRetention').value = target.retention_days;
                document.getElementById('targetMaxBackups').value = target.max_backups;
                
                if (document.getElementById('targetGlobal')) {
                    document.getElementById('targetGlobal').checked = target.is_global == 1;
                }
                
                // Update modal title
                document.querySelector('#targetModal .modal-title').textContent = 'Edit Backup Target';
                
                // Show modal
                new bootstrap.Modal(document.getElementById('targetModal')).show();
            } else {
                showAlert('Error loading target details: ' + data.message, 'danger');
            }
        })
        .catch(error => {
            showAlert('Error loading target details', 'danger');
        });
}

function deleteTarget(targetId) {
    if (confirm('Are you sure you want to delete this backup target?')) {
        fetch('api/backup-actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=delete_target&target_id=${targetId}&nonce=<?php echo $_SESSION['nonce']; ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Target deleted successfully!', 'success');
                setTimeout(() => location.reload(), 1000);
            } else {
                showAlert('Error: ' + data.message, 'danger');
            }
        });
    }
}

function clearTargetForm() {
    // Clear all form fields
    document.getElementById('targetId').value = '';
    document.getElementById('targetName').value = '';
    document.getElementById('targetType').value = 's3';
    document.getElementById('targetEndpoint').value = '';
    document.getElementById('targetBucket').value = '';
    document.getElementById('targetAccessKey').value = '';
    document.getElementById('targetSecretKey').value = '';
    document.getElementById('targetRegion').value = 'us-east-1';
    document.getElementById('targetPathPrefix').value = '';
    document.getElementById('targetRetention').value = '30';
    document.getElementById('targetMaxBackups').value = '10';
    
    if (document.getElementById('targetGlobal')) {
        document.getElementById('targetGlobal').checked = false;
    }
    
    // Reset modal title
    document.querySelector('#targetModal .modal-title').textContent = 'Add Backup Target';
}

// Schedule functions
document.getElementById('scheduleType').addEventListener('change', function() {
    const dayContainer = document.getElementById('scheduleDayContainer');
    const daySelect = document.getElementById('scheduleDay');
    
    daySelect.innerHTML = '<option value="">Select Day</option>';
    
    if (this.value === 'weekly') {
        dayContainer.style.display = 'block';
        const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        days.forEach((day, index) => {
            const option = document.createElement('option');
            option.value = index;
            option.textContent = day;
            daySelect.appendChild(option);
        });
    } else if (this.value === 'monthly') {
        dayContainer.style.display = 'block';
        for (let i = 1; i <= 31; i++) {
            const option = document.createElement('option');
            option.value = i;
            option.textContent = i;
            daySelect.appendChild(option);
        }
    } else {
        dayContainer.style.display = 'none';
    }
});

document.getElementById('scheduleForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', document.getElementById('scheduleId').value ? 'update_schedule' : 'create_schedule');
    formData.append('nonce', '<?php echo $_SESSION['nonce']; ?>');
    
    fetch('api/backup-actions.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Schedule saved successfully!', 'success');
            bootstrap.Modal.getInstance(document.getElementById('scheduleModal')).hide();
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Error: ' + data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error: ' + error.message, 'danger');
    });
});

function editSchedule(scheduleId) {
    // Fetch schedule details and populate the form
    fetch('api/backup-actions.php?action=get_schedule&schedule_id=' + scheduleId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const schedule = data.schedule;
                
                // Populate form fields
                document.getElementById('scheduleId').value = schedule.id;
                document.getElementById('scheduleBackupType').value = schedule.backup_type;
                document.getElementById('scheduleResource').value = schedule.resource_name || '';
                document.getElementById('scheduleTarget').value = schedule.target_id;
                document.getElementById('scheduleType').value = schedule.schedule_type;
                document.getElementById('scheduleTime').value = schedule.schedule_time;
                document.getElementById('scheduleRetention').value = schedule.max_retention;
                document.getElementById('scheduleActive').checked = schedule.is_active == 1;
                
                <?php if ($isRoot): ?>
                if (document.getElementById('scheduleUser')) {
                    document.getElementById('scheduleUser').value = schedule.user;
                }
                <?php endif; ?>
                
                // Handle schedule day based on type
                const scheduleTypeElement = document.getElementById('scheduleType');
                const dayContainer = document.getElementById('scheduleDayContainer');
                const daySelect = document.getElementById('scheduleDay');
                
                daySelect.innerHTML = '<option value="">Select Day</option>';
                
                if (schedule.schedule_type === 'weekly') {
                    dayContainer.style.display = 'block';
                    const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                    days.forEach((day, index) => {
                        const option = document.createElement('option');
                        option.value = index;
                        option.textContent = day;
                        if (schedule.schedule_day == index) {
                            option.selected = true;
                        }
                        daySelect.appendChild(option);
                    });
                } else if (schedule.schedule_type === 'monthly') {
                    dayContainer.style.display = 'block';
                    for (let i = 1; i <= 31; i++) {
                        const option = document.createElement('option');
                        option.value = i;
                        option.textContent = i;
                        if (schedule.schedule_day == i) {
                            option.selected = true;
                        }
                        daySelect.appendChild(option);
                    }
                } else {
                    dayContainer.style.display = 'none';
                }
                
                // Update modal title
                document.querySelector('#scheduleModal .modal-title').textContent = 'Edit Backup Schedule';
                
                // Show modal
                new bootstrap.Modal(document.getElementById('scheduleModal')).show();
            } else {
                showAlert('Error loading schedule details: ' + data.message, 'danger');
            }
        })
        .catch(error => {
            showAlert('Error loading schedule details', 'danger');
        });
}

function testSchedule(scheduleId) {
    if (confirm('This will create a test backup based on this schedule. Continue?')) {
        fetch('api/backup-actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=test_schedule&schedule_id=${scheduleId}&nonce=<?php echo $_SESSION['nonce']; ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Test backup started successfully! Backup ID: ' + data.backup_id, 'success');
                setTimeout(() => location.reload(), 2000);
            } else {
                showAlert('Error: ' + data.message, 'danger');
            }
        })
        .catch(error => {
            showAlert('Error: ' + error.message, 'danger');
        });
    }
}

function deleteSchedule(scheduleId) {
    if (confirm('Are you sure you want to delete this schedule?')) {
        fetch('api/backup-actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=delete_schedule&schedule_id=${scheduleId}&nonce=<?php echo $_SESSION['nonce']; ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Schedule deleted successfully!', 'success');
                setTimeout(() => location.reload(), 1000);
            } else {
                showAlert('Error: ' + data.message, 'danger');
            }
        });
    }
}

function clearScheduleForm() {
    document.getElementById('scheduleId').value = '';
    document.getElementById('scheduleBackupType').value = '';
    document.getElementById('scheduleResource').value = '';
    document.getElementById('scheduleTarget').value = '';
    document.getElementById('scheduleType').value = '';
    document.getElementById('scheduleTime').value = '02:00';
    document.getElementById('scheduleDay').value = '';
    document.getElementById('scheduleRetention').value = '30';
    document.getElementById('scheduleActive').checked = true;
    document.getElementById('scheduleDayContainer').style.display = 'none';
    
    <?php if ($isRoot): ?>
    if (document.getElementById('scheduleUser')) {
        document.getElementById('scheduleUser').value = '';
    }
    <?php endif; ?>
    
    document.querySelector('#scheduleModal .modal-title').textContent = 'Add Backup Schedule';
}

function runSchedulerNow() {
    if (confirm('This will run the backup scheduler immediately to process any due scheduled backups. Continue?')) {
        fetch('api/backup-actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=run_scheduler&nonce=<?php echo $_SESSION['nonce']; ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Scheduler started successfully! Check recent backups for results.', 'success');
                setTimeout(() => location.reload(), 3000);
            } else {
                showAlert('Error: ' + data.message, 'danger');
            }
        })
        .catch(error => {
            showAlert('Error: ' + error.message, 'danger');
        });
    }
}

function showAlert(message, type) {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.querySelector('.container-fluid').insertBefore(alertDiv, document.querySelector('.container-fluid').firstChild);
    
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}

// Convert UTC times to local timezone
function convertToLocalTime() {
    document.querySelectorAll('.local-time').forEach(element => {
        const utcTime = element.getAttribute('data-utc');
        if (utcTime && !element.getAttribute('data-converted')) {
            try {
                // Store original server time for tooltip
                const originalText = element.textContent.trim();
                
                const date = new Date(utcTime);
                const hasSeconds = originalText.includes(':') && originalText.split(':').length > 2;
                
                const localTime = date.toLocaleString(undefined, {
                    year: 'numeric',
                    month: '2-digit',
                    day: '2-digit',
                    hour: '2-digit',
                    minute: '2-digit',
                    second: hasSeconds ? '2-digit' : undefined,
                    hour12: false
                });
                
                element.textContent = localTime;
                element.title = `Server time: ${originalText}`;
                element.setAttribute('data-converted', 'true');
            } catch (error) {
                console.warn('Failed to convert time:', utcTime, error);
            }
        }
    });
}

// Convert times when page loads
document.addEventListener('DOMContentLoaded', convertToLocalTime);
</script>

<?php
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

function getBackupTypeColor($type) {
    switch ($type) {
        case 'site': return 'primary';
        case 'userfiles': return 'info';
        case 'database': return 'warning';
        default: return 'secondary';
    }
}

function getStatusColor($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'running': return 'warning';
        case 'failed': return 'danger';
        case 'pending': return 'info';
        default: return 'secondary';
    }
}
?>