<?php
// Set content type to JSON
header('Content-Type: application/json');

// Check API permission
check_api_permission('sites');

require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('/docker/whp/web/libs/docker-api.php');

$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();
if (!$db) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed']);
    exit;
}

$DockerAPI = new docker_api();

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST' && isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') === 0) {
    $input = json_decode(file_get_contents('php://input'), true);
    if (is_array($input)) {
        foreach ($input as $k => $v) {
            $_POST[$k] = $v;
        }
    }
}

$action = isset($_GET['action']) ? $_GET['action'] : (isset($_POST['action']) ? $_POST['action'] : '');

// Handle different HTTP methods and actions
switch ($method) {
    case 'GET':
        switch ($action) {
            case 'get_site_details':
                if (!empty($_GET['site_id'])) {
                    $site_id = intval($_GET['site_id']);
                    
                    global $db;
                    try {
                        // Get site details
                        $stmt = $db->prepare("
                            SELECT 
                                s.*,
                                d.domain_name as primary_domain,
                                ct.name as container_type_name,
                                ct.base_image,
                                ct.listen_port,
                                ct.description as container_type_description
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            JOIN whp.container_types ct ON s.container_type_id = ct.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Get site domains
                        $stmt = $db->prepare("
                            SELECT 
                                sd.*,
                                d.domain_name
                            FROM whp.site_domains sd
                            JOIN whp.domains d ON sd.domain_id = d.id
                            WHERE sd.site_id = ?
                            ORDER BY sd.is_primary DESC, d.domain_name ASC
                        ");
                        $stmt->execute([$site_id]);
                        $site_domains = $stmt->fetchAll();
                        
                        // Build site_fqdns array for JSON response
                        $site_fqdns = [];
                        foreach ($site_domains as $sd) {
                            $site_fqdns[] = [
                                'fqdn' => $sd['fqdn'] ?: $sd['domain_name'], // Use FQDN if available, fallback to domain_name
                                'is_primary' => $sd['is_primary'],
                                'domain' => $sd['domain_name'],
                            ];
                        }
                        
                        // Get site containers
                        $stmt = $db->prepare("
                            SELECT *
                            FROM whp.site_containers
                            WHERE site_id = ?
                            ORDER BY container_number ASC
                        ");
                        $stmt->execute([$site_id]);
                        $site_containers = $stmt->fetchAll();
                        
                        // Get Docker stats for containers
                        $container_stats = [];
                        if (!empty($site_containers)) {
                            $container_ids = array_column($site_containers, 'container_id');
                            $ids_string = implode(' ', array_map('escapeshellarg', $container_ids));
                            $stats_command = "docker stats --no-stream --format '{\"name\":\"{{.Container}}\",\"cpu\":\"{{.CPUPerc}}\",\"memory\":\"{{.MemUsage}}\",\"memory_pct\":\"{{.MemPerc}}\"}' " . $ids_string;
                            $stats_output = shell_exec($stats_command);
                            
                            if ($stats_output) {
                                $stats_lines = explode("\n", trim($stats_output));
                                foreach ($stats_lines as $line) {
                                    if (!empty($line)) {
                                        $stat = json_decode($line, true);
                                        if ($stat) {
                                            $container_stats[$stat['name']] = $stat;
                                        }
                                    }
                                }
                            }
                        }
                        
                        // Get available domains for this user
                        $stmt = $db->prepare("
                            SELECT d.*
                            FROM whp.domains d
                            WHERE d.username = ? AND d.active = 1
                            AND d.id NOT IN (
                                SELECT domain_id FROM whp.site_domains WHERE site_id = ?
                            )
                            ORDER BY d.domain_name ASC
                        ");
                        $stmt->execute([$site['username'], $site_id]);
                        $available_domains = $stmt->fetchAll();
                        
                        // Generate HTML for modal
                        $html = '<div class="row">';
                        
                        // Container Status (move to top, make compact)
                        $html .= '<div class="col-12">';
                        $html .= '<div class="site-section-header" style="margin-top:0;">Container Status</div>';
                        if (empty($site_containers)) {
                            $html .= '<div class="alert alert-info p-2 m-0" style="font-size:0.95rem;max-width:320px;">No containers created yet</div>';
                        } else {
                            $html .= '<div class="list-group list-group-flush" style="max-width:500px;">';
                            foreach ($site_containers as $container) {
                                $status_class = $container['status'] === 'running' ? 'success' : 'secondary';
                                // Generate container name from primary domain and container number (same as Docker naming)
                                $container_name = $site['primary_domain'] . '-' . str_pad($container['container_number'], 2, '0', STR_PAD_LEFT);
                                
                                $html .= '<div class="list-group-item py-2 px-3">';
                                $html .= '<div class="d-flex justify-content-between align-items-center mb-1">';
                                $html .= '<span style="font-size:0.95rem; font-weight:500;">' . htmlspecialchars($container_name) . '</span>';
                                $html .= '<span class="badge bg-' . $status_class . '" style="font-size:0.95rem;">' . ucfirst($container['status']) . '</span>';
                                $html .= '</div>';
                                
                                // Add container stats if available and running
                                if ($container['status'] === 'running' && isset($container_stats[$container['container_id']])) {
                                    $stats = $container_stats[$container['container_id']];
                                    $html .= '<div class="row g-2 mt-2">';
                                    $html .= '<div class="col-6">';
                                    $html .= '<div class="d-flex align-items-center">';
                                    $html .= '<i class="fa fa-microchip text-primary me-1"></i>';
                                    $html .= '<small style="color: #212529 !important; font-weight: 700 !important;">CPU:</small>';
                                    $html .= '<span class="badge bg-primary ms-1" style="font-size:0.8rem;">' . htmlspecialchars($stats['cpu']) . '</span>';
                                    $html .= '</div>';
                                    $html .= '</div>';
                                    $html .= '<div class="col-6">';
                                    $html .= '<div class="d-flex align-items-center">';
                                    $html .= '<i class="fa fa-memory text-info me-1"></i>';
                                    $html .= '<small style="color: #212529 !important; font-weight: 700 !important;">Memory:</small>';
                                    $html .= '<span class="badge bg-info ms-1" style="font-size:0.8rem;">' . htmlspecialchars($stats['memory_pct']) . '</span>';
                                    $html .= '</div>';
                                    $html .= '</div>';
                                    $html .= '</div>';
                                } elseif ($container['status'] !== 'running') {
                                    $html .= '<div class="text-muted mt-1" style="font-size:0.85rem;">Container not running</div>';
                                }
                                
                                $html .= '</div>';
                            }
                            $html .= '</div>';
                        }
                        $html .= '</div>';
                        
                        // Site Information (now below container status)
                        $html .= '<div class="col-md-6">';
                        $html .= '<div class="site-section-header">Site Information</div>';
                        $html .= '<table class="table table-sm">';
                        $html .= '<tr><td><strong>Name:</strong></td><td>' . htmlspecialchars($site['site_name']) . '</td></tr>';
                        $html .= '<tr><td><strong>Primary Domain:</strong></td><td>' . htmlspecialchars($site['primary_domain']) . '</td></tr>';
                        $html .= '<tr><td><strong>Container Type:</strong></td><td>' . htmlspecialchars($site['container_type_name']) . '</td></tr>';
                        $html .= '<tr><td><strong>Container Count:</strong></td><td>' . $site['container_count'] . '</td></tr>';
                        $html .= '<tr><td><strong>CPU per Container:</strong></td><td>' . $site['cpu_per_container'] . ' cores</td></tr>';
                        $html .= '<tr><td><strong>Memory per Container:</strong></td><td>' . $site['memory_per_container'] . ' MB</td></tr>';
                        $html .= '<tr><td><strong>Status:</strong></td><td>' . ($site['active'] ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-secondary">Inactive</span>') . '</td></tr>';
                        $html .= '</table>';
                        $html .= '</div>';
                        
                        // Domains Section
                        $html .= '<div class="row mt-3">';
                        $html .= '<div class="col-12">';
                        $html .= '<div class="site-section-header">Associated Domains</div>';
                        $html .= '<div class="list-group">';
                        foreach ($site_domains as $site_domain) {
                            $display_fqdn = $site_domain['fqdn'] ?: $site_domain['domain_name']; // Use FQDN if available
                            $html .= '<div class="list-group-item d-flex justify-content-between align-items-center">';
                            $html .= '<span>';
                            $html .= '<a href="https://' . htmlspecialchars($display_fqdn) . '" target="_blank" class="text-decoration-none">';
                            $html .= htmlspecialchars($display_fqdn);
                            $html .= '</a>';
                            if ($site_domain['is_primary']) {
                                $html .= ' <span class="badge bg-primary ms-2">Primary</span>';
                            }
                            $html .= '</span>';
                            // Remove button/form is omitted
                            $html .= '</div>';
                        }
                        $html .= '</div>';
                        $html .= '</div>';
                        $html .= '</div>'; // End row
                        // (Removed Add Domain to Site form)
                        
                        echo json_encode(array(
                            'status' => 'success',
                            'html' => $html,
                            'site' => array(
                                'id' => $site['id'],
                                'site_name' => $site['site_name'],
                                'primary_domain_id' => $site['primary_domain_id'],
                                'primary_domain' => $site['primary_domain'],
                                'container_type_id' => $site['container_type_id'],
                                'container_type_name' => $site['container_type_name'],
                                'container_count' => $site['container_count'],
                                'cpu_per_container' => $site['cpu_per_container'],
                                'memory_per_container' => $site['memory_per_container'],
                                'user_variables' => $site['user_variables'],
                                'active' => $site['active'],
                                'username' => $site['username'],
                                'auto_recreate_on_update' => $site['auto_recreate_on_update'],
                                'fqdns' => $site_fqdns
                            )
                        ));
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'get_user_limits':
                global $db;
                try {
                    $stmt = $db->prepare("SELECT * FROM whp.client_allowances WHERE username = ?");
                    $stmt->execute([AUTH_USER]);
                    $limits = $stmt->fetch();
                    
                    echo json_encode(array(
                        'status' => 'success',
                        'data' => $limits ?: array()
                    ));
                } catch (PDOException $e) {
                    http_response_code(500);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Database error: ' . $e->getMessage()
                    ));
                }
                break;
                
            default:
                http_response_code(400);
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Invalid action'
                ));
                break;
        }
        break;
        
    case 'POST':
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            $input = $_POST;
        }
        
        switch ($action) {
            case 'create_containers':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site details
                        $stmt = $db->prepare("
                            SELECT 
                                s.*,
                                d.domain_name as primary_domain,
                                ct.base_image,
                                ct.startup_env,
                                ct.mount_options
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            JOIN whp.container_types ct ON s.container_type_id = ct.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Get user UID
                        $uid = trim(shell_exec("id -u " . escapeshellarg($site['username'])));
                        if (!is_numeric($uid)) {
                            http_response_code(400);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Invalid user'
                            ));
                            break;
                        }
                        
                        $created_containers = [];
                        $errors = [];
                        
                        // Create containers
                        for ($i = 1; $i <= $site['container_count']; $i++) {
                            // Check if container already exists
                            $stmt = $db->prepare("SELECT container_id FROM whp.site_containers WHERE site_id = ? AND container_number = ?");
                            $stmt->execute([$site_id, $i]);
                            if ($stmt->fetch()) {
                                $errors[] = "Container $i already exists";
                                continue;
                            }
                            
                            // Create container name using domain-based naming
                            $container_name = $site['primary_domain'] . '-' . str_pad($i, 2, '0', STR_PAD_LEFT);
                            
                            // Prepare environment variables
                            $env_vars = [
                                'WHP_USER=' . $site['username'],
                                'WHP_UID=' . $uid,
                                'WHP_DOMAIN=' . $site['primary_domain'],
                                'WHP_CONTAINER_NUMBER=' . $i
                            ];
                            
                            // Add user variables
                            if (!empty($site['user_variables'])) {
                                $user_vars = json_decode($site['user_variables'], true);
                                foreach ($user_vars as $var_name => $var_value) {
                                    $env_vars[] = $var_name . '=' . $var_value;
                                }
                            }
                            
                            // Process startup environment variables
                            if (!empty($site['startup_env'])) {
                                $startup_env = json_decode($site['startup_env'], true);
                                if ($startup_env) {
                                    foreach ($startup_env as $var_name => $var_value) {
                                        // Replace dynamic variables using centralized function
                                        $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                        $env_vars[] = $var_name . '=' . $var_value;
                                    }
                                }
                            }
                            
                            // Prepare volumes
                            $volumes = [
                                '/docker/users/' . $site['username'] . '/' . $site['primary_domain'] . ':/var/www/html'
                            ];
                            
                            // Process mount options
                            if (!empty($site['mount_options'])) {
                                $mount_options = json_decode($site['mount_options'], true);
                                if ($mount_options && isset($mount_options['volumes'])) {
                                    foreach ($mount_options['volumes'] as $volume) {
                                        // Replace dynamic variables in both source and target paths
                                        $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                        $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                        $volumes[] = $source . ':' . $target;
                                    }
                                }
                            }
                            
                            // Create container config
                            $container_config = [
                                'image' => $site['base_image'],
                                'name' => $container_name,
                                'labels' => [
                                    'whp.user' => $site['username'],
                                    'whp.domain' => $site['primary_domain'],
                                    'whp.site_id' => strval($site_id),
                                    'whp.container_number' => strval($i)
                                ],
                                'env' => $env_vars,
                                'volumes' => $volumes,
                                'cpu' => $site['cpu_per_container'],
                                'memory' => $site['memory_per_container'],
                                'network' => 'client-net'
                            ];
                            
                            $result = $DockerAPI->create_container($container_config);
                            
                            if ($result['success']) {
                                // Save container info to database
                                $stmt = $db->prepare("
                                    INSERT INTO whp.site_containers (site_id, container_id, container_number, status) 
                                    VALUES (?, ?, ?, 'running')
                                ");
                                $stmt->execute([$site_id, $result['container_id'], $i]);
                                
                                $created_containers[] = [
                                    'container_number' => $i,
                                    'container_id' => $result['container_id']
                                ];
                            } else {
                                $errors[] = "Failed to create container $i: " . $result['error'];
                            }
                        }
                        
                        // Configure HAProxy if containers were created successfully
                        if (!empty($created_containers)) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, $site);
                            
                            if (!$haproxy_result['success']) {
                                $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        
                        if (empty($errors)) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'All containers created successfully and HAProxy configured',
                                'containers' => $created_containers
                            ));
                        } else {
                            echo json_encode(array(
                                'status' => 'partial',
                                'message' => 'Some containers created with errors',
                                'containers' => $created_containers,
                                'errors' => $errors
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'start_containers':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site containers
                        $stmt = $db->prepare("SELECT * FROM whp.site_containers WHERE site_id = ?");
                        $stmt->execute([$site_id]);
                        $containers = $stmt->fetchAll();
                        
                        $started_containers = [];
                        $errors = [];
                        
                        foreach ($containers as $container) {
                            $result = $DockerAPI->start_container($container['container_id']);
                            
                            if ($result['success']) {
                                // Update status in database
                                $stmt = $db->prepare("UPDATE whp.site_containers SET status = 'running' WHERE container_id = ?");
                                $stmt->execute([$container['container_id']]);
                                
                                $started_containers[] = $container['container_number'];
                            } else {
                                $errors[] = "Failed to start container " . $container['container_number'] . ": " . $result['error'];
                            }
                        }
                        
                        // Configure HAProxy after starting containers
                        if (!empty($started_containers)) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, null);
                            
                            if (!$haproxy_result['success']) {
                                $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        
                        if (empty($errors)) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'All containers started successfully and HAProxy configured',
                                'containers' => $started_containers
                            ));
                        } else {
                            echo json_encode(array(
                                'status' => 'partial',
                                'message' => 'Some containers started with errors',
                                'containers' => $started_containers,
                                'errors' => $errors
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'stop_containers':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site containers
                        $stmt = $db->prepare("SELECT * FROM whp.site_containers WHERE site_id = ?");
                        $stmt->execute([$site_id]);
                        $containers = $stmt->fetchAll();
                        
                        $stopped_containers = [];
                        $errors = [];
                        
                        foreach ($containers as $container) {
                            $result = $DockerAPI->stop_container($container['container_id']);
                            
                            if ($result['success']) {
                                // Update status in database
                                $stmt = $db->prepare("UPDATE whp.site_containers SET status = 'stopped' WHERE container_id = ?");
                                $stmt->execute([$container['container_id']]);
                                
                                $stopped_containers[] = $container['container_number'];
                            } else {
                                $errors[] = "Failed to stop container " . $container['container_number'] . ": " . $result['error'];
                            }
                        }
                        
                        // Configure HAProxy after stopping containers
                        if (!empty($stopped_containers)) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, null);
                            
                            if (!$haproxy_result['success']) {
                                $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        
                        if (empty($errors)) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'All containers stopped successfully and HAProxy updated',
                                'containers' => $stopped_containers
                            ));
                        } else {
                            echo json_encode(array(
                                'status' => 'partial',
                                'message' => 'Some containers stopped with errors',
                                'containers' => $stopped_containers,
                                'errors' => $errors
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'update_site':
                // Update site settings and recreate containers
                if (!empty($input['site_id']) && !empty($input['site_name']) && !empty($input['primary_domain_id']) && !empty($input['container_type_id'])) {
                    $site_id = intval($input['site_id']);
                    $site_name = trim($input['site_name']);
                    $primary_domain_id = trim($input['primary_domain_id']); // Accept FQDN
                    $container_type_id = intval($input['container_type_id']);
                    $container_count = intval($input['container_count'] ?? 1);
                    $cpu_per_container = floatval($input['cpu_per_container'] ?? 0.25);
                    $memory_per_container = intval($input['memory_per_container'] ?? 256);
                    $user_variables = !empty($input['user_variables']) ? $input['user_variables'] : '{}';

                    try {
                        // Get current site info
                        $stmt = $db->prepare("SELECT * FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(['status' => 'error', 'message' => 'Site not found']);
                            break;
                        }
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
                            break;
                        }
                        
                        // Validate container type
                        $stmt = $db->prepare("SELECT * FROM whp.container_types WHERE id = ? AND active = 1");
                        $stmt->execute([$container_type_id]);
                        $container_type = $stmt->fetch();
                        
                        if (!$container_type) {
                            http_response_code(400);
                            echo json_encode(['status' => 'error', 'message' => 'Invalid or inactive container type']);
                            break;
                        }
                        
                        // Validate resource requirements
                        if ($cpu_per_container < $container_type['min_cpu']) {
                            http_response_code(400);
                            echo json_encode(['status' => 'error', 'message' => 'CPU per container must be at least ' . $container_type['min_cpu'] . ' cores']);
                            break;
                        }
                        
                        if ($memory_per_container < $container_type['min_memory']) {
                            http_response_code(400);
                            echo json_encode(['status' => 'error', 'message' => 'Memory per container must be at least ' . $container_type['min_memory'] . ' MB']);
                            break;
                        }
                        
                        // Check user resource limits (exclude current site from calculations)
                        if (!$is_root) {
                            require_once('/docker/whp/web/libs/mysqlmgmt.php');
                            $MySQLMgmt = new mysqlmgmt();
                            
                            $stmt = $db->prepare("SELECT max_cpu_allowance, max_memory_allowance FROM whp.client_allowances WHERE username = ?");
                            $stmt->execute([$site['username']]);
                            $user_limits = $stmt->fetch();
                            
                            if ($user_limits) {
                                // Get current resource usage excluding this site
                                $usage = $MySQLMgmt->getUserResourceUsage($site['username']);
                                $current_cpu_used = floatval($usage['cpu_used'] ?? 0);
                                $current_memory_used = intval($usage['memory_used'] ?? 0);
                                
                                // Subtract current site resources from usage
                                $current_site_cpu = $site['container_count'] * $site['cpu_per_container'];
                                $current_site_memory = $site['container_count'] * $site['memory_per_container'];
                                $adjusted_cpu_used = $current_cpu_used - $current_site_cpu;
                                $adjusted_memory_used = $current_memory_used - $current_site_memory;
                                
                                // Calculate new site resources (use input values for new configuration)
                                $new_site_cpu = $cpu_per_container * $container_count;
                                $new_site_memory = $memory_per_container * $container_count;
                                
                                // Calculate projected totals
                                $projected_cpu = $adjusted_cpu_used + $new_site_cpu;
                                $projected_memory = $adjusted_memory_used + $new_site_memory;
                                
                                // Detailed debug logging
                                error_log("=== RESOURCE VALIDATION DEBUG ===");
                                error_log("Site ID: $site_id, Username: {$site['username']}");
                                error_log("User limits: {$user_limits['max_cpu_allowance']} CPU cores, {$user_limits['max_memory_allowance']} MB RAM");
                                error_log("Current total usage: $current_cpu_used CPU cores, $current_memory_used MB RAM");
                                error_log("Current site config: {$site['container_count']} containers × {$site['cpu_per_container']} CPU = $current_site_cpu cores");
                                error_log("Current site memory: {$site['container_count']} containers × {$site['memory_per_container']} MB = $current_site_memory MB");
                                error_log("Adjusted usage (without current site): $adjusted_cpu_used CPU cores, $adjusted_memory_used MB RAM");
                                error_log("New site config: $container_count containers × $cpu_per_container CPU = $new_site_cpu cores");
                                error_log("New site memory: $container_count containers × $memory_per_container MB = $new_site_memory MB");
                                error_log("Projected total: $projected_cpu CPU cores, $projected_memory MB RAM");
                                error_log("CPU limit exceeded: " . ($projected_cpu > $user_limits['max_cpu_allowance'] ? 'YES' : 'NO'));
                                error_log("Memory limit exceeded: " . ($projected_memory > $user_limits['max_memory_allowance'] ? 'YES' : 'NO'));
                                error_log("=== END DEBUG ===");
                                
                                // Enforce limits
                                if ($projected_cpu > $user_limits['max_cpu_allowance']) {
                                    http_response_code(400);
                                    echo json_encode(['status' => 'error', 'message' => 'Total CPU allocation after update (' . $projected_cpu . ' cores) exceeds your limit (' . $user_limits['max_cpu_allowance'] . ' cores). Current usage: ' . $current_cpu_used . ' cores, this site currently uses: ' . $current_site_cpu . ' cores, new site will use: ' . $new_site_cpu . ' cores. Debug: container_count=' . $container_count . ', cpu_per_container=' . $cpu_per_container]);
                                    break;
                                }
                                
                                if ($projected_memory > $user_limits['max_memory_allowance']) {
                                    http_response_code(400);
                                    echo json_encode(['status' => 'error', 'message' => 'Total memory allocation after update (' . $projected_memory . ' MB) exceeds your limit (' . $user_limits['max_memory_allowance'] . ' MB). Current usage: ' . $current_memory_used . ' MB, this site currently uses: ' . $current_site_memory . ' MB, new site will use: ' . $new_site_memory . ' MB']);
                                    break;
                                }
                            }
                        }
                        
                        // Get auto-recreation setting
                        $auto_recreate_on_update = isset($input['auto_recreate_on_update']) ? 1 : 0;
                        
                        // Accept and process FQDN for primary_domain_id
                        $primary_fqdn = trim($input['primary_domain_id']);
                        $additional_fqdns = isset($input['additional_domains']) ? $input['additional_domains'] : [];
                        if (!is_array($additional_fqdns)) $additional_fqdns = [];
                        $all_domains = $db->query("SELECT id, domain_name, username FROM whp.domains WHERE active=1")->fetchAll(PDO::FETCH_ASSOC);
                        // Find parent for primary
                        $primary_domain_row = null;
                        foreach ($all_domains as $row) {
                            if ($primary_fqdn === $row['domain_name'] || (str_ends_with($primary_fqdn, '.' . $row['domain_name']))) {
                                $primary_domain_row = $row;
                                break;
                            }
                        }
                        if (!$primary_domain_row) {
                            http_response_code(400);
                            echo json_encode([
                                'status' => 'error',
                                'message' => 'Primary domain/subdomain not found or not owned by you.',
                                'debug' => [
                                    'primary_fqdn' => $primary_fqdn,
                                    'all_domains' => $all_domains
                                ]
                            ]);
                            break;
                        }
                        if (!$is_root && $primary_domain_row['username'] !== $site['username']) {
                            http_response_code(403);
                            echo json_encode([
                                'status' => 'error',
                                'message' => 'You can only update sites with your own domains/subdomains',
                                'debug' => [
                                    'primary_fqdn' => $primary_fqdn,
                                    'parent_domain' => $primary_domain_row['domain_name'],
                                    'parent_owner' => $primary_domain_row['username'],
                                    'site_owner' => $site['username']
                                ]
                            ]);
                            break;
                        }
                        // Validate FQDN is a valid A or CNAME for this domain
                        require_once('/docker/whp/web/libs/pdns-integration.php');
                        $PowerDNS = new PowerDNSIntegration($db);
                        $valid_fqdn = false;
                        $fqdn_records = $PowerDNS->getRecords($primary_domain_row['domain_name']);
                        foreach ($fqdn_records as $rec) {
                            if (strcasecmp($rec['name'], $primary_fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                                $valid_fqdn = true;
                                break;
                            }
                        }
                        if (!$valid_fqdn) {
                            http_response_code(400);
                            echo json_encode([
                                'status' => 'error',
                                'message' => 'Selected primary FQDN is not a valid A or CNAME record.',
                                'debug' => [
                                    'primary_fqdn' => $primary_fqdn,
                                    'parent_domain' => $primary_domain_row['domain_name'],
                                    'dns_records' => $fqdn_records
                                ]
                            ]);
                            break;
                        }
                        // Update site table
                        $stmt = $db->prepare("UPDATE whp.sites SET site_name=?, primary_domain_id=?, container_type_id=?, container_count=?, cpu_per_container=?, memory_per_container=?, user_variables=?, auto_recreate_on_update=? WHERE id=?");
                        $stmt->execute([
                            $site_name,
                            $primary_domain_row['id'],
                            $container_type_id,
                            $container_count,
                            $cpu_per_container,
                            $memory_per_container,
                            $user_variables,
                            !empty($input['auto_recreate_on_update']) ? 1 : 0,
                            $site_id
                        ]);
                        // Update site domains - only change what's different
                        $current_domains = [];
                        $stmt = $db->prepare("SELECT domain_id, fqdn, is_primary FROM whp.site_domains WHERE site_id = ?");
                        $stmt->execute([$site_id]);
                        $current_domains = $stmt->fetchAll();
                        
                        // Build target domains list
                        $target_domains = [];
                        $target_domains[] = ['domain_id' => $primary_domain_row['id'], 'fqdn' => $primary_fqdn, 'is_primary' => 1];
                        foreach ($additional_fqdns as $fqdn) {
                            $fqdn = trim($fqdn);
                            if ($fqdn === $primary_fqdn) continue;
                            $parent_row = null;
                            foreach ($all_domains as $row) {
                                if ($fqdn === $row['domain_name'] || (str_ends_with($fqdn, '.' . $row['domain_name']))) {
                                    $parent_row = $row;
                                    break;
                                }
                            }
                            if (!$parent_row) continue;
                            if (!$is_root && $parent_row['username'] !== $site['username']) continue;
                            $valid_fqdn = false;
                            $fqdn_records = $PowerDNS->getRecords($parent_row['domain_name']);
                            foreach ($fqdn_records as $rec) {
                                if (strcasecmp($rec['name'], $fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                                    $valid_fqdn = true;
                                    break;
                                }
                            }
                            if (!$valid_fqdn) continue;
                            $target_domains[] = ['domain_id' => $parent_row['id'], 'fqdn' => $fqdn, 'is_primary' => 0];
                        }
                        
                        // Remove domains that are no longer needed
                        foreach ($current_domains as $current) {
                            $found = false;
                            foreach ($target_domains as $target) {
                                if ($current['domain_id'] == $target['domain_id'] && $current['fqdn'] == $target['fqdn']) {
                                    $found = true;
                                    break;
                                }
                            }
                            if (!$found) {
                                $stmt = $db->prepare("DELETE FROM whp.site_domains WHERE site_id = ? AND domain_id = ? AND fqdn = ?");
                                $stmt->execute([$site_id, $current['domain_id'], $current['fqdn']]);
                            }
                        }
                        
                        // Add new domains
                        foreach ($target_domains as $target) {
                            $found = false;
                            foreach ($current_domains as $current) {
                                if ($current['domain_id'] == $target['domain_id'] && $current['fqdn'] == $target['fqdn']) {
                                    $found = true;
                                    break;
                                }
                            }
                            if (!$found) {
                                $stmt = $db->prepare("INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) VALUES (?, ?, ?, ?)");
                                $stmt->execute([$site_id, $target['domain_id'], $target['fqdn'], $target['is_primary']]);
                            }
                        }
                        // Check if containers need to be recreated
                        $needs_container_recreation = false;
                        $old_container_count = $site['container_count'];
                        $old_container_type_id = $site['container_type_id'];
                        
                        if ($container_count != $old_container_count || $container_type_id != $old_container_type_id) {
                            $needs_container_recreation = true;
                        }
                        
                        if ($needs_container_recreation) {
                            // Remove old containers from DB and Docker
                            $stmt = $db->prepare("SELECT container_id FROM whp.site_containers WHERE site_id=?");
                            $stmt->execute([$site_id]);
                            $old_containers = $stmt->fetchAll();
                            foreach ($old_containers as $c) {
                                $DockerAPI->remove_container($c['container_id']);
                            }
                            $stmt = $db->prepare("DELETE FROM whp.site_containers WHERE site_id=?");
                            $stmt->execute([$site_id]);
                        }
                        $created_containers = [];
                        $errors = [];
                        
                        if ($needs_container_recreation) {
                            // Get updated site info for container creation
                            $stmt = $db->prepare("
                                SELECT s.*, d.domain_name as primary_domain, ct.base_image, ct.startup_env, ct.mount_options
                                FROM whp.sites s
                                JOIN whp.domains d ON s.primary_domain_id = d.id
                                JOIN whp.container_types ct ON s.container_type_id = ct.id
                                WHERE s.id = ?
                            ");
                            $stmt->execute([$site_id]);
                            $site = $stmt->fetch();
                            
                            // Create new containers (same as create_containers logic)
                            $uid = trim(shell_exec("id -u " . escapeshellarg($site['username'])));
                            for ($i = 1; $i <= $site['container_count']; $i++) {
                            $container_name = $site['primary_domain'] . '-' . str_pad($i, 2, '0', STR_PAD_LEFT);
                            $env_vars = [
                                'WHP_USER=' . $site['username'],
                                'WHP_UID=' . $uid,
                                'WHP_DOMAIN=' . $site['primary_domain'],
                                'WHP_CONTAINER_NUMBER=' . $i
                            ];
                            if (!empty($site['user_variables'])) {
                                $user_vars = json_decode($site['user_variables'], true);
                                foreach ($user_vars as $var_name => $var_value) {
                                    $env_vars[] = $var_name . '=' . $var_value;
                                }
                            }
                            if (!empty($site['startup_env'])) {
                                $startup_env = json_decode($site['startup_env'], true);
                                if ($startup_env) {
                                    foreach ($startup_env as $var_name => $var_value) {
                                        // Replace dynamic variables using centralized function
                                        $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                        $env_vars[] = $var_name . '=' . $var_value;
                                    }
                                }
                            }
                            $volumes = [
                                '/docker/users/' . $site['username'] . '/' . $site['primary_domain'] . ':/var/www/html'
                            ];
                            if (!empty($site['mount_options'])) {
                                $mount_options = json_decode($site['mount_options'], true);
                                if ($mount_options && isset($mount_options['volumes'])) {
                                    foreach ($mount_options['volumes'] as $volume) {
                                        // Replace dynamic variables in both source and target paths
                                                                                    $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                            $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], $site['username'], $site['primary_domain'], $uid, $i, $site['listen_port'] ?? 80);
                                        $volumes[] = $source . ':' . $target;
                                    }
                                }
                            }
                            $container_config = [
                                'image' => $site['base_image'],
                                'name' => $container_name,
                                'labels' => [
                                    'whp.user' => $site['username'],
                                    'whp.domain' => $site['primary_domain'],
                                    'whp.site_id' => strval($site_id),
                                    'whp.container_number' => strval($i)
                                ],
                                'env' => $env_vars,
                                'volumes' => $volumes,
                                'cpu' => $site['cpu_per_container'],
                                'memory' => $site['memory_per_container'],
                                'network' => 'client-net'
                            ];
                            $result = $DockerAPI->create_container($container_config);
                            if ($result['success']) {
                                $stmt = $db->prepare("INSERT INTO whp.site_containers (site_id, container_id, container_number, status) VALUES (?, ?, ?, 'running')");
                                $stmt->execute([$site_id, $result['container_id'], $i]);
                                $created_containers[] = [
                                    'container_number' => $i,
                                    'container_id' => $result['container_id']
                                ];
                            } else {
                                $errors[] = "Failed to create container $i: " . $result['error'];
                            }
                        }
                        
                        }
                        
                        // Configure HAProxy if containers were created successfully
                        if (!empty($created_containers)) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, $site);
                            
                            if (!$haproxy_result['success']) {
                                $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        
                        // Always configure HAProxy after site update (regardless of container recreation)
                        require_once('/docker/whp/web/libs/haproxy_manager.php');
                        $haproxy_manager = new haproxy_manager();
                        
                        // Get updated site info for HAProxy configuration
                        $stmt = $db->prepare("
                            SELECT s.*, d.domain_name as primary_domain, ct.base_image, ct.startup_env, ct.mount_options
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            JOIN whp.container_types ct ON s.container_type_id = ct.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $updated_site = $stmt->fetch();
                        
                        $haproxy_result = $haproxy_manager->configureSite($site_id, $updated_site);
                        
                        if (!$haproxy_result['success']) {
                            $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                        }
                        
                        if (empty($errors)) {
                            echo json_encode(['status' => 'success', 'message' => 'Site updated and HAProxy configured', 'containers' => $created_containers]);
                        } else {
                            echo json_encode(['status' => 'partial', 'message' => 'Site updated but some operations failed', 'containers' => $created_containers, 'errors' => $errors]);
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()]);
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(['status' => 'error', 'message' => 'Missing required fields for update_site']);
                }
                break;
                
            case 'recreate_container':
                // Recreate a specific container for a site with the latest image
                if (!empty($input['site_id']) && !empty($input['container_id'])) {
                    $site_id = intval($input['site_id']);
                    $container_id = $input['container_id'];
                    
                    global $db;
                    try {
                        // Get site details
                        $stmt = $db->prepare("
                            SELECT 
                                s.*,
                                d.domain_name as primary_domain,
                                ct.base_image,
                                ct.startup_env,
                                ct.mount_options
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            JOIN whp.container_types ct ON s.container_type_id = ct.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Get container details from database
                        $stmt = $db->prepare("SELECT * FROM whp.site_containers WHERE site_id = ? AND container_id = ?");
                        $stmt->execute([$site_id, $container_id]);
                        $container_info = $stmt->fetch();
                        
                        if (!$container_info) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Container not found'
                            ));
                            break;
                        }
                        
                        // First, pull the latest image
                        $pull_result = $DockerAPI->pull_image($site['base_image']);
                        if (!$pull_result['success']) {
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Failed to pull latest image: ' . $pull_result['error']
                            ));
                            break;
                        }
                        
                        // Stop and remove the old container
                        $stop_result = $DockerAPI->stop_container($container_id);
                        $remove_result = $DockerAPI->remove_container($container_id);
                        
                        // Get user UID
                        $uid = trim(shell_exec("id -u " . escapeshellarg($site['username'])));
                        if (!is_numeric($uid)) {
                            http_response_code(400);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Invalid user'
                            ));
                            break;
                        }
                        
                        // Create new container with same configuration
                        $container_number = $container_info['container_number'];
                        $container_name = $site['primary_domain'] . '-' . str_pad($container_number, 2, '0', STR_PAD_LEFT);
                        
                        // Prepare environment variables
                        $env_vars = [
                            'WHP_USER=' . $site['username'],
                            'WHP_UID=' . $uid,
                            'WHP_DOMAIN=' . $site['primary_domain'],
                            'WHP_CONTAINER_NUMBER=' . $container_number
                        ];
                        
                        // Add user variables
                        if (!empty($site['user_variables'])) {
                            $user_vars = json_decode($site['user_variables'], true);
                            foreach ($user_vars as $var_name => $var_value) {
                                $env_vars[] = $var_name . '=' . $var_value;
                            }
                        }
                        
                        // Process startup environment variables
                        if (!empty($site['startup_env'])) {
                            $startup_env = json_decode($site['startup_env'], true);
                            if ($startup_env) {
                                foreach ($startup_env as $var_name => $var_value) {
                                    // Replace dynamic variables using centralized function
                                    $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                    $env_vars[] = $var_name . '=' . $var_value;
                                }
                            }
                        }
                        
                        // Prepare volumes
                        $volumes = [
                            '/docker/users/' . $site['username'] . '/' . $site['primary_domain'] . ':/var/www/html'
                        ];
                        
                        // Process mount options
                        if (!empty($site['mount_options'])) {
                            $mount_options = json_decode($site['mount_options'], true);
                            if ($mount_options && isset($mount_options['volumes'])) {
                                foreach ($mount_options['volumes'] as $volume) {
                                    // Replace dynamic variables in both source and target paths
                                    $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                    $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                    $volumes[] = $source . ':' . $target;
                                }
                            }
                        }
                        
                        // Create new container config
                        $container_config = [
                            'image' => $site['base_image'],
                            'name' => $container_name,
                            'labels' => [
                                'whp.user' => $site['username'],
                                'whp.domain' => $site['primary_domain'],
                                'whp.site_id' => strval($site_id),
                                'whp.container_number' => strval($container_number)
                            ],
                            'env' => $env_vars,
                            'volumes' => $volumes,
                            'cpu' => $site['cpu_per_container'],
                            'memory' => $site['memory_per_container'],
                            'network' => 'client-net'
                        ];
                        
                        $result = $DockerAPI->create_container($container_config);
                        
                        if ($result['success']) {
                            // Update container info in database
                            $stmt = $db->prepare("UPDATE whp.site_containers SET container_id = ?, status = 'running' WHERE site_id = ? AND container_number = ?");
                            $stmt->execute([$result['container_id'], $site_id, $container_number]);
                            
                            // Configure HAProxy after container recreation
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, $site);
                            
                            if ($haproxy_result['success']) {
                                echo json_encode(array(
                                    'status' => 'success',
                                    'message' => 'Container recreated successfully with latest image and HAProxy configured',
                                    'container_id' => $result['container_id']
                                ));
                            } else {
                                echo json_encode(array(
                                    'status' => 'partial',
                                    'message' => 'Container recreated but HAProxy configuration failed: ' . $haproxy_result['error'],
                                    'container_id' => $result['container_id']
                                ));
                            }
                        } else {
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Failed to create new container: ' . $result['error']
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID and container ID are required'
                    ));
                }
                break;
                
            case 'recreate_all_containers':
                // Recreate all containers for a site with the latest images
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site details
                        $stmt = $db->prepare("
                            SELECT 
                                s.*,
                                d.domain_name as primary_domain,
                                ct.base_image,
                                ct.listen_port,
                                ct.startup_env,
                                ct.mount_options
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            JOIN whp.container_types ct ON s.container_type_id = ct.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Pull the latest image first
                        $pull_result = $DockerAPI->pull_image($site['base_image']);
                        if (!$pull_result['success']) {
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Failed to pull latest image: ' . $pull_result['error']
                            ));
                            break;
                        }
                        
                        // Get all containers for this site
                        $stmt = $db->prepare("SELECT * FROM whp.site_containers WHERE site_id = ? ORDER BY container_number ASC");
                        $stmt->execute([$site_id]);
                        $containers = $stmt->fetchAll();
                        
                        if (empty($containers)) {
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'No containers found for this site'
                            ));
                            break;
                        }
                        
                        // Get user UID
                        $uid = trim(shell_exec("id -u " . escapeshellarg($site['username'])));
                        if (!is_numeric($uid)) {
                            http_response_code(400);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Invalid user'
                            ));
                            break;
                        }
                        
                        $recreated_containers = [];
                        $errors = [];
                        
                        foreach ($containers as $container) {
                            // Stop and remove the old container
                            $stop_result = $DockerAPI->stop_container($container['container_id']);
                            $remove_result = $DockerAPI->remove_container($container['container_id']);
                            
                            // Create new container with same configuration
                            $container_number = $container['container_number'];
                            $container_name = $site['primary_domain'] . '-' . str_pad($container_number, 2, '0', STR_PAD_LEFT);
                            
                            // Prepare environment variables
                            $env_vars = [
                                'WHP_USER=' . $site['username'],
                                'WHP_UID=' . $uid,
                                'WHP_DOMAIN=' . $site['primary_domain'],
                                'WHP_CONTAINER_NUMBER=' . $container_number
                            ];
                            
                            // Add user variables
                            if (!empty($site['user_variables'])) {
                                $user_vars = json_decode($site['user_variables'], true);
                                foreach ($user_vars as $var_name => $var_value) {
                                    $env_vars[] = $var_name . '=' . $var_value;
                                }
                            }
                            
                            // Process startup environment variables
                            if (!empty($site['startup_env'])) {
                                $startup_env = json_decode($site['startup_env'], true);
                                if ($startup_env) {
                                    foreach ($startup_env as $var_name => $var_value) {
                                        // Replace dynamic variables using centralized function
                                        $var_value = $MySQLMgmt->replaceDynamicVariables($var_value, $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                        $env_vars[] = $var_name . '=' . $var_value;
                                    }
                                }
                            }
                            
                            // Prepare volumes
                            $volumes = [
                                '/docker/users/' . $site['username'] . '/' . $site['primary_domain'] . ':/var/www/html'
                            ];
                            
                            // Process mount options
                            if (!empty($site['mount_options'])) {
                                $mount_options = json_decode($site['mount_options'], true);
                                if ($mount_options && isset($mount_options['volumes'])) {
                                    foreach ($mount_options['volumes'] as $volume) {
                                        // Replace dynamic variables in both source and target paths
                                        $source = $MySQLMgmt->replaceDynamicVariables($volume['source'], $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                        $target = $MySQLMgmt->replaceDynamicVariables($volume['target'], $site['username'], $site['primary_domain'], $uid, $container_number, $site['listen_port'] ?? 80);
                                        $volumes[] = $source . ':' . $target;
                                    }
                                }
                            }
                            
                            // Create new container config
                            $container_config = [
                                'image' => $site['base_image'],
                                'name' => $container_name,
                                'labels' => [
                                    'whp.user' => $site['username'],
                                    'whp.domain' => $site['primary_domain'],
                                    'whp.site_id' => strval($site_id),
                                    'whp.container_number' => strval($container_number)
                                ],
                                'env' => $env_vars,
                                'volumes' => $volumes,
                                'cpu' => $site['cpu_per_container'],
                                'memory' => $site['memory_per_container'],
                                'network' => 'client-net'
                            ];
                            
                            $result = $DockerAPI->create_container($container_config);
                            
                            if ($result['success']) {
                                // Update container info in database
                                $stmt = $db->prepare("UPDATE whp.site_containers SET container_id = ?, status = 'running' WHERE site_id = ? AND container_number = ?");
                                $stmt->execute([$result['container_id'], $site_id, $container_number]);
                                
                                $recreated_containers[] = [
                                    'container_number' => $container_number,
                                    'container_id' => $result['container_id']
                                ];
                            } else {
                                $errors[] = "Failed to recreate container $container_number: " . $result['error'];
                            }
                        }
                        
                        // Configure HAProxy if containers were recreated successfully
                        if (!empty($recreated_containers)) {
                            require_once('/docker/whp/web/libs/haproxy_manager.php');
                            $haproxy_manager = new haproxy_manager();
                            $haproxy_result = $haproxy_manager->configureSite($site_id, $site);
                            
                            if (!$haproxy_result['success']) {
                                $errors[] = "HAProxy configuration failed: " . $haproxy_result['error'];
                            }
                        }
                        
                        if (empty($errors)) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'All containers recreated successfully with latest images and HAProxy configured',
                                'containers' => $recreated_containers
                            ));
                        } else {
                            echo json_encode(array(
                                'status' => 'partial',
                                'message' => 'Some containers recreated with errors',
                                'containers' => $recreated_containers,
                                'errors' => $errors
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'validate_dns':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site info and domains
                        $stmt = $db->prepare("
                            SELECT s.username, d.domain_name 
                            FROM whp.sites s
                            JOIN whp.domains d ON s.primary_domain_id = d.id
                            WHERE s.id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Get all domains for this site
                        $stmt = $db->prepare("
                            SELECT d.domain_name 
                            FROM whp.site_domains sd
                            JOIN whp.domains d ON sd.domain_id = d.id
                            WHERE sd.site_id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site_domains = $stmt->fetchAll(PDO::FETCH_COLUMN);
                        
                        // Add primary domain if not already in list
                        if (!in_array($site['domain_name'], $site_domains)) {
                            $site_domains[] = $site['domain_name'];
                        }
                        
                        // Use DNS validator for robust validation
                        require_once('/docker/whp/web/libs/dns_validator.php');
                        $dns_validator = new dns_validator();
                        $validation_results = $dns_validator->validateDomains($site_domains);
                        
                        echo json_encode(array(
                            'status' => 'success',
                            'all_valid' => $validation_results['all_valid'],
                            'results' => $validation_results['domains'],
                            'server_ip' => $validation_results['server_ip']
                        ));
                        
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            // SSL/HAProxy Management Endpoints
            case 'ssl_enable':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site info
                        $stmt = $db->prepare("SELECT username FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Get all domains for this site
                        $stmt = $db->prepare("
                            SELECT d.domain_name 
                            FROM whp.site_domains sd
                            JOIN whp.domains d ON sd.domain_id = d.id
                            WHERE sd.site_id = ?
                        ");
                        $stmt->execute([$site_id]);
                        $site_domains = $stmt->fetchAll(PDO::FETCH_COLUMN);
                        
                        // Add primary domain if not already in list
                        $stmt = $db->prepare("SELECT domain_name FROM whp.domains WHERE id = (SELECT primary_domain_id FROM whp.sites WHERE id = ?)");
                        $stmt->execute([$site_id]);
                        $primary_domain = $stmt->fetchColumn();
                        if (!in_array($primary_domain, $site_domains)) {
                            $site_domains[] = $primary_domain;
                        }
                        
                        // Use DNS validator for robust validation
                        require_once('/docker/whp/web/libs/dns_validator.php');
                        $dns_validator = new dns_validator();
                        $validation_results = $dns_validator->validateDomains($site_domains);
                        
                        if (!$validation_results['all_valid']) {
                            $invalid_domains = [];
                            foreach ($validation_results['domains'] as $domain => $result) {
                                if (!$result['valid']) {
                                    $invalid_domains[] = $domain;
                                }
                            }
                            
                            http_response_code(400);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'The following domains are not pointing to our server: ' . implode(', ', $invalid_domains) . '. Please configure DNS to point to ' . $validation_results['server_ip'] . ' before enabling SSL.',
                                'invalid_domains' => $invalid_domains,
                                'server_ip' => $validation_results['server_ip']
                            ));
                            break;
                        }
                        
                        // Enable SSL
                        require_once('/docker/whp/web/libs/ssl_manager.php');
                        $ssl_manager = new ssl_manager();
                        $result = $ssl_manager->enable_ssl($site_id, 'letsencrypt');
                        
                        if ($result['success']) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'SSL enabled successfully'
                            ));
                        } else {
                            http_response_code(500);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => $result['error']
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                

            case 'ssl_renew':
                if (!empty($input['site_id'])) {
                    $site_id = intval($input['site_id']);
                    
                    global $db;
                    try {
                        // Get site info
                        $stmt = $db->prepare("SELECT username FROM whp.sites WHERE id = ?");
                        $stmt->execute([$site_id]);
                        $site = $stmt->fetch();
                        
                        if (!$site) {
                            http_response_code(404);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Site not found'
                            ));
                            break;
                        }
                        
                        // Check ownership
                        $is_root = (defined('AUTH_USER') && AUTH_USER === 'root');
                        if (!$is_root && $site['username'] !== AUTH_USER) {
                            http_response_code(403);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => 'Access denied'
                            ));
                            break;
                        }
                        
                        // Renew SSL
                        require_once('/docker/whp/web/libs/ssl_manager.php');
                        $ssl_manager = new ssl_manager();
                        $result = $ssl_manager->renew_ssl($site_id);
                        
                        if ($result['success']) {
                            echo json_encode(array(
                                'status' => 'success',
                                'message' => 'SSL certificates renewed successfully'
                            ));
                        } else {
                            http_response_code(500);
                            echo json_encode(array(
                                'status' => 'error',
                                'message' => $result['error']
                            ));
                        }
                    } catch (PDOException $e) {
                        http_response_code(500);
                        echo json_encode(array(
                            'status' => 'error',
                            'message' => 'Database error: ' . $e->getMessage()
                        ));
                    }
                } else {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Site ID is required'
                    ));
                }
                break;
                
            case 'haproxy_configure':
                // TODO: Implement HAProxy configuration
                // This endpoint will configure HAProxy backends for the site
                http_response_code(501);
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'HAProxy configuration not yet implemented'
                ));
                break;
                
            default:
                http_response_code(400);
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Invalid action'
                ));
                break;
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Method not allowed'
        ));
        break;
}
?> 