#!/bin/bash
# Update script to add PostgreSQL support to existing WHP installations

echo "==============================================="
echo "WHP PostgreSQL Support Installation"
echo "==============================================="

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo "Please run as root"
    exit 1
fi

# Function to generate secure password
generate_password() {
    tr -dc 'A-Za-z0-9!?&()-+=' </dev/urandom | head -c 18
}

# Create PostgreSQL data directory
echo "Creating PostgreSQL data directory..."
mkdir -p /docker/postgresql

# Generate PostgreSQL root password
echo "Generating PostgreSQL root password..."
postgres_password=$(generate_password)

# Start PostgreSQL container
echo "Starting PostgreSQL container..."
docker pull postgres:latest
docker run -d \
    --network client-net \
    --name postgresql \
    --env POSTGRES_PASSWORD="$postgres_password" \
    -v /docker/postgresql:/var/lib/postgresql/data:Z \
    -p 5432:5432 \
    --restart unless-stopped \
    postgres:latest

# Wait for PostgreSQL to be ready
echo "Waiting for PostgreSQL to be ready..."
sleep 10

# Test PostgreSQL connection
echo "Testing PostgreSQL connection..."
docker exec postgresql psql -U postgres -c "SELECT version();" >/dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "✓ PostgreSQL is running and accessible"
else
    echo "✗ PostgreSQL connection test failed"
    echo "Check logs with: docker logs postgresql"
    exit 1
fi

# Store PostgreSQL password in config
echo "Updating configuration..."
echo "" >> /docker/whp/web/configs/config.php
echo "// PostgreSQL Configuration" >> /docker/whp/web/configs/config.php
echo "define('POSTGRES_PASS', '$postgres_password');" >> /docker/whp/web/configs/config.php

# Install PostgreSQL PHP module
echo "Installing PostgreSQL PHP module..."
dnf install -y php-pgsql

# Restart PHP-FPM to load the new module
echo "Restarting PHP-FPM to load PostgreSQL module..."
systemctl restart php-fpm

# Install pgAdmin 4 via RPM
echo "Installing pgAdmin 4..."
dnf install -y https://ftp.postgresql.org/pub/pgadmin/pgadmin4/yum/pgadmin4-redhat-repo-2-1.noarch.rpm
dnf install -y pgadmin4-web

# Install required Python module for pgAdmin setup
echo "Installing required Python modules for pgAdmin..."
pip3 install typer

# Configure pgAdmin
echo "Configuring pgAdmin 4..."
PGADMIN_SETUP_EMAIL='admin@localhost' PGADMIN_SETUP_PASSWORD="$postgres_password" /usr/pgadmin4/bin/setup-web.sh --yes
if [ $? -eq 0 ]; then
    echo "✓ pgAdmin 4 setup completed successfully"
else
    echo "⚠ pgAdmin 4 setup failed - continuing anyway"
fi

# Configure Apache for pgAdmin
cat > /etc/httpd/conf.d/pgadmin4.conf << 'EOF'
LoadModule wsgi_module modules/mod_wsgi_python3.so
WSGIDaemonProcess pgadmin processes=1 threads=25 python-home=/usr/pgadmin4/venv
WSGIScriptAlias /pgadmin4 /usr/pgadmin4/web/pgAdmin4.wsgi

<Directory /usr/pgadmin4/web/>
    WSGIProcessGroup pgadmin
    WSGIApplicationGroup %{GLOBAL}
    Require all granted
</Directory>
EOF

# Set proper permissions
chown -R root:root /var/lib/pgadmin
chown -R root:root /var/log/pgadmin

# Apply database migrations
echo "Applying database migrations..."
mysql < /root/whp/sql/migrations/staging/001_postgresql_support.sql

# Update index.php to include PostgreSQL management
echo "Updating navigation menu..."
sed -i "/'database-management' => 'Database Management',/a\    'postgresql-management' => 'PostgreSQL Management'," /docker/whp/web/index.php

# Update permission manager to allow access to PostgreSQL management
echo "Updating permissions..."
sed -i "/'database-management' => array('all'), \/\/ All authenticated users/a\        'postgresql-management' => array('all'), \/\/ All authenticated users" /docker/whp/web/libs/permission_manager.php
sed -i "/'database-management' => array('all'),$/a\        'postgresql-management' => array('all')," /docker/whp/web/libs/permission_manager.php

# Restart Apache to load pgAdmin
echo "Restarting Apache..."
systemctl restart httpd

# Display summary
echo ""
echo "==============================================="
echo "PostgreSQL Installation Complete!"
echo "==============================================="
echo ""
echo "PostgreSQL Details:"
echo "- Host: 127.0.0.1"
echo "- Port: 5432"
echo "- Username: postgres"
echo "- Password: $postgres_password"
echo ""
echo "pgAdmin 4 Access:"
echo "- URL: https://your-server/pgadmin4"
echo "- Email: admin@localhost"
echo "- Password: $postgres_password"
echo ""
echo "PostgreSQL container status:"
docker ps | grep postgresql >/dev/null 2>&1 && echo "✓ PostgreSQL container is running" || echo "✗ PostgreSQL container is not running"
echo ""
echo "To manage PostgreSQL databases, visit the PostgreSQL Management page in WHP."
echo "==============================================="