<?php
// Set content type to JSON
header('Content-Type: application/json');

require_once('/docker/whp/web/libs/password_manager.php');
require_once('/docker/whp/web/libs/usermgmt.php');
$PasswordManager = new password_manager();
$UserMgmt = new usermgmt();

// Handle different HTTP methods
switch ($_SERVER['REQUEST_METHOD']) {
    case 'POST':
        // Check if user is authenticated for password changes
        if (!defined('AUTH_USER')) {
            http_response_code(401);
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Authentication required'
            ));
            exit();
        }
        // Get JSON input
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Invalid JSON input'
            ));
            break;
        }
        
        // Validate required fields
        if (empty($input['current_password']) || empty($input['new_password']) || empty($input['confirm_password'])) {
            http_response_code(400);
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Current password, new password, and confirmation are required'
            ));
            break;
        }
        
        // Validate password change (AUTH_USER is guaranteed to be defined here)
        $validation = $PasswordManager->validate_password_change(
            $input['current_password'],
            $input['new_password'],
            $input['confirm_password'],
            AUTH_USER
        );
        
        if (!$validation['valid']) {
            http_response_code(400);
            echo json_encode(array(
                'status' => 'error',
                'message' => implode(', ', $validation['errors']),
                'strength' => $validation['strength']
            ));
            break;
        }
        
        // Change password using usermgmt class to handle both system and MySQL passwords
        $result = $UserMgmt->change_password(AUTH_USER, $input['new_password']);
        if ($result['status'] === '0') {
            echo json_encode(array(
                'status' => 'success',
                'message' => 'Password changed successfully',
                'strength' => $validation['strength']
            ));
        } else {
            http_response_code(500);
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Failed to change password: ' . $result['msg']
            ));
        }
        break;
        
    case 'GET':
        // Return password requirements and generate password
        $action = isset($_GET['action']) ? $_GET['action'] : '';
        
        switch ($action) {
            case 'requirements':
                $requirements = $PasswordManager->get_password_requirements();
                echo json_encode(array(
                    'status' => 'success',
                    'data' => $requirements
                ));
                break;
                
            case 'generate':
                // Check authentication for password generation
                if (!defined('AUTH_USER')) {
                    http_response_code(401);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Authentication required'
                    ));
                    break;
                }
                
                $length = isset($_GET['length']) ? (int)$_GET['length'] : 16;
                $length = max(8, min(32, $length)); // Limit between 8 and 32
                
                $password = $PasswordManager->generate_password($length);
                $strength = $PasswordManager->check_password_strength($password);
                
                echo json_encode(array(
                    'status' => 'success',
                    'data' => array(
                        'password' => $password,
                        'strength' => $strength
                    )
                ));
                break;
                
            case 'check-strength':
                // No authentication required for password strength checking
                $password = isset($_GET['password']) ? $_GET['password'] : '';
                
                if (empty($password)) {
                    http_response_code(400);
                    echo json_encode(array(
                        'status' => 'error',
                        'message' => 'Password parameter required'
                    ));
                    break;
                }
                
                $strength = $PasswordManager->check_password_strength($password);
                echo json_encode(array(
                    'status' => 'success',
                    'data' => $strength
                ));
                break;
                
            default:
                http_response_code(400);
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Invalid action. Use requirements, generate, or check-strength.'
                ));
                break;
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Method not allowed. Use GET or POST.'
        ));
        break;
}
?> 