#!/usr/bin/env bash

# This script must run as root
user=$(whoami)
if [ $user != "root" ]; then
  echo "Not root user, please re-run as root"
  exit 1
fi

# Check if WHP is already installed
if [ -f /etc/whp-version ] || [ -f /root/whp/VERSION ] || [ -d /docker/whp ]; then
    echo "Error: WHP appears to be already installed."
    
    # Try to get current version
    if [ -f /etc/whp-version ]; then
        echo "Current version: $(cat /etc/whp-version)"
    elif [ -f /root/whp/VERSION ]; then
        echo "Current version: $(cat /root/whp/VERSION)"
    else
        echo "Current version: unknown"
    fi
    
    echo ""
    echo "To update an existing installation, please run:"
    echo "  cd /root/whp && ./scripts/download-update.sh"
    echo ""
    echo "Or manually download the latest update from:"
    echo "  https://whp-releases.cloud-hosting.io"
    echo ""
    echo "If you're sure you want to reinstall, remove these first:"
    echo "  rm -f /etc/whp-version"
    echo "  rm -rf /docker/whp"
    echo "  rm -rf /root/whp"
    exit 1
fi

# Give people a chance to back out
echo "Getting ready to install the control panel, waiting 10 seconds"
sleep 10
# Do initial setup and install Docker
setenforce 0
sed -i -e '/SELINUX=/ s/=.*/=disabled/' /etc/selinux/config
dnf install -y qemu-guest-agent git yum-utils rsync wget curl tar zip
dnf install -y epel-release
dnf config-manager --set-enabled crb
dnf install --nogpgcheck https://mirrors.rpmfusion.org/free/el/rpmfusion-free-release-$(rpm -E %rhel).noarch.rpm -y
dnf install --nogpgcheck https://mirrors.rpmfusion.org/nonfree/el/rpmfusion-nonfree-release-$(rpm -E %rhel).noarch.rpm -y
dnf install ffmpeg ffmpeg-devel -y
curl -LsS https://r.mariadb.com/downloads/mariadb_repo_setup | sudo bash -s -- --mariadb-server-version=10.11
dnf install -y mariadb
yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo
dnf install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
systemctl enable qemu-guest-agent
# Configure Docker to use a different data root directory instead of bind mounting overlay2
mkdir -p /docker/docker-data
# Create Docker daemon configuration
mkdir -p /etc/docker
cat > /etc/docker/daemon.json << EOF
{
  "data-root": "/docker/docker-data"
}
EOF
systemctl daemon-reload
systemctl enable --now docker
dnf update -y
mkdir -p /docker/{users,mysql}
mkdir -p /docker/whp/{haproxy,ssl,tasks,sql,backups,web}
# Add WHP user
adduser -M -d "/docker/whp"  -c "Webhosting Panel User" whp
chown -R whp:whp "/docker/whp"
usermod -aG docker whp
# Install WHP Elements
dnf install https://dl.fedoraproject.org/pub/epel/epel-release-latest-9.noarch.rpm -y
dnf install -y https://rpms.remirepo.net/enterprise/remi-release-9.rpm
dnf install -y httpd mod_ssl wget procps
openssl req -newkey rsa:2048 -nodes -keyout /etc/pki/tls/private/localhost.key -x509 -days 3650 -subj "/CN=localhost" -out /etc/pki/tls/certs/localhost.crt
dnf module enable php:remi-8.2 -y
dnf install -y php php-fpm php-mysqlnd php-xml php-pecl-zip php-sodium php-soap php-pecl-xmlrpc \
php-pecl-redis5 php-pecl-memcached php-pecl-memcache php-pecl-ip2location php-pecl-imagick php-pecl-geoip \
php-mysqlnd php-mbstring php-intl php-gd libzip php-cli php-pecl-pam

# Allow the local PHP as root (I don't like it but works the best)
sed -i -e '/ExecStart=\/usr\/sbin\/php-fpm/ s/php-fpm.*/php-fpm -R --nodaemonize/' /lib/systemd/system/php-fpm.service
systemctl daemon-reload
# Clean up default Files
rm -f /etc/httpd/conf/httpd.conf
rm -f /etc/php-fpm.d/www.conf
rm -f /etc/httpd/conf.d/ssl.conf
# Add default Configs
cp ./configs/httpd.conf /etc/httpd/conf/
cp ./configs/php-fpm-whp.conf /etc/php-fpm.d/
cp ./configs/ssl.conf /etc/httpd/conf.d/
cp ./configs/php.pam /etc/pam.d/php

# Install sudoers configuration for whp user
cp ./configs/whp-sudoers /etc/sudoers.d/whp
chmod 440 /etc/sudoers.d/whp

# Create private network for namebased resolution
docker network create --driver bridge client-net
docker_host_ip=$(docker network inspect client-net |grep Subnet|awk -F '"' {'print $4'})
echo "RemoteIPHeader X-Forward-For" > /etc/httpd/conf.modules.d/remoteip.conf 
echo "RemoteIPTrustedProxy $docker_host_ip" >> /etc/httpd/conf.modules.d/remoteip.conf 

systemctl enable --now httpd
sleep 5
systemctl enable --now php-fpm

# Copy Web Files (including dot files)
rsync -auzvp ./web-files/ /docker/whp/web/

# Install ProFTPD package (configuration happens after database setup)
echo "Installing ProFTPD..."
dnf install -y proftpd proftpd-mysql

# Create ProFTPD system user (without password prompt)
useradd -r -d /var/lib/proftpd -s /bin/false --system proftpd 2>/dev/null || true

echo "✓ ProFTPD package installed (configuration will happen after database setup)"

# Enable Ports in the server
firewall-cmd --permanent --add-port 22/tcp
firewall-cmd --permanent --add-port 53/tcp
firewall-cmd --permanent --add-port 53/udp
firewall-cmd --permanent --add-port 80/tcp
firewall-cmd --permanent --add-port 443/tcp
firewall-cmd --permanent --add-port 8080/tcp
firewall-cmd --permanent --add-port 8081/tcp
firewall-cmd --permanent --add-port 8123/tcp
firewall-cmd --permanent --add-port 8443/tcp
firewall-cmd --permanent --add-port 9393/tcp
# FTP ports
firewall-cmd --permanent --add-port 21/tcp
firewall-cmd --permanent --add-port 60000-60100/tcp
firewall-cmd --reload

#Pull the Docker Images
docker pull repo.anhonesthost.net/cloud-hosting-platform/cmc:latest
docker pull repo.anhonesthost.net/cloud-hosting-platform/haproxy-manager-base:latest

# pull MariaDB image
docker pull mariadb:10.11
# Generate root MySQL Password
mysql_password=$(tr -dc 'A-Za-z0-9!?&()-+=' </dev/urandom | head -c 18)
# Start MariaDB
docker run -d --network client-net --name mysql --env MARIADB_ROOT_PASSWORD="$mysql_password" -v /docker/mysql:/var/lib/mysql:Z  -p 3306:3306 --restart unless-stopped mariadb:10.11
# Store mysql login details
echo "[client]" > /root/.my.cnf
echo "host=127.0.0.1" >> /root/.my.cnf
echo "user=root" >> /root/.my.cnf
echo "password=$mysql_password" >> /root/.my.cnf

# Wait for MySQL to be ready
echo "Waiting for MySQL to be ready..."
while ! mysql -e "SELECT 1" >/dev/null 2>&1; do
    echo "MySQL is not ready yet, waiting 5 seconds..."
    sleep 5
done
echo "✓ MySQL is ready!"

# Initialize database schemas directly with MySQL
echo "Setting up complete WHP database structure..."
mysql < ./sql/complete-database-setup.sql
if [ $? -eq 0 ]; then
    echo "✓ Database setup completed successfully"
    
    # Verify critical tables were created
    echo "Verifying database structure..."
    tables_check=$(mysql -e "USE whp; SHOW TABLES;" 2>/dev/null | wc -l)
    if [ $tables_check -gt 5 ]; then
        echo "✓ Database tables created successfully ($((tables_check-1)) tables)"
    else
        echo "⚠ Warning: Expected more tables, only found $((tables_check-1))"
    fi
else
    echo "✗ Database setup failed"
    exit 1
fi

# Set up phpMyAdmin configuration storage
echo "Setting up phpMyAdmin configuration storage..."
mysql < ./web-files/phpMyAdmin/sql/create_tables.sql
if [ $? -eq 0 ]; then
    echo "✓ phpMyAdmin configuration storage created"
else
    echo "⚠ phpMyAdmin setup failed - basic functionality will still work"
fi

# Apply any pending database migrations
echo "Applying database migrations..."
CURRENT_VERSION=$(cat VERSION)

# Apply migrations for the current version
if [ -d "./sql/migrations/$CURRENT_VERSION" ]; then
    migration_files=$(find "./sql/migrations/$CURRENT_VERSION" -name "*.sql" | sort)
    if [ -n "$migration_files" ]; then
        for migration_file in $migration_files; do
            echo "Applying migration: $(basename $migration_file)"
            mysql < "$migration_file"
            if [ $? -eq 0 ]; then
                echo "✓ Migration $(basename $migration_file) applied successfully"
            else
                echo "⚠ Migration $(basename $migration_file) failed - continuing anyway"
            fi
        done
    else
        echo "✓ No migrations found for version $CURRENT_VERSION"
    fi
else
    echo "✓ No migrations directory found for version $CURRENT_VERSION"
fi

# Create security database directory in target location
mkdir -p /docker/whp/sql
chown -R whp:whp /docker/whp/sql

# Ensure sql directory exists in source
mkdir -p /root/whp/sql
chmod 750 /root/whp/sql

# Ensure all scripts are executable
echo "Setting executable permissions on scripts..."
chmod +x scripts/*.sh scripts/*.php 2>/dev/null || true

# Set up user quotas
echo "Setting up user quotas..."
./scripts/setup-quotas.sh setup -p "$mysql_password"
if [ $? -eq 0 ]; then
    echo "✓ User quota setup completed"
else
    echo "⚠ User quota setup failed - continuing without quotas"
fi

# Configure ProFTPD with MySQL integration (now that database is ready)
echo "Configuring ProFTPD with MySQL virtual users integration..."

# Configure ProFTPD with MySQL integration
cp ./configs/proftpd.conf /etc/proftpd.conf

# Create required directories
mkdir -p /var/log/proftpd
mkdir -p /etc/proftpd/conf.d
chown -R proftpd:proftpd /var/log/proftpd

# Create welcome messages
echo "Welcome to WHP FTP Server" > /etc/proftpd/welcome.msg
echo "Login successful. Files are located in your home directory." > /etc/proftpd/login.msg

# Determine which ProFTPD configuration to use
echo "Configuring FTP authentication method..."
echo "Options:"
echo "  1) PAM Authentication (recommended - integrates with system users and quotas)"
echo "  2) MySQL Authentication (database-driven virtual users)"
echo ""

# For automated setup, default to PAM (option 1)
# In interactive mode, you could prompt: read -p "Choose authentication method (1 or 2): " auth_method
auth_method=1

if [ "$auth_method" = "1" ]; then
    echo "✓ Using PAM authentication (integrates with system users)"
    cp ./configs/proftpd-pam.conf /etc/proftpd.conf
    
    # Get server IP for passive mode configuration  
    # For testing environments, prefer internal IP
    server_ip=""
    
    # Method 1: Try to get IP from main network interface (will be internal for testing)
    server_ip=$(ip route get 8.8.8.8 | awk '{print $7}' | head -n1 2>/dev/null)
    
    # Method 2: If that fails, try hostname -I
    if [ -z "$server_ip" ] || [ "$server_ip" = "127.0.0.1" ]; then
        server_ip=$(hostname -I | awk '{print $1}' 2>/dev/null)
    fi
    
    # For testing: prefer internal IPs, don't try to get external IP
    if [ -n "$server_ip" ]; then
        echo "✓ Configured FTP passive mode with IP: $server_ip (using internal IP for testing)"
        # Update the MasqueradeAddress in the config file
        sed -i "s/MasqueradeAddress.*192\.168\.1\.105/MasqueradeAddress               $server_ip/" /etc/proftpd.conf
    else
        # Fallback to default testing IP
        server_ip="192.168.1.105"
        echo "⚠ Using default testing IP: $server_ip"
    fi
    
    echo "✓ ProFTPD configured with PAM authentication - uses system users and existing quotas"
else
    echo "✓ Using MySQL authentication (database-driven virtual users)"
    cp ./configs/proftpd.conf /etc/proftpd.conf
    
    # Get server IP for passive mode configuration
    server_ip=$(ip route get 8.8.8.8 | awk '{print $7}' | head -n1 2>/dev/null)
    if [ -z "$server_ip" ] || [ "$server_ip" = "127.0.0.1" ]; then
        server_ip=$(hostname -I | awk '{print $1}' 2>/dev/null)
    fi
    
    if [ -n "$server_ip" ]; then
        # Replace placeholders in MySQL config
        sed -i "s|\$PUBLIC_IP|$server_ip|g" /etc/proftpd.conf
        echo "✓ Configured FTP passive mode with IP: $server_ip"
        
        # Set up MySQL authentication
        actual_mysql_password=$(grep '^password=' /root/.my.cnf | cut -d'=' -f2-)
        sed -i "s|\$MYSQL_PASSWORD|$actual_mysql_password|g" /etc/proftpd.conf
        
        # Create dedicated MySQL user for ProFTPD
        mysql -e "
        CREATE USER IF NOT EXISTS 'proftpd'@'localhost' IDENTIFIED BY 'proftpd_secure_pass';
        GRANT SELECT, INSERT, UPDATE, DELETE ON whp.ftp_users TO 'proftpd'@'localhost';
        GRANT SELECT, INSERT, UPDATE, DELETE ON whp.ftp_groups TO 'proftpd'@'localhost';
        GRANT SELECT, INSERT, UPDATE, DELETE ON whp.ftp_access_log TO 'proftpd'@'localhost';
        GRANT SELECT, INSERT, UPDATE, DELETE ON whp.ftp_quotas TO 'proftpd'@'localhost';
        FLUSH PRIVILEGES;
        "
        echo "✓ Created dedicated MySQL user for ProFTPD"
    else
        echo "⚠ Could not determine server IP - FTP passive mode may need manual configuration"
    fi
fi

# Set proper permissions
chmod 644 /etc/proftpd.conf
chmod 644 /etc/proftpd/welcome.msg
chmod 644 /etc/proftpd/login.msg

# Enable and start ProFTPD service
systemctl enable proftpd
systemctl start proftpd

echo "✓ ProFTPD configured with MySQL virtual users integration"

# Set up PowerDNS with secure password management
echo "Setting up PowerDNS DNS server..."
if [ -f "./scripts/setup-powerdns-enhanced.sh" ]; then
    chmod +x ./scripts/setup-powerdns-enhanced.sh
    ./scripts/setup-powerdns-enhanced.sh
    if [ $? -eq 0 ]; then
        echo "✓ PowerDNS setup completed successfully"
        
        # Make password manager executable
        if [ -f "./scripts/powerdns-password-manager.sh" ]; then
            chmod +x ./scripts/powerdns-password-manager.sh
            echo "✓ PowerDNS password manager configured"
        fi
        
        # Verify PowerDNS installation
        echo "Verifying PowerDNS installation..."
        if [ -f "./scripts/verify-powerdns.sh" ]; then
            chmod +x ./scripts/verify-powerdns.sh
            ./scripts/verify-powerdns.sh
            if [ $? -eq 0 ]; then
                echo "✓ PowerDNS verification completed successfully"
            else
                echo "⚠ PowerDNS verification failed - check logs for details"
            fi
        fi
    else
        echo "⚠ PowerDNS setup failed - continuing without DNS server"
    fi
else
    echo "⚠ PowerDNS setup script not found - skipping DNS server setup"
fi

#start HAPROXY Container
# Check if API key exists in settings
haproxy_api_key=""
if [ -f "/docker/whp/settings.json" ]; then
    haproxy_api_key=$(grep -o '"haproxy_api_key":"[^"]*"' /docker/whp/settings.json | cut -d'"' -f4 2>/dev/null || echo "")
fi

# Build environment variable string
api_key_env=""
if [ -n "$haproxy_api_key" ]; then
    api_key_env="-e HAPROXY_API_KEY=$haproxy_api_key "
fi

docker run -d --network=client-net --add-host=host.docker.internal:host-gateway -p 80:80 -p 443:443 -p 8000:8000 -v lets-encrypt:/etc/letsencrypt -v haproxy:/etc/haproxy $api_key_env--name haproxy-manager --restart unless-stopped repo.anhonesthost.net/cloud-hosting-platform/haproxy-manager-base:latest

#start memcache Container
docker run -d --network=client-net --add-host=host.docker.internal:host-gateway -p 11211:11211  --restart unless-stopped --name memcache repo.anhonesthost.net/cloud-hosting-platform/cmc:latest

# Final verification
echo ""
echo "Performing final system verification..."
echo "======================================"

# Check service status
echo "Checking service status:"
systemctl is-active httpd >/dev/null 2>&1 && echo "✓ Apache is running" || echo "✗ Apache is not running"
systemctl is-active php-fpm >/dev/null 2>&1 && echo "✓ PHP-FPM is running" || echo "✗ PHP-FPM is not running"
systemctl is-active docker >/dev/null 2>&1 && echo "✓ Docker is running" || echo "✗ Docker is not running"
systemctl is-active proftpd >/dev/null 2>&1 && echo "✓ FTP server is running" || echo "✗ FTP server is not running"

# Check container status
echo ""
echo "Checking container status:"
docker ps | grep mysql >/dev/null 2>&1 && echo "✓ MySQL container is running" || echo "✗ MySQL container is not running"
docker ps | grep haproxy-manager >/dev/null 2>&1 && echo "✓ HAProxy container is running" || echo "✗ HAProxy container is not running"
docker ps | grep memcache >/dev/null 2>&1 && echo "✓ Memcache container is running" || echo "✗ Memcache container is not running"
docker ps | grep powerdns >/dev/null 2>&1 && echo "✓ PowerDNS container is running" || echo "✗ PowerDNS container is not running"

# Check database connectivity
echo ""
echo "Checking database connectivity:"
mysql -e "SELECT 1" >/dev/null 2>&1 && echo "✓ MySQL connection successful" || echo "✗ MySQL connection failed"

# Set up SSH chroot jail for users in /docker/users/
if ! grep -q 'Match User \*,!root' /etc/ssh/sshd_config; then
  cat << 'EOF' >> /etc/ssh/sshd_config

# WHP: Chroot all non-root users with home in /docker/users/ to their home directory and force SFTP
Match User *,!root
    ChrootDirectory /docker/users/%u
    ForceCommand internal-sftp
    AllowTcpForwarding no
    X11Forwarding no
EOF
  systemctl restart sshd
  echo "✓ SSH chroot jail for /docker/users/ configured and sshd restarted"
else
  echo "✓ SSH chroot jail already configured in sshd_config"
fi
mkdir -p /etc/certs

# Check web access
echo ""
echo "Checking web access:"
curl -s -o /dev/null -w "%{http_code}" http://localhost:8080 | grep -q "200\|302" && echo "✓ Web interface accessible" || echo "✗ Web interface not accessible"

echo ""
echo "======================================"
# Setup backup system cron jobs
echo "Setting up backup system cron jobs..."
(crontab -l 2>/dev/null; echo "# WHP Backup System - Cleanup old backups daily at 2 AM") | crontab -
(crontab -l 2>/dev/null; echo "0 2 * * * /usr/bin/php /root/whp/scripts/cleanup-backups.php >/dev/null 2>&1") | crontab -
(crontab -l 2>/dev/null; echo "# WHP Backup System - Process scheduled backups every 5 minutes") | crontab -
(crontab -l 2>/dev/null; echo "*/5 * * * * /usr/bin/php /root/whp/scripts/backup-scheduler.php >/dev/null 2>&1") | crontab -

echo "✓ Backup system cron jobs configured"

# Setup WordPress auto-update system
echo "Setting up WordPress auto-update system..."
# Create log file with proper permissions
touch /var/log/wordpress-auto-update.log
chmod 644 /var/log/wordpress-auto-update.log
chown root:root /var/log/wordpress-auto-update.log

# Add WordPress auto-update cron job (every 6 hours: 2 AM, 8 AM, 2 PM, 8 PM)
(crontab -l 2>/dev/null; echo "# WHP WordPress Auto-Update System - Run updates every 6 hours") | crontab -
(crontab -l 2>/dev/null; echo "0 2,8,14,20 * * * /usr/bin/php /root/whp/scripts/wordpress-auto-update.php >/dev/null 2>&1") | crontab -

echo "✓ WordPress auto-update system configured"

# Create backup system directories
mkdir -p /tmp/whp-backups
chown whp:whp /tmp/whp-backups
chmod 755 /tmp/whp-backups

echo "✓ Backup system directories created"

# Setup WebSocket Terminal Server
echo ""
echo "======================================"
echo "Setting up WebSocket Terminal Server..."

# Install build dependencies for node-pty
echo "Installing build dependencies..."
dnf groupinstall -y "Development Tools"
dnf install -y python3-devel

# Install Node.js via NVM if not present
if ! command -v node &> /dev/null; then
    echo "Installing Node.js 22 LTS via NVM..."
    
    # Download and install nvm
    curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.3/install.sh | bash
    
    # Source nvm for current session
    export NVM_DIR="$HOME/.nvm"
    [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"
    [ -s "$NVM_DIR/bash_completion" ] && \. "$NVM_DIR/bash_completion"
    
    # Install Node.js 22 LTS
    nvm install 22
    nvm use 22
    nvm alias default 22
    
    echo "✓ Node.js $(node -v) installed via NVM"
    echo "✓ npm $(npm -v) available"
else
    echo "✓ Node.js already installed: $(node -v)"
fi

# Download xterm.js files and sync to web root
echo "Downloading xterm.js files..."
chmod +x /root/whp/scripts/download-xterm.sh
if ! /root/whp/scripts/download-xterm.sh; then
    echo "Error: Failed to download xterm.js files"
    exit 1
fi
echo "✓ xterm.js files downloaded and synced successfully"

# Install terminal server dependencies
echo "Installing terminal server dependencies..."
cd /root/whp/terminal-server
if ! npm install --production; then
    echo "Error: Failed to install terminal server dependencies"
    echo "Check that build dependencies are installed and Node.js is working"
    exit 1
fi
echo "✓ Terminal server dependencies installed successfully"

# Copy Apache WebSocket configuration
echo "Configuring Apache WebSocket proxy..."
cp /root/whp/configs/whp-terminal-websocket.conf /etc/httpd/conf.d/

# Install and enable systemd service
echo "Installing terminal server service..."
chmod +x /root/whp/scripts/create-terminal-service.sh
if ! /root/whp/scripts/create-terminal-service.sh; then
    echo "Error: Failed to create terminal service"
    exit 1
fi

systemctl daemon-reload
if ! systemctl enable whp-terminal; then
    echo "Error: Failed to enable terminal service"
    exit 1
fi

if ! systemctl start whp-terminal; then
    echo "Error: Failed to start terminal service"
    echo "Check service logs: journalctl -u whp-terminal -f"
    exit 1
fi

echo "✓ Terminal service installed and started successfully"

echo "✓ WebSocket Terminal Server configured"
cd /root/whp

echo "Setup completed!"
echo ""
echo "Access your control panel at: http://your-server-ip:8080"
echo "Default login uses system PAM authentication"
echo ""
echo "Next steps:"
echo "1. Create system users for hosting clients"
echo "2. Database structure is fully configured and verified"
echo "3. Run 'php /root/whp/scripts/setup-mysql-users.php' to set up MySQL users for existing system users"
echo "4. User quotas are configured - use '/root/whp/scripts/quota-helper.sh' for quota management"
echo "5. ProFTPD is configured with MySQL virtual users - hosting users automatically get FTP access when created"
echo "6. PowerDNS DNS server is configured with secure password management"
echo "7. Use '/root/whp/scripts/powerdns-password-manager.sh' for DNS password management"
echo "8. Configure backup targets in the WHP control panel under 'Backup Management'"
echo "9. WordPress auto-updates are configured to run every 6 hours (2 AM, 8 AM, 2 PM, 8 PM)"
echo "10. WordPress security settings include comprehensive options: file editor blocking, XML-RPC protection, version hiding, security headers, and more"
echo "11. WebSocket terminal server provides full terminal access to containers via the web interface"
echo ""
echo "Troubleshooting:"
echo "- Check logs in /var/log/httpd/ for Apache issues"
echo "- Use 'docker logs mysql' for database issues"
echo "- Ensure .user.ini is copied to web directory if authentication fails"
echo "- Check backup logs in /var/log/whp-backup-*.log"
echo "- Check WordPress auto-update logs in /var/log/wordpress-auto-update.log"
echo "- WordPress management includes WP-CLI integration, security settings, auto-updates, and backup functionality"
echo "- Check terminal server logs: journalctl -u whp-terminal -f"
echo "- Verify WebSocket proxy: curl -I http://localhost:8080/terminal-ws"
echo "REBOOT THE SERVER NOW"
echo ""

# Create version file to mark installation complete
echo "$(cat /root/whp/VERSION)" > /etc/whp-version
echo "WHP version $(cat /etc/whp-version) installed successfully"
