<?php

class security_manager {
    
    private $db_path = '/docker/whp/sql/security.db';
    private $db;
    private $max_failed_attempts = 5;
    private $block_duration = 3600; // 1 hour in seconds
    private $whitelist_ips = array('127.0.0.1', '::1'); // Localhost
    
    public function __construct() {
        $this->init_database();
    }
    
    private function init_database() {
        if (!file_exists($this->db_path)) {
            $this->db = new SQLite3($this->db_path);
            $this->db->busyTimeout(6000);
            
            // Create failed login attempts table
            $this->db->exec('CREATE TABLE failed_logins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT NOT NULL,
                username TEXT NOT NULL,
                attempt_time INTEGER NOT NULL,
                user_agent TEXT
            )');
            
            // Create blocked IPs table
            $this->db->exec('CREATE TABLE blocked_ips (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT UNIQUE NOT NULL,
                block_time INTEGER NOT NULL,
                unblock_time INTEGER NOT NULL,
                reason TEXT,
                failed_attempts INTEGER DEFAULT 0,
                manual_block INTEGER DEFAULT 0
            )');
            
            // Create whitelist IPs table
            $this->db->exec('CREATE TABLE whitelist_ips (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                ip_address TEXT UNIQUE NOT NULL,
                description TEXT,
                added_time INTEGER NOT NULL
            )');
            
            // Create indexes for better performance
            $this->db->exec('CREATE INDEX idx_failed_logins_ip ON failed_logins(ip_address)');
            $this->db->exec('CREATE INDEX idx_failed_logins_time ON failed_logins(attempt_time)');
            $this->db->exec('CREATE INDEX idx_blocked_ips_time ON blocked_ips(unblock_time)');
        } else {
            $this->db = new SQLite3($this->db_path);
            $this->db->busyTimeout(6000);
        }
    }
    
    public function record_failed_login($ip_address, $username, $user_agent = '') {
        // Don't track whitelisted IPs
        if ($this->is_ip_whitelisted($ip_address)) {
            return true;
        }
        
        $stmt = $this->db->prepare('INSERT INTO failed_logins (ip_address, username, attempt_time, user_agent) VALUES (:ip, :user, :time, :agent)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':user', $username, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->bindValue(':agent', $user_agent, SQLITE3_TEXT);
        $stmt->execute();
        
        // Check if IP should be blocked
        $this->check_and_block_ip($ip_address);
        
        return true;
    }
    
    public function record_successful_login($ip_address) {
        // Clear failed attempts for this IP on successful login
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        return true;
    }
    
    private function check_and_block_ip($ip_address) {
        // Count recent failed attempts
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM failed_logins WHERE ip_address = :ip AND attempt_time > :time');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time() - 3600, SQLITE3_INTEGER); // Last hour
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        if ($row['count'] >= $this->max_failed_attempts) {
            $this->block_ip($ip_address, 'Too many failed login attempts', $row['count']);
        }
    }
    
    public function block_ip($ip_address, $reason = 'Manual block', $failed_attempts = 0) {
        // Don't block whitelisted IPs
        if ($this->is_ip_whitelisted($ip_address)) {
            return false;
        }
        
        $stmt = $this->db->prepare('INSERT OR REPLACE INTO blocked_ips (ip_address, block_time, unblock_time, reason, failed_attempts, manual_block) VALUES (:ip, :block_time, :unblock_time, :reason, :attempts, :manual)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':block_time', time(), SQLITE3_INTEGER);
        $stmt->bindValue(':unblock_time', time() + $this->block_duration, SQLITE3_INTEGER);
        $stmt->bindValue(':reason', $reason, SQLITE3_TEXT);
        $stmt->bindValue(':attempts', $failed_attempts, SQLITE3_INTEGER);
        $stmt->bindValue(':manual', ($reason === 'Manual block') ? 1 : 0, SQLITE3_INTEGER);
        $stmt->execute();
        
        return true;
    }
    
    public function unblock_ip($ip_address) {
        $stmt = $this->db->prepare('DELETE FROM blocked_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        // Also clear failed attempts for this IP
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        return true;
    }
    
    public function is_ip_blocked($ip_address) {
        // Check if IP is whitelisted
        if ($this->is_ip_whitelisted($ip_address)) {
            return false;
        }
        
        // Check if IP is blocked
        $stmt = $this->db->prepare('SELECT * FROM blocked_ips WHERE ip_address = :ip AND unblock_time > :time');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        if ($row) {
            // Auto-unblock if time has expired
            if ($row['unblock_time'] <= time()) {
                $this->unblock_ip($ip_address);
                return false;
            }
            return $row;
        }
        
        return false;
    }
    
    public function is_ip_whitelisted($ip_address) {
        // Check built-in whitelist
        if (in_array($ip_address, $this->whitelist_ips)) {
            return true;
        }
        
        // Check database whitelist
        $stmt = $this->db->prepare('SELECT * FROM whitelist_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $result = $stmt->execute();
        $row = $result->fetchArray(SQLITE3_ASSOC);
        
        return $row !== false;
    }
    
    public function add_whitelist_ip($ip_address, $description = '') {
        $stmt = $this->db->prepare('INSERT OR REPLACE INTO whitelist_ips (ip_address, description, added_time) VALUES (:ip, :desc, :time)');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':desc', $description, SQLITE3_TEXT);
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->execute();
        
        return true;
    }
    
    public function remove_whitelist_ip($ip_address) {
        $stmt = $this->db->prepare('DELETE FROM whitelist_ips WHERE ip_address = :ip');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->execute();
        
        return true;
    }
    
    public function get_failed_attempts($ip_address, $hours = 1) {
        $stmt = $this->db->prepare('SELECT * FROM failed_logins WHERE ip_address = :ip AND attempt_time > :time ORDER BY attempt_time DESC');
        $stmt->bindValue(':ip', $ip_address, SQLITE3_TEXT);
        $stmt->bindValue(':time', time() - ($hours * 3600), SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        $attempts = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $attempts[] = $row;
        }
        
        return $attempts;
    }
    
    public function get_blocked_ips() {
        $stmt = $this->db->prepare('SELECT * FROM blocked_ips WHERE unblock_time > :time ORDER BY block_time DESC');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        $blocked = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $blocked[] = $row;
        }
        
        return $blocked;
    }
    
    public function get_whitelist_ips() {
        $stmt = $this->db->prepare('SELECT * FROM whitelist_ips ORDER BY added_time DESC');
        $result = $stmt->execute();
        
        $whitelist = array();
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $whitelist[] = $row;
        }
        
        return $whitelist;
    }
    
    public function cleanup_expired_blocks() {
        $stmt = $this->db->prepare('DELETE FROM blocked_ips WHERE unblock_time <= :time');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $stmt->execute();
        
        // Clean up old failed attempts (older than 24 hours)
        $stmt = $this->db->prepare('DELETE FROM failed_logins WHERE attempt_time <= :time');
        $stmt->bindValue(':time', time() - 86400, SQLITE3_INTEGER);
        $stmt->execute();
        
        // Clean up old permission logs (older than 90 days)
        if (file_exists('/docker/whp/sql/permissions.db')) {
            require_once('/docker/whp/web/libs/permission_manager.php');
            $PermManager = new permission_manager();
            $PermManager->cleanup_old_logs(90);
        }
        
        return true;
    }
    
    public function get_security_stats() {
        // Get current blocked IPs count
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM blocked_ips WHERE unblock_time > :time');
        $stmt->bindValue(':time', time(), SQLITE3_INTEGER);
        $result = $stmt->execute();
        $blocked_count = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        // Get failed attempts in last 24 hours
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM failed_logins WHERE attempt_time > :time');
        $stmt->bindValue(':time', time() - 86400, SQLITE3_INTEGER);
        $result = $stmt->execute();
        $failed_24h = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        // Get whitelist count
        $stmt = $this->db->prepare('SELECT COUNT(*) as count FROM whitelist_ips');
        $result = $stmt->execute();
        $whitelist_count = $result->fetchArray(SQLITE3_ASSOC)['count'];
        
        return array(
            'blocked_ips' => $blocked_count,
            'failed_attempts_24h' => $failed_24h,
            'whitelist_ips' => $whitelist_count
        );
    }
    
    public function __destruct() {
        if ($this->db) {
            $this->db->close();
        }
    }
} 