# WHP Migration System

This document explains how the WHP migration system works and how to add new database migrations.

## How It Works

**Migration Organization:**
- Migrations are organized by release version in folders like `2025.07.1/`, `2025.07.2/`, etc.
- Each version folder contains numbered SQL files: `001_feature_name.sql`, `002_another_feature.sql`
- The `update.sh` script automatically applies migrations in version order

**Version Assignment:**
- Release versions are automatically managed by `build-release.sh`
- Developers should NOT create version folders manually
- Use the staging system instead (see below)

## Adding New Migrations

### Step 1: Create Migration in Staging

```bash
# Create your migration file in the staging directory
vim sql/migrations/staging/001_your_feature_name.sql
```

**Migration File Requirements:**
- Must start with `USE whp;` statement
- Use `CREATE TABLE IF NOT EXISTS` for new tables
- Include proper indexes and constraints
- Add comments explaining the changes

**Example Migration:**
```sql
USE whp;

-- Create table for new feature
CREATE TABLE IF NOT EXISTS new_feature (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_name (name)
);

-- Insert initial data if needed
INSERT INTO system_logs (log_level, message, created_at) 
VALUES ('info', 'New feature tables created successfully', NOW())
ON DUPLICATE KEY UPDATE message = VALUES(message);
```

### Step 2: Test Migration Locally

```bash
# Test the migration manually
mysql whp < sql/migrations/staging/001_your_feature_name.sql
```

### Step 3: Commit to Git

```bash
git add sql/migrations/staging/001_your_feature_name.sql
git commit -m "Add migration for new feature"
git push
```

## Build Process

When `build-release.sh` runs, it automatically:

1. **Detects staging migrations** in `sql/migrations/staging/`
2. **Assigns version number** (e.g., moves to `sql/migrations/2025.07.3/`)
3. **Updates complete database setup** (see Complete Database Setup section)
4. **Commits organized migrations** back to git
5. **Includes in release package** for distribution
6. **Pushes changes** to remote repository

**Example Build Output:**
```bash
$ ./scripts/build-release.sh
Building WHP Release 2025.07.3
Found staging migrations, organizing for version 2025.07.3
Staging migrations moved to sql/migrations/2025.07.3/
Committing organized migrations to git...
Pushing organized migrations to remote...
```

## Production Deployment

When production servers update via `update.sh`:

1. **Compares versions** to determine which migrations to apply
2. **Runs migrations** in version order automatically
3. **Handles errors** and provides rollback information
4. **Logs results** for audit trail

**Migration Application:**
- Only applies migrations for versions between current and target
- Skips already-applied migrations
- Stops on first error to prevent partial updates

## Migration Guidelines

### DO:
- ✅ Use `CREATE TABLE IF NOT EXISTS`
- ✅ Include proper indexes
- ✅ Add explanatory comments
- ✅ Test migrations before committing
- ✅ Use staging directory for new migrations
- ✅ Follow naming convention: `001_feature_description.sql`

### DON'T:
- ❌ Create version folders manually
- ❌ Modify existing migrations after release
- ❌ Use `DROP TABLE` without proper safeguards
- ❌ Forget the `USE whp;` statement
- ❌ Include user-specific data

### File Naming Convention:
```
sql/migrations/staging/001_feature_name.sql
sql/migrations/staging/002_another_feature.sql
sql/migrations/staging/003_bug_fixes.sql
```

## Troubleshooting

**Migration Failed During Update:**
```bash
# Check migration logs
tail -f /var/log/mysql/error.log

# Apply specific migration manually
mysql whp < /root/whp/sql/migrations/2025.07.3/001_failed_migration.sql
```

**Staging Directory Issues:**
```bash
# Check staging contents
ls -la sql/migrations/staging/

# Manually organize staging (if build-release.sh fails)
mkdir -p sql/migrations/2025.07.X/
mv sql/migrations/staging/* sql/migrations/2025.07.X/
```

**Testing Migrations:**
```bash
# Test on local database
mysql test_whp < sql/migrations/staging/001_test_migration.sql

# Use apply-migrations.sh for version testing
./scripts/apply-migrations.sh 2025.07.1 2025.07.2 /root/whp/sql/migrations
```

## Migration History

The build system maintains migration history in `MIGRATION_HISTORY.json`:

```json
{
  "current_version": "2025.07.3",
  "oldest_supported": "2025.07.1",
  "migration_count": 3,
  "build_date": "2025-07-23T15:30:00Z"
}
```

This ensures that production servers only attempt supported upgrades and skip unsupported version jumps.

## Complete Database Setup

**Purpose:**
The `sql/complete-database-setup.sql` file contains the complete database schema for fresh WHP installations. This file must be kept in sync with all migrations to ensure new installations have the same database structure as upgraded installations.

**What It Contains:**
- All core WHP tables and structures
- All schema changes from historical migrations
- Default data and initial records
- Stored procedures and events
- Proper indexes and constraints

**Maintenance Process:**

### When Adding New Migrations:
1. **Create migration** in `sql/migrations/staging/` as normal
2. **Run build-release.sh** - this will organize the migration and update complete setup
3. **Manually verify** that `sql/complete-database-setup.sql` includes your changes
4. **Test fresh installation** to ensure complete setup works

### Manual Updates (if needed):
```bash
# Add your table definitions before the SETUP COMPLETE section
vim sql/complete-database-setup.sql

# Find this section:
# -- ================================================
# -- SETUP COMPLETE  
# -- ================================================

# Insert your tables before it, following this format:
# -- ================================================
# -- YOUR FEATURE NAME TABLES
# -- ================================================
# 
# CREATE TABLE IF NOT EXISTS your_table (
#     id INT AUTO_INCREMENT PRIMARY KEY,
#     ...
# ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

### Important Guidelines:
- ✅ **Always use `CREATE TABLE IF NOT EXISTS`** for compatibility
- ✅ **Include proper ENGINE and CHARSET** specifications
- ✅ **Add appropriate indexes** for performance
- ✅ **Test with fresh database** before committing
- ❌ **Never include version-specific migration logic** 
- ❌ **Don't include `USE whp;`** - it's already at the top
- ❌ **Avoid ALTER TABLE statements** - use CREATE IF NOT EXISTS instead

### Testing Complete Database Setup:
```bash
# Test on clean database
mysql -e "DROP DATABASE IF EXISTS whp_test; CREATE DATABASE whp_test;"
mysql whp_test < sql/complete-database-setup.sql

# Verify tables were created
mysql whp_test -e "SHOW TABLES;"

# Check for any errors
mysql whp_test -e "SELECT * FROM system_logs WHERE log_level = 'error';"
```

### Build Integration:
The `build-release.sh` script automatically commits changes to both migration files and `complete-database-setup.sql`, ensuring they stay synchronized across releases.