#!/bin/bash

# Manual update script to add WebSocket Terminal feature to existing WHP installations
# Run this script after pulling the latest code and running the standard update process

echo "======================================"
echo "WHP Terminal Feature Installation"
echo "======================================"
echo ""

# Check if we're running as root
if [[ $EUID -ne 0 ]]; then
   echo "This script must be run as root"
   exit 1
fi

# Check if we're in the correct directory
if [ ! -f "/root/whp/setup.sh" ]; then
    echo "Error: Please run this script from /root/whp directory"
    echo "Usage: cd /root/whp && ./scripts/update-add-terminal.sh"
    exit 1
fi

echo "Installing terminal feature for existing WHP installation..."
echo ""

# 1. Apply database migrations
echo "1. Applying database migrations..."
if [ -f "/root/whp/sql/migrations/staging/001_terminal_access.sql" ]; then
    mysql < /root/whp/sql/migrations/staging/001_terminal_access.sql
    echo "✓ Terminal database tables created"
else
    echo "⚠ Warning: Terminal migration file not found"
fi

# 2. Install Node.js via NVM if not present
echo ""
echo "2. Installing Node.js 22 LTS via NVM..."
if ! command -v node &> /dev/null; then
    # Download and install nvm
    curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.3/install.sh | bash
    
    # Source nvm for current session
    export NVM_DIR="$HOME/.nvm"
    [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"
    [ -s "$NVM_DIR/bash_completion" ] && \. "$NVM_DIR/bash_completion"
    
    # Install Node.js 22 LTS
    nvm install 22
    nvm use 22
    nvm alias default 22
    
    echo "✓ Node.js $(node -v) installed via NVM"
    echo "✓ npm $(npm -v) available"
else
    echo "✓ Node.js already installed: $(node -v)"
fi

# 3. Download xterm.js files
echo ""
echo "3. Downloading xterm.js files..."
chmod +x /root/whp/scripts/download-xterm.sh
/root/whp/scripts/download-xterm.sh

# 4. Sync web files including xterm.js to document root
echo ""
echo "4. Syncing files to web root..."
rsync -av /root/whp/web-files/ /docker/whp/web/
echo "✓ Web files synced"

# 5. Install terminal server dependencies
echo ""
echo "5. Installing terminal server dependencies..."
cd /root/whp/terminal-server
npm install --production
cd /root/whp

# 6. Copy Apache WebSocket configuration
echo ""
echo "6. Configuring Apache WebSocket proxy..."
cp /root/whp/configs/whp-terminal-websocket.conf /etc/httpd/conf.d/
echo "✓ Apache configuration added"

# 7. Create and install systemd service
echo ""
echo "7. Installing terminal server service..."
chmod +x /root/whp/scripts/create-terminal-service.sh
/root/whp/scripts/create-terminal-service.sh
systemctl daemon-reload
systemctl enable whp-terminal
systemctl start whp-terminal

# 8. Restart Apache to load new configuration
echo ""
echo "8. Restarting Apache..."
systemctl restart httpd

# 9. Check service status
echo ""
echo "9. Checking service status..."
if systemctl is-active --quiet whp-terminal; then
    echo "✓ Terminal server is running"
else
    echo "⚠ Warning: Terminal server failed to start"
    echo "Check logs with: journalctl -u whp-terminal -f"
fi

echo ""
echo "======================================"
echo "Terminal Feature Installation Complete!"
echo "======================================"
echo ""
echo "The terminal feature is now available at:"
echo "- Navigate to 'Terminal' in the main navigation menu"
echo "- Select a container and connect to access the terminal"
echo ""
echo "Troubleshooting:"
echo "- Check terminal server logs: journalctl -u whp-terminal -f"
echo "- Verify WebSocket proxy: curl -I http://localhost/terminal-ws"
echo "- Check xterm.js files: ls -la /docker/whp/web/js/xterm/"
echo ""
echo "If you encounter issues, restart the services:"
echo "- systemctl restart whp-terminal"
echo "- systemctl restart httpd"