<?php
/**
 * SSO API for WHMCS Integration
 * Provides single sign-on functionality via API keys
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Configure appropriately for production
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-API-Key, X-API-Secret, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once(__DIR__ . '/../../libs/api_auth.php');
require_once(__DIR__ . '/../../libs/ApiKeyManager.php');
require_once(__DIR__ . '/../../libs/network.php');

try {
    // Store raw input early since php://input can only be read once
    $rawRequestBody = file_get_contents('php://input');
    
    // Debug incoming request
    error_log("SSO API Request - Method: " . $_SERVER['REQUEST_METHOD']);
    error_log("SSO API Request - Headers: " . json_encode(getallheaders()));
    error_log("SSO API Request - GET params: " . json_encode($_GET));
    error_log("SSO API Request - Raw body: " . $rawRequestBody);
    
    // Authenticate and check permissions
    $authData = api_authenticate('sso_access');
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    switch ($method) {
        case 'POST':
            handlePostRequest($action, $authData, $rawRequestBody);
            break;
            
        default:
            api_response(false, null, 'Method not allowed', 405);
    }
    
} catch (Exception $e) {
    error_log("SSO API Error: " . $e->getMessage() . " Stack: " . $e->getTraceAsString());
    api_response(false, null, 'Internal server error: ' . $e->getMessage(), 500);
}

function handlePostRequest($action, $authData, $rawRequestBody) {
    switch ($action) {
        case 'generate_token':
            generateSSOToken($authData, $rawRequestBody);
            break;
            
        case 'validate_user':
            validateUserForSSO($rawRequestBody);
            break;
            
        default:
            api_response(false, null, 'Invalid action. Use: generate_token, validate_user', 400);
    }
}

function generateSSOToken($authData, $rawInput) {
    $input = json_decode($rawInput, true);
    
    // Enhanced logging for debugging
    error_log("SSO Token Request - Raw input: " . $rawInput);
    error_log("SSO Token Request - Parsed input: " . json_encode($input));
    
    if (!$input) {
        api_response(false, null, 'Invalid JSON input', 400);
    }
    
    $required = ['username'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            api_response(false, null, "Field '$field' is required", 400);
        }
    }
    
    $username = $input['username'];
    $redirectUrl = $input['redirect_url'] ?? null;
    $expiryMinutes = intval($input['expiry_minutes'] ?? 10);
    
    error_log("SSO Token Request - Username: $username, Redirect: $redirectUrl, Expiry: $expiryMinutes");
    
    // Validate expiry time (1-60 minutes)
    if ($expiryMinutes < 1 || $expiryMinutes > 60) {
        api_response(false, null, 'Expiry time must be between 1 and 60 minutes', 400);
    }
    
    // Verify user exists
    require_once(__DIR__ . '/../../libs/usermgmt.php');
    $userMgmt = new usermgmt();
    
    error_log("SSO Token Request - Checking if user '$username' exists...");
    
    if (!$userMgmt->user_exists($username)) {
        error_log("SSO Token Request - User '$username' does not exist");
        api_response(false, null, 'User does not exist', 404);
    }
    
    error_log("SSO Token Request - User '$username' exists, generating token...");
    
    // Generate SSO token
    $apiKeyManager = new ApiKeyManager();
    $netInfo = new network_info();
    $ipAddress = $netInfo->get_client_ip();
    
    $result = $apiKeyManager->generateSSOToken(
        $username,
        $authData['key_id'],
        $ipAddress,
        $redirectUrl,
        $expiryMinutes
    );
    
    error_log("SSO Token Request - Token generation result: " . json_encode($result));
    
    if ($result['success']) {
        // Add full login URL with the site's base URL
        $baseUrl = getBaseUrl();
        $result['login_url'] = $baseUrl . '/api/sso-login.php?token=' . $result['token'];
        
        error_log("SSO Token Request - SUCCESS - Login URL: " . $result['login_url']);
        api_response(true, $result, 'SSO token generated successfully');
    } else {
        error_log("SSO Token Request - FAILED - Error: " . $result['error']);
        api_response(false, null, $result['error'], 500);
    }
}

function validateUserForSSO($rawInput) {
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        api_response(false, null, 'Invalid JSON input', 400);
    }
    
    $required = ['username'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            api_response(false, null, "Field '$field' is required", 400);
        }
    }
    
    $username = $input['username'];
    
    // Check if user exists and get info
    require_once(__DIR__ . '/../../libs/usermgmt.php');
    $userMgmt = new usermgmt();
    
    if (!$userMgmt->user_exists($username)) {
        api_response(false, null, 'User does not exist', 404);
    }
    
    // Get user information
    $users = $userMgmt->list_users();
    $userInfo = null;
    
    foreach ($users as $user) {
        if ($user['username'] === $username) {
            $userInfo = $user;
            break;
        }
    }
    
    if ($userInfo) {
        // Remove sensitive information
        unset($userInfo['password']);
        unset($userInfo['shadow']);
        
        api_response(true, [
            'username' => $username,
            'exists' => true,
            'can_login' => true,
            'user_info' => $userInfo
        ], 'User validated for SSO');
    } else {
        api_response(false, null, 'User validation failed', 500);
    }
}

function getBaseUrl() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME'];
    $path = dirname($_SERVER['SCRIPT_NAME']);
    
    // Remove /api/external from path to get base path
    $path = str_replace('/api/external', '', $path);
    
    return $protocol . '://' . $host . $path;
}
?>