<?php
/**
 * API Debug Script
 * Helps troubleshoot API key authentication issues
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Load WHP configuration
require_once 'web-files/configs/config.php';
require_once 'web-files/libs/mysqlmgmt.php';

// Connect to database
$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();
$db->exec("USE whp");

echo "=== WHP API Debug Script ===\n\n";

// 1. Check if API tables exist
echo "1. Checking API tables...\n";
$tables = ['api_keys', 'api_usage_logs', 'sso_tokens'];
foreach ($tables as $table) {
    try {
        $stmt = $db->query("SELECT COUNT(*) FROM $table");
        $count = $stmt->fetchColumn();
        echo "   ✓ Table '$table' exists with $count records\n";
    } catch (Exception $e) {
        echo "   ✗ Table '$table' missing or error: " . $e->getMessage() . "\n";
    }
}

// 2. List all API keys
echo "\n2. API Keys in database:\n";
try {
    $stmt = $db->query("SELECT id, key_name, api_key, is_active, created_by, created_at FROM api_keys ORDER BY created_at DESC");
    $keys = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($keys)) {
        echo "   No API keys found in database!\n";
        echo "   → You need to create an API key via Settings → Server Settings → API Key Management\n";
    } else {
        foreach ($keys as $key) {
            $status = $key['is_active'] ? 'ACTIVE' : 'INACTIVE';
            echo "   - {$key['key_name']} | {$key['api_key']} | $status | by {$key['created_by']}\n";
        }
    }
} catch (Exception $e) {
    echo "   Error querying API keys: " . $e->getMessage() . "\n";
}

// 3. Test the specific API key from the curl command
echo "\n3. Testing specific API key: whp_f529dec44c4201b07566d3e01d5592bc12a1d4acdf7a3c4c\n";
$testApiKey = 'whp_f529dec44c4201b07566d3e01d5592bc12a1d4acdf7a3c4c';

try {
    $stmt = $db->prepare("SELECT id, key_name, is_active, api_secret, ip_whitelist, rate_limit_per_hour FROM api_keys WHERE api_key = ?");
    $stmt->execute([$testApiKey]);
    $keyData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$keyData) {
        echo "   ✗ API key NOT FOUND in database\n";
        echo "   → This explains the 'Authentication required' error\n";
        echo "   → Create a new API key or use an existing one\n";
    } else {
        echo "   ✓ API key found: {$keyData['key_name']}\n";
        echo "   - Status: " . ($keyData['is_active'] ? 'ACTIVE' : 'INACTIVE') . "\n";
        echo "   - IP Whitelist: " . ($keyData['ip_whitelist'] ?: 'None') . "\n";
        echo "   - Rate Limit: {$keyData['rate_limit_per_hour']}/hour\n";
        
        if (!$keyData['is_active']) {
            echo "   ✗ API key is INACTIVE - this will cause authentication to fail\n";
        }
    }
} catch (Exception $e) {
    echo "   Error checking API key: " . $e->getMessage() . "\n";
}

// 4. Check API authentication middleware
echo "\n4. Testing API authentication middleware...\n";
try {
    require_once 'web-files/libs/ApiKeyManager.php';
    $apiManager = new ApiKeyManager($db);
    
    $result = $apiManager->validateApiKey(
        $testApiKey, 
        '48cf1f3fbcb0c3ec8893331f5b1a9f04fcc642e5fc2dcd3dc9ce0b2973bc515f',
        $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
    );
    
    if ($result['success']) {
        echo "   ✓ API key validation successful\n";
    } else {
        echo "   ✗ API key validation failed: {$result['error']}\n";
    }
} catch (Exception $e) {
    echo "   Error testing API manager: " . $e->getMessage() . "\n";
}

echo "\n=== Debug Complete ===\n";
?>