-- Enhanced Container Types Schema
-- This script adds enhanced container type management with dynamic environment variables

USE whp;

-- Add new columns to existing container_types table
ALTER TABLE whp.container_types 
ADD COLUMN IF NOT EXISTS min_cpu DECIMAL(3,2) DEFAULT 0.25 COMMENT 'Minimum CPU cores required',
ADD COLUMN IF NOT EXISTS min_memory INT DEFAULT 256 COMMENT 'Minimum memory in MB required',
ADD COLUMN IF NOT EXISTS startup_env TEXT COMMENT 'JSON object of startup environment variables',
ADD COLUMN IF NOT EXISTS user_variables TEXT COMMENT 'JSON object of user-configurable variables',
ADD COLUMN IF NOT EXISTS mount_options TEXT COMMENT 'JSON object of mount options and volumes',
ADD COLUMN IF NOT EXISTS last_image_pull TIMESTAMP NULL DEFAULT NULL COMMENT 'Last time the base image was pulled',
ADD COLUMN IF NOT EXISTS image_digest VARCHAR(255) NULL DEFAULT NULL COMMENT 'Current image digest/hash';

-- Remove default resource columns as they're not needed
ALTER TABLE whp.container_types 
DROP COLUMN IF EXISTS default_cpu,
DROP COLUMN IF EXISTS default_memory;

-- Create sample enhanced container types with mount options
INSERT IGNORE INTO whp.container_types (name, description, base_image, min_cpu, min_memory, startup_env, user_variables, mount_options, active) VALUES
('Enhanced Apache PHP', 'Advanced Apache with PHP, dynamic environment variables, and custom mount options', 'httpd:alpine', 0.25, 256, 
'{"APACHE_DOCUMENT_ROOT": "/var/www/html/${WHP_DOMAIN}", "PHP_VERSION": "${PHP_VERSION}", "WHP_USER": "${WHP_USER}", "WHP_UID": "${WHP_UID}", "WHP_GID": "${WHP_GID}", "WHP_HOME": "${WHP_HOME}"}',
'{"PHP_VERSION": {"type": "select", "label": "PHP Version", "required": true, "default": "8.2", "options": ["7.4", "8.0", "8.1", "8.2", "8.3"]}, "MEMORY_LIMIT": {"type": "number", "label": "PHP Memory Limit (MB)", "required": false, "default": 128, "min": 64, "max": 512}}',
'{"volumes": [{"source": "${WHP_HOME}/public_html", "target": "/var/www/html", "type": "bind"}, {"source": "${WHP_HOME}/logs", "target": "/var/log/apache2", "type": "bind"}], "tmpfs": ["/tmp"], "options": ["--cap-drop=ALL", "--cap-add=CHOWN", "--cap-add=SETUID", "--cap-add=SETGID"]}',
1),

('Node.js Application', 'Node.js runtime with user-defined environment and flexible mount options', 'node:alpine', 0.5, 512,
'{"NODE_ENV": "${NODE_ENV}", "PORT": "${APP_PORT}", "WHP_USER": "${WHP_USER}", "WHP_DOMAIN": "${WHP_DOMAIN}", "HOME": "${WHP_HOME}"}',
'{"NODE_ENV": {"type": "select", "label": "Node Environment", "required": true, "default": "production", "options": ["development", "production", "staging"]}, "APP_PORT": {"type": "number", "label": "Application Port", "required": true, "default": 3000, "min": 3000, "max": 9999}}',
'{"volumes": [{"source": "${WHP_HOME}/app", "target": "/usr/src/app", "type": "bind"}, {"source": "${WHP_HOME}/node_modules", "target": "/usr/src/app/node_modules", "type": "volume"}], "options": ["--user=${WHP_UID}:${WHP_GID}", "--workdir=/usr/src/app"]}',
1);

-- Create environment variable templates table (for future reference/documentation)
CREATE TABLE IF NOT EXISTS whp.container_env_templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    variable_name VARCHAR(100) NOT NULL,
    description TEXT,
    example_value VARCHAR(255),
    category ENUM('system', 'user', 'domain', 'path') DEFAULT 'system',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_variable (variable_name)
);

-- Insert standard WHP dynamic variables
INSERT IGNORE INTO whp.container_env_templates (variable_name, description, example_value, category) VALUES
('${WHP_USER}', 'System username of the container owner', 'john', 'user'),
('${WHP_UID}', 'User ID (UID) of the container owner', '1001', 'user'),
('${WHP_GID}', 'Group ID (GID) of the container owner', '1001', 'user'),
('${WHP_HOME}', 'Home directory path of the user', '/docker/users/john', 'path'),
('${WHP_DOMAIN}', 'Primary domain name assigned to the container', 'example.com', 'domain'),
('${WHP_SUBDOMAIN}', 'Subdomain (if applicable)', 'www', 'domain'),
('${WHP_DOMAIN_LIST}', 'Comma-separated list of all domains for this user', 'example.com,test.com', 'domain'),
('${WHP_CONTAINER_NAME}', 'Unique container name', 'john-example-com', 'system'),
('${WHP_PUBLIC_HTML}', 'Public HTML directory path', '/docker/users/john/public_html', 'path'),
('${WHP_LOGS_DIR}', 'Logs directory path', '/docker/users/john/logs', 'path'),
('${WHP_DATA_DIR}', 'User data directory path', '/docker/users/john/data', 'path');

-- Create user variables table for storing container-specific user configurations
CREATE TABLE IF NOT EXISTS whp.container_user_variables (
    id INT AUTO_INCREMENT PRIMARY KEY,
    container_type_id INT NOT NULL,
    domain_id INT NOT NULL,
    variable_name VARCHAR(100) NOT NULL,
    variable_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (container_type_id) REFERENCES whp.container_types(id) ON DELETE CASCADE,
    FOREIGN KEY (domain_id) REFERENCES whp.domains(id) ON DELETE CASCADE,
    UNIQUE KEY unique_container_variable (container_type_id, domain_id, variable_name)
); 