<?php
/**
 * External User Management API for WHMCS Integration
 * Provides user management functionality via API keys
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Configure appropriately for production
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-API-Key, X-API-Secret, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once(__DIR__ . '/../../libs/api_auth.php');
require_once(__DIR__ . '/../../libs/usermgmt.php');

try {
    // Authenticate and check permissions
    $authData = api_authenticate('user_management');
    
    $userMgmt = new usermgmt();
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    switch ($method) {
        case 'GET':
            handleGetRequest($userMgmt, $action, $authData);
            break;
            
        case 'POST':
            handlePostRequest($userMgmt, $action, $authData);
            break;
            
        case 'PUT':
            handlePutRequest($userMgmt, $action, $authData);
            break;
            
        case 'DELETE':
            handleDeleteRequest($userMgmt, $action, $authData);
            break;
            
        default:
            api_response(false, null, 'Method not allowed', 405);
    }
    
} catch (Exception $e) {
    error_log("External Users API Error: " . $e->getMessage());
    api_response(false, null, 'Internal server error', 500);
}

function handleGetRequest($userMgmt, $action, $authData) {
    switch ($action) {
        case 'list':
            // List all users
            $users = $userMgmt->list_users();
            api_response(true, $users, 'Users retrieved successfully');
            break;
            
        case 'info':
            // Get specific user info
            $username = $_GET['username'] ?? '';
            if (empty($username)) {
                api_response(false, null, 'Username parameter required', 400);
            }
            
            $userInfo = getUserInfo($userMgmt, $username);
            if ($userInfo) {
                api_response(true, $userInfo, 'User info retrieved successfully');
            } else {
                api_response(false, null, 'User not found', 404);
            }
            break;
            
        case 'exists':
            // Check if user exists
            $username = $_GET['username'] ?? '';
            if (empty($username)) {
                api_response(false, null, 'Username parameter required', 400);
            }
            
            $exists = $userMgmt->user_exists($username);
            api_response(true, ['exists' => $exists], 'User existence checked');
            break;
            
        default:
            api_response(false, null, 'Invalid action. Use: list, info, exists', 400);
    }
}

function handlePostRequest($userMgmt, $action, $authData) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        api_response(false, null, 'Invalid JSON input', 400);
    }
    
    switch ($action) {
        case 'create':
            createUser($userMgmt, $input, $authData);
            break;
            
        case 'validate':
            // Validate user credentials without logging in
            validateUser($userMgmt, $input);
            break;
            
        default:
            api_response(false, null, 'Invalid action. Use: create, validate', 400);
    }
}

function handlePutRequest($userMgmt, $action, $authData) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        api_response(false, null, 'Invalid JSON input', 400);
    }
    
    switch ($action) {
        case 'password':
            // Change user password
            changePassword($userMgmt, $input, $authData);
            break;
            
        case 'resources':
            // Update user resource limits
            updateResources($userMgmt, $input, $authData);
            break;
            
        default:
            api_response(false, null, 'Invalid action. Use: password, resources', 400);
    }
}

function handleDeleteRequest($userMgmt, $action, $authData) {
    $input = json_decode(file_get_contents('php://input'), true);
    $username = $input['username'] ?? $_GET['username'] ?? '';
    
    if (empty($username)) {
        api_response(false, null, 'Username parameter required', 400);
    }
    
    switch ($action) {
        case 'user':
            deleteUser($userMgmt, $username, $authData);
            break;
            
        default:
            api_response(false, null, 'Invalid action. Use: user', 400);
    }
}

function createUser($userMgmt, $input, $authData) {
    $required = ['username', 'password'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            api_response(false, null, "Field '$field' is required", 400);
        }
    }
    
    // Optional resource limits
    $resources = $input['resources'] ?? [];
    
    $result = $userMgmt->create_new_user($input['username'], $input['password']);
    
    if ($result['status'] === '0') {
        $userData = [
            'username' => $input['username'],
            'uid' => $result['uid'],
            'created_at' => date('c')
        ];
        
        // Set resource limits if provided
        if (!empty($resources)) {
            $resourceResult = setUserResources($input['username'], $resources);
            if ($resourceResult) {
                $userData['resources'] = $resources;
            }
        }
        
        api_response(true, $userData, 'User created successfully');
    } else {
        api_response(false, null, $result['msg'], 400);
    }
}

function deleteUser($userMgmt, $username, $authData) {
    $result = $userMgmt->delete_user($username);
    
    if ($result['status'] === '0') {
        api_response(true, ['username' => $username, 'deleted_at' => date('c')], 'User deleted successfully');
    } else {
        api_response(false, null, $result['msg'], 400);
    }
}

function changePassword($userMgmt, $input, $authData) {
    $required = ['username', 'password'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            api_response(false, null, "Field '$field' is required", 400);
        }
    }
    
    $result = $userMgmt->change_password($input['username'], $input['password']);
    
    if ($result['status'] === '0') {
        api_response(true, [
            'username' => $input['username'],
            'updated_at' => date('c')
        ], 'Password changed successfully');
    } else {
        api_response(false, null, $result['msg'], 400);
    }
}

function validateUser($userMgmt, $input) {
    $required = ['username', 'password'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            api_response(false, null, "Field '$field' is required", 400);
        }
    }
    
    // Use PAM to validate credentials
    if (function_exists('pam_auth')) {
        $valid = pam_auth($input['username'], $input['password']);
        api_response(true, ['valid' => $valid], 'Credentials validated');
    } else {
        api_response(false, null, 'PAM authentication not available', 500);
    }
}

function getUserInfo($userMgmt, $username) {
    // Get basic user info
    $users = $userMgmt->list_users();
    $userInfo = null;
    
    foreach ($users as $user) {
        if ($user['username'] === $username) {
            $userInfo = $user;
            break;
        }
    }
    
    if (!$userInfo) {
        return null;
    }
    
    // Add resource information
    $resources = getUserResources($username);
    if ($resources) {
        $userInfo['resources'] = $resources;
    }
    
    // Add usage statistics
    $usage = getUserUsage($username);
    if ($usage) {
        $userInfo['usage'] = $usage;
    }
    
    return $userInfo;
}

function getUserResources($username) {
    try {
        require_once(__DIR__ . '/../../libs/mysqlmgmt.php');
        $MySQLMgmt = new mysqlmgmt();
        $db = $MySQLMgmt->getMySQLConnection();
        
        $stmt = $db->prepare("
            SELECT max_cpu_allowance, max_memory_allowance, max_disk_space, max_domains, 
                   max_sites, max_email_accounts, max_email_storage_mb, max_databases, 
                   api_enabled, api_rate_limit
            FROM client_allowances 
            WHERE username = ?
        ");
        
        $stmt->execute([$username]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            // Map database column names back to API field names
            return [
                'cpu_cores' => $result['max_cpu_allowance'],
                'memory_mb' => $result['max_memory_allowance'],
                'disk_mb' => $result['max_disk_space'],
                'domains' => $result['max_domains'],
                'sites' => $result['max_sites'],
                'email_accounts' => $result['max_email_accounts'],
                'email_storage_mb' => $result['max_email_storage_mb'],
                'databases' => $result['max_databases'],
                'api_enabled' => $result['api_enabled'],
                'api_rate_limit' => $result['api_rate_limit']
            ];
        }
        
        return null;
        
    } catch (Exception $e) {
        error_log("Error getting user resources: " . $e->getMessage());
        return null;
    }
}

function setUserResources($username, $resources) {
    try {
        require_once(__DIR__ . '/../../libs/mysqlmgmt.php');
        $MySQLMgmt = new mysqlmgmt();
        $db = $MySQLMgmt->getMySQLConnection();
        
        // Check if user already has resource limits
        $stmt = $db->prepare("SELECT id FROM client_allowances WHERE username = ?");
        $stmt->execute([$username]);
        $exists = $stmt->fetch();
        
        if ($exists) {
            // Update existing limits
            $fields = [];
            $values = [];
            
            // Map API field names to database column names
            $fieldMapping = [
                'cpu_cores' => 'max_cpu_allowance',
                'memory_mb' => 'max_memory_allowance',
                'disk_mb' => 'max_disk_space',
                'domains' => 'max_domains',
                'sites' => 'max_sites',
                'email_accounts' => 'max_email_accounts',
                'email_storage_mb' => 'max_email_storage_mb',
                'databases' => 'max_databases',
                'api_enabled' => 'api_enabled',
                'api_rate_limit' => 'api_rate_limit'
            ];
            
            foreach ($fieldMapping as $apiField => $dbColumn) {
                if (isset($resources[$apiField])) {
                    $fields[] = "$dbColumn = ?";
                    $values[] = $resources[$apiField];
                }
            }
            
            if (!empty($fields)) {
                $values[] = $username;
                $sql = "UPDATE client_allowances SET " . implode(', ', $fields) . " WHERE username = ?";
                $stmt = $db->prepare($sql);
                return $stmt->execute($values);
            }
        } else {
            // Insert new limits
            $stmt = $db->prepare("
                INSERT INTO client_allowances (username, max_cpu_allowance, max_memory_allowance, max_disk_space, 
                                             max_domains, max_sites, max_email_accounts, max_email_storage_mb,
                                             max_databases, api_enabled, api_rate_limit)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            return $stmt->execute([
                $username,
                $resources['cpu_cores'] ?? 0.5,
                $resources['memory_mb'] ?? 512,
                $resources['disk_mb'] ?? 1000,
                $resources['domains'] ?? 1,
                $resources['sites'] ?? 1,
                $resources['email_accounts'] ?? 0,
                $resources['email_storage_mb'] ?? 1000,
                $resources['databases'] ?? 1,
                $resources['api_enabled'] ?? false,
                $resources['api_rate_limit'] ?? 100
            ]);
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Error setting user resources: " . $e->getMessage());
        return false;
    }
}

function updateResources($userMgmt, $input, $authData) {
    $username = $input['username'] ?? '';
    $resources = $input['resources'] ?? [];
    
    if (empty($username)) {
        api_response(false, null, 'Username is required', 400);
    }
    
    if (empty($resources)) {
        api_response(false, null, 'Resources data is required', 400);
    }
    
    $result = setUserResources($username, $resources);
    
    if ($result) {
        api_response(true, [
            'username' => $username,
            'resources' => $resources,
            'updated_at' => date('c')
        ], 'User resources updated successfully');
    } else {
        api_response(false, null, 'Failed to update user resources', 500);
    }
}

function getUserUsage($username) {
    try {
        // Get disk usage
        $homePath = "/docker/users/$username";
        $diskUsage = 0;
        
        if (is_dir($homePath)) {
            $output = shell_exec("du -s $homePath 2>/dev/null | cut -f1");
            $diskUsage = intval($output) * 1024; // Convert from KB to bytes
        }
        
        // Get additional usage stats (can be extended)
        return [
            'disk_used_bytes' => $diskUsage,
            'disk_used_mb' => round($diskUsage / (1024 * 1024), 2),
            'last_updated' => date('c')
        ];
        
    } catch (Exception $e) {
        error_log("Error getting user usage: " . $e->getMessage());
        return null;
    }
}
?>