<?php
/**
 * WHMCS Server Module for WHP (Web Hosting Panel)
 * 
 * This module allows WHMCS to manage accounts on WHP servers
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

/**
 * Define module related meta data.
 */
function whp_MetaData() {
    return array(
        'DisplayName' => 'WHP (Web Hosting Panel)',
        'APIVersion' => '1.1',
        'RequiresServer' => true,
        'DefaultNonSSLPort' => '80',
        'DefaultSSLPort' => '443',
        'ServiceSingleSignOnLabel' => 'Login to Control Panel',
        'AdminSingleSignOnLabel' => 'Login to Control Panel as User',
    );
}

/**
 * Define product configuration options.
 */
function whp_ConfigOptions() {
    return array(
        'CPU Cores' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '1',
            'Description' => 'Number of CPU cores allocated',
        ),
        'Memory (GB)' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '1',
            'Description' => 'Memory allocation in GB',
        ),
        'Disk Space (GB)' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '5',
            'Description' => 'Disk space allocation in GB',
        ),
        'Domains' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '1',
            'Description' => 'Number of domains allowed',
        ),
        'Sites' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '1',
            'Description' => 'Number of sites allowed',
        ),
        'Email Accounts' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '5',
            'Description' => 'Number of email accounts allowed',
        ),
        'Databases' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '1',
            'Description' => 'Number of databases allowed',
        ),
        'API Access' => array(
            'Type' => 'yesno',
            'Description' => 'Allow API access for this account',
        ),
        'API Rate Limit' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '100',
            'Description' => 'API requests per hour limit',
        ),
    );
}

/**
 * Create account on the remote server.
 */
function whp_CreateAccount(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        $password = $params['password'];
        $configoptions = $params['configoptions'];
        
        // Prepare API request
        $apiData = array(
            'username' => $username,
            'password' => $password,
            'resources' => array(
                'cpu_cores' => floatval($configoptions['CPU Cores'] ?: 1),
                'memory_gb' => floatval($configoptions['Memory (GB)'] ?: 1),
                'disk_gb' => floatval($configoptions['Disk Space (GB)'] ?: 5),
                'domains' => intval($configoptions['Domains'] ?: 1),
                'sites' => intval($configoptions['Sites'] ?: 1),
                'email_accounts' => intval($configoptions['Email Accounts'] ?: 5),
                'databases' => intval($configoptions['Databases'] ?: 1),
                'api_enabled' => $configoptions['API Access'] === 'on',
                'api_rate_limit' => intval($configoptions['API Rate Limit'] ?: 100),
            ),
        );
        
        // Make API call to create user
        $response = whp_api_call($serverdata, 'POST', 'users.php?action=create', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Account creation failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Suspend account on the remote server.
 */
function whp_SuspendAccount(array $params) {
    try {
        // WHP doesn't have built-in suspension, but we can disable API access
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        
        $apiData = array(
            'username' => $username,
            'resources' => array(
                'api_enabled' => false,
            ),
        );
        
        $response = whp_api_call($serverdata, 'PUT', 'users.php?action=resources', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Account suspension failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Unsuspend account on the remote server.
 */
function whp_UnsuspendAccount(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        $configoptions = $params['configoptions'];
        
        $apiData = array(
            'username' => $username,
            'resources' => array(
                'api_enabled' => $configoptions['API Access'] === 'on',
            ),
        );
        
        $response = whp_api_call($serverdata, 'PUT', 'users.php?action=resources', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Account unsuspension failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Terminate account on the remote server.
 */
function whp_TerminateAccount(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        
        $apiData = array(
            'username' => $username,
        );
        
        $response = whp_api_call($serverdata, 'DELETE', 'users.php?action=user', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Account termination failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Change the password for an account.
 */
function whp_ChangePassword(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        $password = $params['password'];
        
        $apiData = array(
            'username' => $username,
            'password' => $password,
        );
        
        $response = whp_api_call($serverdata, 'PUT', 'users.php?action=password', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Password change failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Change the package for an account.
 */
function whp_ChangePackage(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        $configoptions = $params['configoptions'];
        
        $apiData = array(
            'username' => $username,
            'resources' => array(
                'cpu_cores' => floatval($configoptions['CPU Cores'] ?: 1),
                'memory_gb' => floatval($configoptions['Memory (GB)'] ?: 1),
                'disk_gb' => floatval($configoptions['Disk Space (GB)'] ?: 5),
                'domains' => intval($configoptions['Domains'] ?: 1),
                'sites' => intval($configoptions['Sites'] ?: 1),
                'email_accounts' => intval($configoptions['Email Accounts'] ?: 5),
                'databases' => intval($configoptions['Databases'] ?: 1),
                'api_enabled' => $configoptions['API Access'] === 'on',
                'api_rate_limit' => intval($configoptions['API Rate Limit'] ?: 100),
            ),
        );
        
        $response = whp_api_call($serverdata, 'PUT', 'users.php?action=resources', $apiData);
        
        if ($response['status'] === 'success') {
            return 'success';
        } else {
            return 'Package change failed: ' . $response['message'];
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return 'Error: ' . $e->getMessage();
    }
}

/**
 * Test connection to the server.
 */
function whp_TestConnection(array $params) {
    try {
        $serverdata = $params['serverdata'];
        
        $response = whp_api_call($serverdata, 'GET', 'system.php?action=health');
        
        if ($response['status'] === 'success') {
            $health = $response['data'];
            if ($health['status'] === 'healthy') {
                return array(
                    'success' => true,
                    'message' => 'Connection successful - Server is healthy',
                );
            } else {
                return array(
                    'success' => true,
                    'message' => 'Connection successful - Server status: ' . $health['status'],
                );
            }
        } else {
            return array(
                'success' => false,
                'message' => 'Connection failed: ' . $response['message'],
            );
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return array(
            'success' => false,
            'message' => 'Connection test failed: ' . $e->getMessage(),
        );
    }
}

/**
 * Client area single sign-on.
 */
function whp_ServiceSingleSignOn(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        
        // Generate SSO token
        $apiData = array(
            'username' => $username,
            'redirect_url' => '/index.php', // Redirect to dashboard
            'expiry_minutes' => 10,
        );
        
        $response = whp_api_call($serverdata, 'POST', 'sso.php?action=generate_token', $apiData);
        
        if ($response['status'] === 'success') {
            return array(
                'success' => true,
                'redirectTo' => $response['data']['login_url'],
            );
        } else {
            return array(
                'success' => false,
                'errorMsg' => 'SSO failed: ' . $response['message'],
            );
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return array(
            'success' => false,
            'errorMsg' => 'SSO error: ' . $e->getMessage(),
        );
    }
}

/**
 * Admin area single sign-on.
 */
function whp_AdminSingleSignOn(array $params) {
    // For admin SSO, we'll use the same mechanism but potentially with different permissions
    return whp_ServiceSingleSignOn($params);
}

/**
 * Client area actions.
 */
function whp_ClientArea(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        
        // Get user information and usage statistics
        $userResponse = whp_api_call($serverdata, 'GET', 'users.php?action=info&username=' . urlencode($username));
        $statsResponse = whp_api_call($serverdata, 'GET', 'system.php?action=stats');
        
        $userInfo = $userResponse['status'] === 'success' ? $userResponse['data'] : null;
        $serverStats = $statsResponse['status'] === 'success' ? $statsResponse['data'] : null;
        
        return array(
            'templatefile' => 'overview',
            'vars' => array(
                'user_info' => $userInfo,
                'server_stats' => $serverStats,
                'username' => $username,
            ),
        );
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return array(
            'templatefile' => 'error',
            'vars' => array(
                'error' => 'Unable to load account information: ' . $e->getMessage(),
            ),
        );
    }
}

/**
 * Admin area actions.
 */
function whp_AdminServicesTabFields(array $params) {
    try {
        $serverdata = $params['serverdata'];
        $username = $params['username'];
        
        // Get detailed user information
        $response = whp_api_call($serverdata, 'GET', 'users.php?action=info&username=' . urlencode($username));
        
        if ($response['status'] === 'success') {
            $userInfo = $response['data'];
            
            return array(
                'User Information' => array(
                    'Username' => $userInfo['username'],
                    'UID' => $userInfo['uid'] ?? 'N/A',
                    'Home Directory' => $userInfo['home_directory'] ?? 'N/A',
                    'Shell' => $userInfo['shell'] ?? 'N/A',
                ),
                'Resource Allocation' => array(
                    'CPU Cores' => $userInfo['resources']['cpu_cores'] ?? 'N/A',
                    'Memory (GB)' => $userInfo['resources']['memory_gb'] ?? 'N/A',
                    'Disk Space (GB)' => $userInfo['resources']['disk_gb'] ?? 'N/A',
                    'Domains' => $userInfo['resources']['domains'] ?? 'N/A',
                    'Sites' => $userInfo['resources']['sites'] ?? 'N/A',
                    'Email Accounts' => $userInfo['resources']['email_accounts'] ?? 'N/A',
                    'Databases' => $userInfo['resources']['databases'] ?? 'N/A',
                ),
                'Current Usage' => array(
                    'Disk Used (MB)' => $userInfo['usage']['disk_used_mb'] ?? 'N/A',
                    'Last Updated' => $userInfo['usage']['last_updated'] ?? 'N/A',
                ),
                'API Settings' => array(
                    'API Enabled' => $userInfo['resources']['api_enabled'] ? 'Yes' : 'No',
                    'API Rate Limit' => $userInfo['resources']['api_rate_limit'] ?? 'N/A',
                ),
            );
        } else {
            return array(
                'Error' => array(
                    'Message' => 'Unable to retrieve user information: ' . $response['message'],
                ),
            );
        }
        
    } catch (Exception $e) {
        logModuleCall('whp', __FUNCTION__, $params, $e->getMessage());
        return array(
            'Error' => array(
                'Message' => 'Error retrieving user information: ' . $e->getMessage(),
            ),
        );
    }
}

/**
 * Execute an API call to the WHP server.
 */
function whp_api_call($serverdata, $method, $endpoint, $data = null) {
    $hostname = $serverdata['hostname'];
    $port = $serverdata['secure'] ? $serverdata['port'] ?: 443 : $serverdata['port'] ?: 80;
    $protocol = $serverdata['secure'] ? 'https' : 'http';
    
    // Server authentication details
    $apiKey = $serverdata['username']; // Store API key in username field
    $apiSecret = $serverdata['password']; // Store API secret in password field
    
    if (empty($apiKey) || empty($apiSecret)) {
        throw new Exception('API credentials not configured');
    }
    
    $url = $protocol . '://' . $hostname . ':' . $port . '/api/external/' . $endpoint;
    
    $ch = curl_init();
    
    // Basic cURL options
    curl_setopt_array($ch, array(
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false, // Configure appropriately for production
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT => 'WHMCS-WHP-Module/1.0',
    ));
    
    // Set authentication headers
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: application/json',
        'X-API-Key: ' . $apiKey,
        'X-API-Secret: ' . $apiSecret,
    ));
    
    // Set method-specific options
    switch (strtoupper($method)) {
        case 'POST':
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
            break;
            
        case 'PUT':
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
            break;
            
        case 'DELETE':
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
            break;
            
        case 'GET':
        default:
            // GET is default, no additional options needed
            break;
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    
    curl_close($ch);
    
    if ($curlError) {
        throw new Exception('cURL Error: ' . $curlError);
    }
    
    if ($httpCode >= 400) {
        throw new Exception('HTTP Error: ' . $httpCode . ' - ' . $response);
    }
    
    $decodedResponse = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON response: ' . json_last_error_msg());
    }
    
    // Log the API call for debugging
    logModuleCall('whp', $method . ' ' . $endpoint, $data, $decodedResponse);
    
    return $decodedResponse;
}
?>