-- API System Tables for WHMCS Integration
-- This migration adds support for external API access with key-based authentication

USE whp;

-- API Keys table for system-to-system authentication
CREATE TABLE IF NOT EXISTS api_keys (
    id INT AUTO_INCREMENT PRIMARY KEY,
    key_name VARCHAR(255) NOT NULL,
    api_key VARCHAR(64) NOT NULL UNIQUE,
    api_secret VARCHAR(128) NOT NULL,
    created_by VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_used_at TIMESTAMP NULL,
    is_active BOOLEAN DEFAULT TRUE,
    permissions JSON DEFAULT NULL,
    rate_limit_per_hour INT DEFAULT 1000,
    usage_count INT DEFAULT 0,
    ip_whitelist TEXT DEFAULT NULL,
    notes TEXT DEFAULT NULL,
    INDEX idx_api_key (api_key),
    INDEX idx_created_by (created_by),
    INDEX idx_active (is_active),
    INDEX idx_last_used (last_used_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- API usage logs for monitoring and rate limiting
CREATE TABLE IF NOT EXISTS api_usage_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    api_key_id INT NOT NULL,
    endpoint VARCHAR(255) NOT NULL,
    method VARCHAR(10) NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    user_agent TEXT DEFAULT NULL,
    request_data JSON DEFAULT NULL,
    response_code INT NOT NULL,
    response_time_ms INT DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (api_key_id) REFERENCES api_keys(id) ON DELETE CASCADE,
    INDEX idx_api_key_id (api_key_id),
    INDEX idx_endpoint (endpoint),
    INDEX idx_created_at (created_at),
    INDEX idx_ip_address (ip_address)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- SSO tokens for auto-login functionality
CREATE TABLE IF NOT EXISTS sso_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    token VARCHAR(64) NOT NULL UNIQUE,
    username VARCHAR(50) NOT NULL,
    api_key_id INT NOT NULL,
    redirect_url VARCHAR(500) DEFAULT NULL,
    expires_at TIMESTAMP NOT NULL,
    used_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    ip_address VARCHAR(45) NOT NULL,
    FOREIGN KEY (api_key_id) REFERENCES api_keys(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_username (username),
    INDEX idx_expires_at (expires_at),
    INDEX idx_api_key_id (api_key_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add API permissions to existing client_allowances table
ALTER TABLE client_allowances 
ADD COLUMN IF NOT EXISTS api_enabled BOOLEAN DEFAULT FALSE AFTER email_accounts,
ADD COLUMN IF NOT EXISTS api_rate_limit INT DEFAULT 100 AFTER api_enabled;

-- Create initial admin API key (to be updated with real values)
-- Note: This is commented out as API keys should be created via the UI
-- INSERT INTO api_keys (key_name, api_key, api_secret, created_by, permissions, notes) 
-- VALUES (
--     'WHMCS Integration Key',
--     'whp_admin_key_replace_this_value',
--     'replace_this_secret_value_with_secure_hash',
--     'system',
--     JSON_OBJECT(
--         'user_management', true,
--         'resource_management', true,
--         'sso_access', true,
--         'system_stats', true
--     ),
--     'Default API key for WHMCS integration - update with secure values'
-- ) ON DUPLICATE KEY UPDATE key_name = key_name;