<?php

/**
 * cPanel Backup Importer for WHP
 * 
 * Handles the import of cPanel backup tarballs into WHP system
 * Extracts domains, files, databases, and users from cPanel backups
 */
class CpanelBackupImporter {
    
    private $pdo;
    private $backupPath;
    private $extractPath;
    private $username;
    private $importLog = [];
    private $siteGroups = [];
    private $mysqlMgmt;
    private $powerDNS;
    
    public function __construct($pdo, $username) {
        $this->pdo = $pdo;
        $this->username = $username;
        $this->extractPath = "/tmp/cpanel_import_" . uniqid();
    }
    
    /**
     * Import a cPanel backup tarball
     * 
     * @param string $backupFilePath Path to the cPanel backup .tar.gz file
     * @return array Import results with success/failure status
     */
    public function importBackup($backupFilePath) {
        try {
            $this->log("Starting cPanel backup import for user: " . $this->username);
            
            // Move backup file from userfiles to temp directory to avoid storage quota
            $this->backupPath = $this->moveBackupToTemp($backupFilePath);
            
            // Validate backup file before processing
            $this->validateBackupFile();
            
            // Extract the backup tarball
            $this->extractBackup();
            
            // Validate extracted backup structure
            $this->validateBackupStructure();
            
            // Parse backup structure
            $backupData = $this->parseBackupStructure();
            
            // Validate parsed data
            $this->validateBackupData($backupData);
            
            // Import domains and subdomains
            $domains = $this->importDomains($backupData);
            
            // Create WHP sites for domains
            $sites = $this->createSites($domains);
            
            // Import website files
            $this->importFiles($domains);
            
            // Import databases
            $this->importDatabases($backupData);
            
            // Import MySQL users
            $this->importMysqlUsers($backupData);
            
            // Create DNS records for imported domains
            $this->createDnsRecords($domains);
            
            // Cleanup temporary files
            $this->cleanup();
            
            return [
                'success' => true,
                'domains_imported' => count($domains),
                'sites_created' => count($sites),
                'databases_imported' => count($backupData['mysql_databases'] ?? []),
                'log' => $this->importLog
            ];
            
        } catch (Exception $e) {
            $this->log("Import failed: " . $e->getMessage(), 'error');
            $this->cleanup();
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'log' => $this->importLog
            ];
        }
    }
    
    /**
     * Move backup file from userfiles to temp directory
     * This prevents the backup file from counting against user storage quota during processing
     */
    private function moveBackupToTemp($originalPath) {
        // Create temp directory
        $tempDir = "/tmp/cpanel_processing_" . uniqid();
        if (!mkdir($tempDir, 0755, true)) {
            throw new Exception("Failed to create temporary processing directory");
        }
        
        $tempPath = $tempDir . "/" . basename($originalPath);
        
        // Move file to temp directory
        if (!rename($originalPath, $tempPath)) {
            throw new Exception("Failed to move backup file to temporary directory");
        }
        
        $this->log("Moved backup file to temporary directory: " . $tempPath);
        return $tempPath;
    }
    
    /**
     * Validate backup file before processing
     */
    private function validateBackupFile() {
        if (!file_exists($this->backupPath)) {
            throw new Exception("Backup file not found: " . $this->backupPath);
        }
        
        if (!is_readable($this->backupPath)) {
            throw new Exception("Backup file is not readable");
        }
        
        // Check file size (max 2GB)
        $fileSize = filesize($this->backupPath);
        if ($fileSize === false) {
            throw new Exception("Cannot determine backup file size");
        }
        
        if ($fileSize > 2 * 1024 * 1024 * 1024) { // 2GB limit
            throw new Exception("Backup file too large (max 2GB allowed)");
        }
        
        if ($fileSize < 1024) { // At least 1KB
            throw new Exception("Backup file appears to be empty or corrupted");
        }
        
        // Validate file type by checking magic bytes
        $handle = fopen($this->backupPath, 'rb');
        if ($handle === false) {
            throw new Exception("Cannot open backup file for validation");
        }
        
        $header = fread($handle, 10);
        fclose($handle);
        
        // Check for gzip magic bytes (1f 8b) or tar signature
        if (!($header[0] === "\x1f" && $header[1] === "\x8b")) {
            throw new Exception("Invalid backup file format (not a valid .tar.gz file)");
        }
        
        $this->log("Backup file validation passed - Size: " . number_format($fileSize / 1024 / 1024, 2) . " MB");
    }
    
    /**
     * Validate extracted backup structure
     */
    private function validateBackupStructure() {
        if (!is_dir($this->extractPath)) {
            throw new Exception("Extraction directory not found");
        }
        
        $dirs = glob($this->extractPath . "/*", GLOB_ONLYDIR);
        if (empty($dirs)) {
            throw new Exception("No directories found in extracted backup - invalid backup structure");
        }
        
        $mainDir = $dirs[0];
        
        // Check for essential cPanel backup directories/files
        $requiredPaths = [
            'homedir',     // Website files
            'userdata'     // Domain configurations
        ];
        
        $foundPaths = [];
        foreach ($requiredPaths as $path) {
            $fullPath = $mainDir . '/' . $path;
            if (file_exists($fullPath)) {
                $foundPaths[] = $path;
            }
        }
        
        if (empty($foundPaths)) {
            throw new Exception("Invalid cPanel backup - missing essential directories (homedir, userdata)");
        }
        
        $this->log("Backup structure validation passed - Found: " . implode(', ', $foundPaths));
    }
    
    /**
     * Validate parsed backup data
     */
    private function validateBackupData($backupData) {
        if (empty($backupData)) {
            throw new Exception("No data extracted from backup");
        }
        
        // Check if we have at least some domains or files to import
        $domainCount = count($backupData['domains'] ?? []);
        $addonCount = count($backupData['addons'] ?? []);
        $subdomainCount = count($backupData['subdomains'] ?? []);
        
        if ($domainCount === 0 && $addonCount === 0 && $subdomainCount === 0) {
            throw new Exception("No domains found in backup - nothing to import");
        }
        
        // Validate domain names
        foreach ($backupData['domains'] ?? [] as $domain => $config) {
            if (!$this->isValidDomain($domain)) {
                $this->log("Invalid domain name found: " . $domain, 'warning');
            }
        }
        
        // Check if homedir exists and has content
        if (isset($backupData['homedir']) && is_dir($backupData['homedir'])) {
            $homedirSize = $this->getDirectorySize($backupData['homedir']);
            if ($homedirSize === 0) {
                $this->log("Home directory appears to be empty", 'warning');
            } else {
                $this->log("Home directory size: " . number_format($homedirSize / 1024 / 1024, 2) . " MB");
            }
        }
        
        $totalDomains = $domainCount + $addonCount + $subdomainCount;
        $this->log("Backup data validation passed - Total domains: " . $totalDomains);
    }
    
    /**
     * Extract the cPanel backup tarball
     */
    private function extractBackup() {
        if (!file_exists($this->backupPath)) {
            throw new Exception("Backup file not found: " . $this->backupPath);
        }
        
        // Create extraction directory
        if (!mkdir($this->extractPath, 0755, true)) {
            throw new Exception("Failed to create extraction directory");
        }
        
        $this->log("Extracting backup to: " . $this->extractPath);
        
        // Extract using tar command
        $command = "cd " . escapeshellarg($this->extractPath) . " && tar -xzf " . escapeshellarg($this->backupPath);
        $output = [];
        $returnCode = 0;
        
        exec($command, $output, $returnCode);
        
        if ($returnCode !== 0) {
            throw new Exception("Failed to extract backup: " . implode("\n", $output));
        }
        
        $this->log("Backup extracted successfully");
    }
    
    /**
     * Parse the extracted backup structure
     */
    private function parseBackupStructure() {
        $backupData = [];
        
        // Find the main backup directory (should be username directory)
        $dirs = glob($this->extractPath . "/*", GLOB_ONLYDIR);
        if (empty($dirs)) {
            throw new Exception("No directories found in backup");
        }
        
        $mainDir = $dirs[0]; // Usually the cPanel username directory
        $this->log("Found main backup directory: " . basename($mainDir));
        
        // Parse userdata for domain information
        $backupData['domains'] = $this->parseUserdata($mainDir . '/userdata');
        
        // Parse addon domains
        $backupData['addons'] = $this->parseAddons($mainDir . '/addons');
        
        // Parse subdomains
        $backupData['subdomains'] = $this->parseSubdomains($mainDir . '/sds');
        
        // Parse MySQL information
        $backupData['mysql_grants'] = $this->parseMysqlGrants($mainDir . '/mysql.sql');
        $backupData['mysql_databases'] = $this->getMysqlDatabases($mainDir . '/mysql');
        
        // Set file paths
        $backupData['homedir'] = $mainDir . '/homedir';
        $backupData['ssl_certs'] = $mainDir . '/sslcerts';
        $backupData['ssl_keys'] = $mainDir . '/sslkeys';
        
        return $backupData;
    }
    
    /**
     * Parse userdata directory for domain configurations
     * Maps domains to their document roots and creates site groups
     */
    private function parseUserdata($userdataDir) {
        $domains = [];
        $siteGroups = []; // Group domains by document root
        
        if (!is_dir($userdataDir)) {
            $this->log("No userdata directory found", 'warning');
            return $domains;
        }
        
        foreach (glob($userdataDir . "/*") as $file) {
            if (is_file($file)) {
                $domain = basename($file);
                $config = $this->parseUserdataFile($file);
                if ($config) {
                    $domains[$domain] = $config;
                    
                    // Group domains by document root for site creation
                    $docRoot = $config['document_root'] ?? '/public_html';
                    if (!isset($siteGroups[$docRoot])) {
                        $siteGroups[$docRoot] = [
                            'primary_domain' => $domain,
                            'additional_domains' => [],
                            'document_root' => $docRoot
                        ];
                    } else {
                        $siteGroups[$docRoot]['additional_domains'][] = $domain;
                    }
                }
            }
        }
        
        // Store site groups for later use
        $this->siteGroups = $siteGroups;
        
        $this->log("Found " . count($domains) . " domains in userdata, grouped into " . count($siteGroups) . " sites");
        return $domains;
    }
    
    /**
     * Parse individual userdata file
     */
    private function parseUserdataFile($file) {
        $content = file_get_contents($file);
        if (!$content) return null;
        
        $config = [];
        $lines = explode("\n", $content);
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line) || strpos($line, ':') === false) continue;
            
            list($key, $value) = explode(':', $line, 2);
            $key = trim($key);
            $value = trim($value);
            
            if ($key === 'documentroot') {
                $config['document_root'] = $value;
            } elseif ($key === 'serveralias') {
                $config['aliases'] = array_filter(explode(' ', $value));
            } elseif ($key === 'ssl') {
                $config['ssl_enabled'] = ($value === '1');
            }
        }
        
        return $config;
    }
    
    /**
     * Parse addon domains file
     */
    private function parseAddons($addonsFile) {
        if (!file_exists($addonsFile)) {
            return [];
        }
        
        $content = file_get_contents($addonsFile);
        $addons = [];
        
        // Parse the addon domains format
        $lines = explode("\n", trim($content));
        foreach ($lines as $line) {
            if (trim($line)) {
                // Format is typically: domain.com=subdomain.maindomain.com
                if (strpos($line, '=') !== false) {
                    list($addon, $subdomain) = explode('=', $line, 2);
                    $addons[trim($addon)] = trim($subdomain);
                }
            }
        }
        
        $this->log("Found " . count($addons) . " addon domains");
        return $addons;
    }
    
    /**
     * Parse subdomains file
     */
    private function parseSubdomains($subdomainsFile) {
        if (!file_exists($subdomainsFile)) {
            return [];
        }
        
        $content = file_get_contents($subdomainsFile);
        $subdomains = [];
        
        $lines = explode("\n", trim($content));
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line) {
                $subdomains[] = $line;
            }
        }
        
        $this->log("Found " . count($subdomains) . " subdomains");
        return $subdomains;
    }
    
    /**
     * Parse MySQL grants file
     */
    private function parseMysqlGrants($grantsFile) {
        if (!file_exists($grantsFile)) {
            return [];
        }
        
        $content = file_get_contents($grantsFile);
        $grants = [];
        
        // Parse GRANT statements to extract database users
        preg_match_all("/GRANT .* TO '([^']+)'@'([^']+)'/", $content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $username = $match[1];
            $host = $match[2];
            $grants[] = ['username' => $username, 'host' => $host];
        }
        
        $this->log("Found " . count($grants) . " MySQL user grants");
        return $grants;
    }
    
    /**
     * Get MySQL database files
     */
    private function getMysqlDatabases($mysqlDir) {
        $databases = [];
        
        if (!is_dir($mysqlDir)) {
            return $databases;
        }
        
        foreach (glob($mysqlDir . "/*.sql") as $file) {
            $dbName = basename($file, '.sql');
            $databases[$dbName] = $file;
        }
        
        $this->log("Found " . count($databases) . " MySQL databases");
        return $databases;
    }
    
    /**
     * Import domains and create WHP domain entries
     */
    private function importDomains($backupData) {
        $importedDomains = [];
        
        // Import main domains from userdata
        foreach ($backupData['domains'] as $domain => $config) {
            $importedDomains[] = [
                'domain' => $domain,
                'type' => 'primary',
                'config' => $config
            ];
        }
        
        // Import addon domains
        foreach ($backupData['addons'] as $addon => $subdomain) {
            $importedDomains[] = [
                'domain' => $addon,
                'type' => 'addon',
                'subdomain' => $subdomain
            ];
        }
        
        // Import subdomains
        foreach ($backupData['subdomains'] as $subdomain) {
            $importedDomains[] = [
                'domain' => $subdomain,
                'type' => 'subdomain'
            ];
        }
        
        $this->log("Prepared " . count($importedDomains) . " domains for import");
        return $importedDomains;
    }
    
    /**
     * Create WHP sites for imported domains using document root groupings
     */
    private function createSites($domains) {
        $createdSites = [];
        
        // Create sites based on document root groupings
        foreach ($this->siteGroups as $docRoot => $siteGroup) {
            try {
                $primaryDomain = $siteGroup['primary_domain'];
                $additionalDomains = $siteGroup['additional_domains'];
                
                // Create site entry in WHP with primary domain
                $siteId = $this->createWhpSite($primaryDomain, $siteGroup);
                
                if ($siteId) {
                    $createdSites[] = [
                        'domain' => $primaryDomain,
                        'site_id' => $siteId,
                        'type' => 'primary',
                        'document_root' => $docRoot,
                        'additional_domains' => $additionalDomains
                    ];
                    
                    $this->log("Created WHP site for primary domain: " . $primaryDomain);
                    
                    // Add additional domains to the same site
                    foreach ($additionalDomains as $additionalDomain) {
                        $this->addDomainToSite($siteId, $additionalDomain);
                        $this->log("Added additional domain to site: " . $additionalDomain);
                    }
                }
                
            } catch (Exception $e) {
                $this->log("Failed to create site for document root " . $docRoot . ": " . $e->getMessage(), 'error');
            }
        }
        
        return $createdSites;
    }
    
    /**
     * Create a WHP site entry
     */
    private function createWhpSite($domain, $domainData) {
        // First ensure we have a default container type
        $stmt = $this->pdo->prepare("SELECT id FROM whp.container_types WHERE name = 'PHP 8.2' AND active = 1 LIMIT 1");
        $stmt->execute();
        $containerType = $stmt->fetch();
        
        if (!$containerType) {
            // Use first active container type
            $stmt = $this->pdo->prepare("SELECT id FROM whp.container_types WHERE active = 1 LIMIT 1");
            $stmt->execute();
            $containerType = $stmt->fetch();
            
            if (!$containerType) {
                throw new Exception("No active container types available");
            }
        }
        
        $containerTypeId = $containerType['id'];
        
        // Check if domain exists in WHP
        $stmt = $this->pdo->prepare("SELECT id, domain_name FROM whp.domains WHERE domain_name = ? AND username = ?");
        $stmt->execute([$domain, $this->username]);
        $domainRow = $stmt->fetch();
        
        if (!$domainRow) {
            // Create domain first
            $stmt = $this->pdo->prepare("
                INSERT INTO whp.domains (domain_name, username, active, created_at) 
                VALUES (?, ?, 1, NOW())
            ");
            $stmt->execute([$domain, $this->username]);
            $domainId = $this->pdo->lastInsertId();
        } else {
            $domainId = $domainRow['id'];
        }
        
        // Create site directory
        $siteDir = '/docker/users/' . $this->username . '/' . $domain;
        if (!is_dir($siteDir)) {
            mkdir($siteDir, 0755, true);
            
            // Set ownership
            $uid = trim(shell_exec("id -u " . escapeshellarg($this->username)));
            $gid = trim(shell_exec("id -g " . escapeshellarg($this->username)));
            if (is_numeric($uid) && is_numeric($gid)) {
                shell_exec("chown -R {$uid}:{$gid} " . escapeshellarg($siteDir));
            }
        }
        
        // Insert site into database with proper structure
        $siteName = $domain . " (Imported from cPanel)";
        
        $stmt = $this->pdo->prepare("
            INSERT INTO whp.sites (
                site_name, 
                primary_domain_id, 
                username, 
                container_type_id, 
                container_count, 
                cpu_per_container, 
                memory_per_container, 
                user_variables, 
                auto_recreate_on_update,
                created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $siteName,
            $domainId,
            $this->username,
            $containerTypeId,
            1, // container_count
            0.25, // cpu_per_container
            256, // memory_per_container
            '{}', // user_variables
            0 // auto_recreate_on_update
        ]);
        
        $siteId = $this->pdo->lastInsertId();
        
        // Insert into site_domains table for relationship
        $stmt = $this->pdo->prepare("
            INSERT INTO whp.site_domains (site_id, domain_id, is_primary) 
            VALUES (?, ?, 1)
        ");
        $stmt->execute([$siteId, $domainId]);
        
        return $siteId;
    }
    
    
    /**
     * Import files for sites based on document root groupings
     */
    private function importFiles($domains) {
        foreach ($this->siteGroups as $docRoot => $siteGroup) {
            $primaryDomain = $siteGroup['primary_domain'];
            $this->importSiteFiles($primaryDomain, $docRoot);
        }
    }
    
    /**
     * Import files for a specific site based on document root
     */
    private function importSiteFiles($primaryDomain, $documentRoot) {
        // Map cPanel document root to actual path in backup
        $sourceSubPath = $this->mapDocumentRootToPath($documentRoot);
        $sourcePath = $this->extractPath . "/*/homedir" . $sourceSubPath;
        $destPath = "/docker/users/{$this->username}/{$primaryDomain}/public_html";
        
        // Create destination directory
        if (!is_dir(dirname($destPath))) {
            mkdir(dirname($destPath), 0755, true);
        }
        
        // Find the actual source path (accounting for glob pattern)
        $actualSourcePaths = glob($sourcePath);
        if (empty($actualSourcePaths)) {
            $this->log("Source path not found for document root: " . $documentRoot, 'warning');
            return;
        }
        
        $actualSourcePath = $actualSourcePaths[0];
        
        // Copy files using rsync for better performance
        $command = "rsync -av " . escapeshellarg($actualSourcePath . "/") . " " . escapeshellarg($destPath . "/");
        $output = [];
        $returnCode = 0;
        
        exec($command, $output, $returnCode);
        
        if ($returnCode === 0) {
            $this->log("Files imported for site: " . $primaryDomain . " (document root: " . $documentRoot . ")");
        } else {
            $this->log("Failed to import files for site: " . $primaryDomain . " - " . implode(" ", $output), 'error');
        }
    }
    
    /**
     * Map cPanel document root to actual path in backup
     */
    private function mapDocumentRootToPath($documentRoot) {
        // Handle different document root formats from cPanel
        if ($documentRoot === '/public_html' || empty($documentRoot)) {
            return '/public_html';
        } elseif (strpos($documentRoot, '/public_html/') === 0) {
            return $documentRoot;
        } else {
            // Handle cases like '/home/user/public_html/subdirectory'
            if (strpos($documentRoot, '/public_html/') !== false) {
                return substr($documentRoot, strpos($documentRoot, '/public_html/'));
            }
            // Default fallback
            return '/public_html';
        }
    }
    
    /**
     * Import MySQL databases
     */
    private function importDatabases($backupData) {
        foreach ($backupData['mysql_databases'] as $dbName => $sqlFile) {
            $this->importDatabase($dbName, $sqlFile);
        }
    }
    
    /**
     * Import a single database using WHP's mysqlmgmt class
     */
    private function importDatabase($originalDbName, $sqlFile) {
        // Initialize mysqlmgmt if not already done
        if (!isset($this->mysqlMgmt)) {
            if (!class_exists('mysqlmgmt')) {
                require_once('/docker/whp/web/libs/mysqlmgmt.php');
            }
            $this->mysqlMgmt = new mysqlmgmt();
        }
        
        // Generate WHP-compatible database name
        $random = substr(md5(uniqid()), 0, 8);
        $whpDbName = $this->username . '_' . preg_replace('/[^a-zA-Z0-9]/', '', $originalDbName) . '_' . $random;
        
        try {
            // Create database using WHP's method
            $result = $this->mysqlMgmt->createDatabase($whpDbName, $this->username);
            
            if ($result['status'] === '0') {
                $this->log("Database created: " . $originalDbName . " -> " . $whpDbName);
                
                // Import SQL file
                $command = "mysql " . escapeshellarg($whpDbName) . " < " . escapeshellarg($sqlFile);
                $output = [];
                $returnCode = 0;
                
                exec($command, $output, $returnCode);
                
                if ($returnCode === 0) {
                    $this->log("Database imported: " . $originalDbName . " -> " . $whpDbName);
                } else {
                    throw new Exception("Failed to import database: " . implode("\n", $output));
                }
            } else {
                $this->log("Failed to create database " . $originalDbName . ": " . $result['msg'], 'error');
            }
            
        } catch (Exception $e) {
            $this->log("Database import failed for " . $originalDbName . ": " . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Import MySQL users
     */
    private function importMysqlUsers($backupData) {
        foreach ($backupData['mysql_grants'] as $grant) {
            $this->createMysqlUser($grant);
        }
    }
    
    /**
     * Create a MySQL user with WHP naming convention using mysqlmgmt class
     */
    private function createMysqlUser($grant) {
        // Initialize mysqlmgmt if not already done
        if (!isset($this->mysqlMgmt)) {
            if (!class_exists('mysqlmgmt')) {
                require_once('/docker/whp/web/libs/mysqlmgmt.php');
            }
            $this->mysqlMgmt = new mysqlmgmt();
        }
        
        $originalUsername = $grant['username'];
        
        // Generate WHP-compatible username
        $random = substr(md5(uniqid()), 0, 8);
        $whpUsername = $this->username . '_' . $random;
        $password = bin2hex(random_bytes(16));
        
        try {
            // Create user using WHP's method
            $result = $this->mysqlMgmt->createMySQLUser($whpUsername, $password);
            
            if ($result['status'] === '0') {
                $this->log("MySQL user created: " . $originalUsername . " -> " . $whpUsername);
            } else {
                $this->log("Failed to create MySQL user " . $originalUsername . ": " . $result['msg'], 'error');
            }
            
        } catch (Exception $e) {
            $this->log("Failed to create MySQL user " . $originalUsername . ": " . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Create DNS records for imported domains using PowerDNS integration
     */
    private function createDnsRecords($domains) {
        // Initialize PowerDNS integration if not already done
        if (!isset($this->powerDNS)) {
            try {
                require_once('/docker/whp/web/libs/pdns-integration.php');
                $this->powerDNS = new PowerDNSIntegration($this->pdo);
            } catch (Exception $e) {
                $this->log("PowerDNS integration not available - skipping DNS record creation: " . $e->getMessage(), 'warning');
                return;
            }
        }
        
        foreach ($this->siteGroups as $docRoot => $siteGroup) {
            $primaryDomain = $siteGroup['primary_domain'];
            $additionalDomains = $siteGroup['additional_domains'];
            
            // Create DNS records for primary domain
            $this->createDomainDnsRecords($primaryDomain);
            
            // Create DNS records for additional domains
            foreach ($additionalDomains as $domain) {
                $this->createDomainDnsRecords($domain);
            }
        }
    }
    
    /**
     * Create DNS records for a specific domain using PowerDNS
     */
    private function createDomainDnsRecords($domain) {
        try {
            // Check if domain zone exists
            $zoneExists = false;
            $zones = $this->powerDNS->getZones();
            foreach ($zones as $zone) {
                if ($zone['name'] === $domain . '.') {
                    $zoneExists = true;
                    break;
                }
            }
            
            if (!$zoneExists) {
                // Create zone
                $result = $this->powerDNS->createZone($domain, $this->username);
                if ($result) {
                    $this->log("DNS zone created for: $domain");
                } else {
                    $this->log("Failed to create DNS zone for: $domain", 'warning');
                    return;
                }
            }
            
            // Get server IP
            $serverIp = $this->getServerIp();
            if (!$serverIp) {
                $this->log("Could not determine server IP - skipping DNS records for: $domain", 'warning');
                return;
            }
            
            // Add A records pointing to server IP
            try {
                $this->powerDNS->addRecord($domain, $domain, 'A', $serverIp, 300);
                $this->log("Created DNS A record: $domain -> $serverIp");
                
                $this->powerDNS->addRecord($domain, 'www.' . $domain, 'A', $serverIp, 300);
                $this->log("Created DNS A record: www.$domain -> $serverIp");
            } catch (Exception $e) {
                $this->log("Failed to create DNS A records for $domain: " . $e->getMessage(), 'warning');
            }
            
        } catch (Exception $e) {
            $this->log("Failed to create DNS records for " . $domain . ": " . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Check if DNS manager is available
     */
    private function isDnsManagerAvailable() {
        return file_exists('/home/jknapp/code/whp/web-files/includes/dns_manager.php');
    }
    
    /**
     * Get server IP address
     */
    private function getServerIp() {
        // Try to get from server settings first
        $settingsFile = '/docker/whp/settings.json';
        if (file_exists($settingsFile)) {
            $settings = json_decode(file_get_contents($settingsFile), true);
            if (!empty($settings['server_ip'])) {
                return $settings['server_ip'];
            }
        }
        
        // Fallback to detecting server IP
        $ip = trim(file_get_contents("http://ipv4.icanhazip.com"));
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
            return $ip;
        }
        
        // Final fallback
        return '127.0.0.1';
    }
    
    /**
     * Add domain to existing site (for additional domains)
     */
    private function addDomainToSite($siteId, $domain) {
        try {
            // Check if domain exists in WHP
            $stmt = $this->pdo->prepare("SELECT id FROM whp.domains WHERE domain_name = ? AND username = ?");
            $stmt->execute([$domain, $this->username]);
            $domainRow = $stmt->fetch();
            
            if (!$domainRow) {
                // Create domain first
                $stmt = $this->pdo->prepare("
                    INSERT INTO whp.domains (domain_name, username, active, created_at) 
                    VALUES (?, ?, 1, NOW())
                ");
                $stmt->execute([$domain, $this->username]);
                $domainId = $this->pdo->lastInsertId();
            } else {
                $domainId = $domainRow['id'];
            }
            
            // Add to site_domains using the correct schema (domain_id, not domain)
            $stmt = $this->pdo->prepare("
                INSERT INTO whp.site_domains (site_id, domain_id, is_primary) 
                VALUES (?, ?, 0)
                ON DUPLICATE KEY UPDATE is_primary = 0
            ");
            $stmt->execute([$siteId, $domainId]);
            
        } catch (PDOException $e) {
            $this->log("Failed to add domain " . $domain . " to site " . $siteId . ": " . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Cleanup temporary files
     */
    private function cleanup() {
        // Clean up extraction directory
        if (is_dir($this->extractPath)) {
            $this->removeDirectory($this->extractPath);
        }
        
        // Clean up backup file and its temp directory
        if (file_exists($this->backupPath)) {
            $tempDir = dirname($this->backupPath);
            unlink($this->backupPath);
            if (is_dir($tempDir) && strpos($tempDir, '/tmp/cpanel_processing_') === 0) {
                rmdir($tempDir);
            }
        }
        
        $this->log("Cleanup completed");
    }
    
    /**
     * Recursively remove directory
     */
    private function removeDirectory($dir) {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (is_dir($dir . "/" . $object)) {
                        $this->removeDirectory($dir . "/" . $object);
                    } else {
                        unlink($dir . "/" . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }
    
    /**
     * Log import progress
     */
    private function log($message, $level = 'info') {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[$timestamp] [$level] $message";
        
        $this->importLog[] = $logEntry;
        
        // Also log to system log
        error_log("CpanelBackupImporter: $logEntry");
    }
    
    /**
     * Get import log
     */
    public function getLog() {
        return $this->importLog;
    }
    
    /**
     * Validate domain name format
     */
    private function isValidDomain($domain) {
        // Basic domain validation
        if (empty($domain) || strlen($domain) > 253) {
            return false;
        }
        
        // Check for valid characters and structure
        return preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]*[a-zA-Z0-9]$/', $domain) === 1;
    }
    
    /**
     * Get directory size recursively
     */
    private function getDirectorySize($directory) {
        $size = 0;
        
        if (!is_dir($directory)) {
            return 0;
        }
        
        try {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
            );
            
            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $size += $file->getSize();
                }
            }
        } catch (Exception $e) {
            $this->log("Failed to calculate directory size: " . $e->getMessage(), 'warning');
            return 0;
        }
        
        return $size;
    }
    
    /**
     * Create rollback point for import
     */
    private function createRollbackPoint() {
        $rollbackData = [
            'timestamp' => time(),
            'username' => $this->username,
            'sites_before_import' => $this->getCurrentUserSites(),
            'databases_before_import' => $this->getCurrentUserDatabases()
        ];
        
        $rollbackFile = "/tmp/cpanel_rollback_" . $this->username . "_" . uniqid() . ".json";
        file_put_contents($rollbackFile, json_encode($rollbackData, JSON_PRETTY_PRINT));
        
        $this->log("Rollback point created: " . $rollbackFile);
        return $rollbackFile;
    }
    
    /**
     * Get current user sites for rollback
     */
    private function getCurrentUserSites() {
        try {
            $stmt = $this->pdo->prepare("SELECT id, domain FROM sites WHERE username = ?");
            $stmt->execute([$this->username]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            $this->log("Failed to get current sites for rollback: " . $e->getMessage(), 'warning');
            return [];
        }
    }
    
    /**
     * Get current user databases for rollback
     */
    private function getCurrentUserDatabases() {
        try {
            $stmt = $this->pdo->prepare("SHOW DATABASES LIKE ?");
            $stmt->execute([$this->username . '_%']);
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            $this->log("Failed to get current databases for rollback: " . $e->getMessage(), 'warning');
            return [];
        }
    }
    
    /**
     * Perform rollback using rollback file
     */
    public function performRollback($rollbackFile) {
        if (!file_exists($rollbackFile)) {
            throw new Exception("Rollback file not found: " . $rollbackFile);
        }
        
        $rollbackData = json_decode(file_get_contents($rollbackFile), true);
        if (!$rollbackData) {
            throw new Exception("Invalid rollback file format");
        }
        
        $this->log("Starting rollback process");
        
        // Remove sites created during import
        $currentSites = $this->getCurrentUserSites();
        $originalSites = $rollbackData['sites_before_import'] ?? [];
        $originalSiteIds = array_column($originalSites, 'id');
        
        foreach ($currentSites as $site) {
            if (!in_array($site['id'], $originalSiteIds)) {
                $this->removeSite($site['id']);
                $this->log("Removed site: " . $site['domain']);
            }
        }
        
        // Remove databases created during import
        $currentDatabases = $this->getCurrentUserDatabases();
        $originalDatabases = $rollbackData['databases_before_import'] ?? [];
        
        foreach ($currentDatabases as $database) {
            if (!in_array($database, $originalDatabases)) {
                $this->removeDatabase($database);
                $this->log("Removed database: " . $database);
            }
        }
        
        // Clean up rollback file
        unlink($rollbackFile);
        
        $this->log("Rollback completed successfully");
    }
    
    /**
     * Remove a site (placeholder - integrate with your site management)
     */
    private function removeSite($siteId) {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM sites WHERE id = ? AND username = ?");
            $stmt->execute([$siteId, $this->username]);
        } catch (PDOException $e) {
            $this->log("Failed to remove site " . $siteId . ": " . $e->getMessage(), 'error');
        }
    }
    
    /**
     * Remove a database
     */
    private function removeDatabase($databaseName) {
        try {
            $stmt = $this->pdo->prepare("DROP DATABASE IF EXISTS " . $databaseName);
            $stmt->execute();
        } catch (PDOException $e) {
            $this->log("Failed to remove database " . $databaseName . ": " . $e->getMessage(), 'error');
        }
    }
}