<?php
require_once 'libs/session_check.php';
require_once 'libs/CpanelBackupImporter.php';

$pageTitle = "cPanel Backup Import";
$currentPage = 'cpanel-import';

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        switch ($_POST['action']) {
            case 'get_backup_files':
                $result = ['success' => true, 'files' => getAvailableBackupFiles()];
                echo json_encode($result);
                break;
                
            case 'select_backup':
                $result = selectBackupFile();
                echo json_encode($result);
                break;
                
            case 'start_import':
                $result = startImport($_POST['backup_file']);
                echo json_encode($result);
                break;
                
            case 'get_import_status':
                $result = getImportStatus($_POST['import_id']);
                echo json_encode($result);
                break;
                
            default:
                echo json_encode(['success' => false, 'error' => 'Invalid action']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

function getAvailableBackupFiles() {
    $userfilesDir = '/docker/users/' . $_SESSION['username'] . '/userfiles';
    $backupFiles = [];
    
    if (is_dir($userfilesDir)) {
        $files = glob($userfilesDir . '/*.{tar.gz,tgz}', GLOB_BRACE);
        foreach ($files as $file) {
            $backupFiles[] = [
                'filename' => basename($file),
                'path' => $file,
                'size' => filesize($file),
                'modified' => filemtime($file)
            ];
        }
        
        // Sort by modification time (newest first)
        usort($backupFiles, function($a, $b) {
            return $b['modified'] - $a['modified'];
        });
    }
    
    return $backupFiles;
}

function selectBackupFile() {
    if (!isset($_POST['backup_file'])) {
        return ['success' => false, 'error' => 'No backup file selected'];
    }
    
    $selectedFile = $_POST['backup_file'];
    $userfilesDir = '/docker/users/' . $_SESSION['username'] . '/userfiles';
    $fullPath = $userfilesDir . '/' . basename($selectedFile);
    
    // Validate file exists and is in userfiles directory
    if (!file_exists($fullPath) || dirname($fullPath) !== $userfilesDir) {
        return ['success' => false, 'error' => 'Invalid backup file selected'];
    }
    
    // Validate file extension
    if (!preg_match('/\.(tar\.gz|tgz)$/i', $selectedFile)) {
        return ['success' => false, 'error' => 'Invalid file type. Must be a .tar.gz or .tgz file'];
    }
    
    return [
        'success' => true,
        'filename' => basename($selectedFile),
        'path' => $fullPath,
        'size' => filesize($fullPath)
    ];
}

function startImport($backupFilePath) {
    global $pdo;
    
    if (!file_exists($backupFilePath)) {
        return ['success' => false, 'error' => 'Backup file not found'];
    }
    
    // Create import record
    $importId = uniqid('import_');
    $stmt = $pdo->prepare("
        INSERT INTO cpanel_imports (import_id, username, backup_file, status, created_at) 
        VALUES (?, ?, ?, 'running', NOW())
    ");
    $stmt->execute([$importId, $_SESSION['username'], $backupFilePath]);
    
    // Start import in background
    $command = "php " . __DIR__ . "/../scripts/run-cpanel-import.php " . 
               escapeshellarg($importId) . " " . 
               escapeshellarg($_SESSION['username']) . " " . 
               escapeshellarg($backupFilePath) . " > /dev/null 2>&1 &";
    
    exec($command);
    
    return [
        'success' => true,
        'import_id' => $importId
    ];
}

function getImportStatus($importId) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT * FROM cpanel_imports WHERE import_id = ? AND username = ?");
    $stmt->execute([$importId, $_SESSION['username']]);
    $import = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$import) {
        return ['success' => false, 'error' => 'Import not found'];
    }
    
    return [
        'success' => true,
        'status' => $import['status'],
        'progress' => $import['progress'] ?? 0,
        'log' => $import['log'] ? json_decode($import['log'], true) : [],
        'error' => $import['error_message'],
        'completed_at' => $import['completed_at']
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> - WHP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bi bi-upload"></i> cPanel Backup Import</h2>
                    <a href="?page=dashboard" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Dashboard
                    </a>
                </div>
                
                <!-- Early Release Notice -->
                <div class="alert alert-warning">
                    <h6><i class="bi bi-exclamation-triangle"></i> Early Release Feature</h6>
                    <p class="mb-0">This cPanel backup import feature is currently in early release. While functional, please test with non-critical data first and report any issues you encounter.</p>
                </div>
            </div>
        </div>

        <!-- Import Steps -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Import Process</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="text-center">
                                    <div class="bg-primary text-white rounded-circle d-inline-flex align-items-center justify-content-center" 
                                         style="width: 40px; height: 40px;">
                                        <span>1</span>
                                    </div>
                                    <p class="mt-2 mb-0">Upload Backup</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <div class="bg-secondary text-white rounded-circle d-inline-flex align-items-center justify-content-center" 
                                         style="width: 40px; height: 40px;" id="step2">
                                        <span>2</span>
                                    </div>
                                    <p class="mt-2 mb-0">Analyze Backup</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <div class="bg-secondary text-white rounded-circle d-inline-flex align-items-center justify-content-center" 
                                         style="width: 40px; height: 40px;" id="step3">
                                        <span>3</span>
                                    </div>
                                    <p class="mt-2 mb-0">Import Data</p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <div class="bg-secondary text-white rounded-circle d-inline-flex align-items-center justify-content-center" 
                                         style="width: 40px; height: 40px;" id="step4">
                                        <span>4</span>
                                    </div>
                                    <p class="mt-2 mb-0">Complete</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Backup Selection Form -->
        <div class="row" id="selection-section">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Step 1: Select cPanel Backup File</h5>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-info">
                            <h6><i class="bi bi-info-circle"></i> Upload Instructions</h6>
                            <p><strong>Before using this tool:</strong></p>
                            <ol class="mb-2">
                                <li>Upload your cPanel backup (.tar.gz or .tgz) file to your <strong>userfiles</strong> directory</li>
                                <li>You can access userfiles through the File Manager or FTP at: <code>/userfiles/</code></li>
                                <li>Once uploaded, refresh this page and select your backup file below</li>
                            </ol>
                            <p class="mb-0"><strong>This tool will import:</strong> Domains, website files, MySQL databases, and create DNS records. SSL certificates are handled automatically by our system.</p>
                        </div>

                        <div id="backup-files-container">
                            <div class="mb-3">
                                <label for="backup_file_select" class="form-label">Available Backup Files</label>
                                <select class="form-select" id="backup_file_select" required>
                                    <option value="">Loading backup files...</option>
                                </select>
                                <div class="form-text">
                                    Backup files from your userfiles directory. 
                                    <button type="button" class="btn btn-link btn-sm p-0" onclick="refreshBackupFiles()">
                                        <i class="bi bi-arrow-clockwise"></i> Refresh List
                                    </button>
                                </div>
                            </div>
                            
                            <div id="selected-backup-info" class="alert alert-secondary" style="display: none;">
                                <h6>Selected Backup:</h6>
                                <div id="backup-details"></div>
                            </div>

                            <button type="button" class="btn btn-primary" id="select-backup-btn" disabled>
                                <i class="bi bi-check-circle"></i> Select This Backup
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Import Progress -->
        <div class="row" id="import-section" style="display: none;">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Import Progress</h5>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <div class="progress">
                                <div class="progress-bar progress-bar-striped progress-bar-animated" 
                                     id="import-progress" role="progressbar" style="width: 0%"></div>
                            </div>
                        </div>
                        
                        <div id="import-status" class="mb-3">
                            <span class="badge bg-primary">Starting Import...</span>
                        </div>
                        
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0">Import Log</h6>
                            </div>
                            <div class="card-body" style="max-height: 300px; overflow-y: auto;">
                                <pre id="import-log" class="mb-0"></pre>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Import Results -->
        <div class="row" id="results-section" style="display: none;">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Import Complete</h5>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-success">
                            <h6><i class="bi bi-check-circle"></i> Import Successful!</h6>
                            <div id="import-summary"></div>
                        </div>
                        
                        <div class="d-flex gap-2">
                            <a href="?page=sites" class="btn btn-primary">
                                <i class="bi bi-globe"></i> View Sites
                            </a>
                            <a href="?page=dashboard" class="btn btn-outline-secondary">
                                <i class="bi bi-house"></i> Dashboard
                            </a>
                            <button type="button" class="btn btn-outline-primary" onclick="startNewImport()">
                                <i class="bi bi-upload"></i> Import Another Backup
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const nonce = '<?php echo $_SESSION['nonce'] ?? ''; ?>';
        let currentImportId = null;
        let statusCheckInterval = null;
        let availableBackups = [];

        // Load backup files on page load
        document.addEventListener('DOMContentLoaded', function() {
            refreshBackupFiles();
        });

        // Refresh backup files list
        async function refreshBackupFiles() {
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'get_backup_files',
                        nonce: nonce
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    availableBackups = result.files;
                    updateBackupFilesList();
                } else {
                    console.error('Failed to load backup files:', result.error);
                    updateBackupFilesList([]);
                }
            } catch (error) {
                console.error('Error loading backup files:', error);
                updateBackupFilesList([]);
            }
        }

        // Update the backup files dropdown
        function updateBackupFilesList() {
            const select = document.getElementById('backup_file_select');
            select.innerHTML = '';

            if (availableBackups.length === 0) {
                select.innerHTML = '<option value="">No backup files found in userfiles directory</option>';
                document.getElementById('select-backup-btn').disabled = true;
            } else {
                select.innerHTML = '<option value="">Select a backup file...</option>';
                
                availableBackups.forEach(backup => {
                    const option = document.createElement('option');
                    option.value = backup.filename;
                    option.textContent = `${backup.filename} (${formatFileSize(backup.size)} - ${formatDate(backup.modified)})`;
                    select.appendChild(option);
                });
            }
        }

        // Handle backup file selection
        document.getElementById('backup_file_select').addEventListener('change', function() {
            const selectedFilename = this.value;
            const selectedBackup = availableBackups.find(b => b.filename === selectedFilename);
            
            if (selectedBackup) {
                // Show backup details
                const detailsDiv = document.getElementById('backup-details');
                detailsDiv.innerHTML = `
                    <strong>File:</strong> ${selectedBackup.filename}<br>
                    <strong>Size:</strong> ${formatFileSize(selectedBackup.size)}<br>
                    <strong>Modified:</strong> ${formatDate(selectedBackup.modified)}
                `;
                document.getElementById('selected-backup-info').style.display = 'block';
                document.getElementById('select-backup-btn').disabled = false;
            } else {
                document.getElementById('selected-backup-info').style.display = 'none';
                document.getElementById('select-backup-btn').disabled = true;
            }
        });

        // Handle backup selection confirmation
        document.getElementById('select-backup-btn').addEventListener('click', async function() {
            const selectedFilename = document.getElementById('backup_file_select').value;
            
            if (!selectedFilename) {
                alert('Please select a backup file');
                return;
            }

            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'select_backup',
                        backup_file: selectedFilename,
                        nonce: nonce
                    })
                });

                const result = await response.json();

                if (result.success) {
                    // Update step indicator
                    updateStepIndicator(2);
                    
                    // Start import
                    startImport(result.path);
                } else {
                    alert('Failed to select backup: ' + result.error);
                }
            } catch (error) {
                alert('Failed to select backup: ' + error.message);
            }
        });

        async function startImport(backupFilePath) {
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'start_import',
                        backup_file: backupFilePath,
                        nonce: nonce
                    })
                });

                const result = await response.json();

                if (result.success) {
                    currentImportId = result.import_id;
                    
                    // Hide selection section and show import progress
                    document.getElementById('selection-section').style.display = 'none';
                    document.getElementById('import-section').style.display = 'block';
                    
                    // Update step indicator
                    updateStepIndicator(3);
                    
                    // Start status checking
                    startStatusChecking();
                } else {
                    alert('Failed to start import: ' + result.error);
                }
            } catch (error) {
                alert('Failed to start import: ' + error.message);
            }
        }

        function startStatusChecking() {
            statusCheckInterval = setInterval(checkImportStatus, 2000);
            checkImportStatus(); // Check immediately
        }

        async function checkImportStatus() {
            if (!currentImportId) return;

            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'get_import_status',
                        import_id: currentImportId,
                        nonce: nonce
                    })
                });

                const result = await response.json();

                if (result.success) {
                    updateImportProgress(result);
                    
                    if (result.status === 'completed' || result.status === 'failed') {
                        clearInterval(statusCheckInterval);
                        
                        if (result.status === 'completed') {
                            showImportResults(result);
                        } else {
                            showImportError(result.error);
                        }
                    }
                }
            } catch (error) {
                console.error('Failed to check import status:', error);
            }
        }

        function updateImportProgress(status) {
            const progressBar = document.getElementById('import-progress');
            const statusDiv = document.getElementById('import-status');
            const logDiv = document.getElementById('import-log');
            
            // Update progress bar
            progressBar.style.width = (status.progress || 0) + '%';
            
            // Update status
            let badgeClass = 'bg-primary';
            if (status.status === 'completed') badgeClass = 'bg-success';
            else if (status.status === 'failed') badgeClass = 'bg-danger';
            
            statusDiv.innerHTML = `<span class="badge ${badgeClass}">${status.status.toUpperCase()}</span>`;
            
            // Update log
            if (status.log && status.log.length > 0) {
                logDiv.textContent = status.log.join('\n');
                logDiv.scrollTop = logDiv.scrollHeight;
            }
        }

        function showImportResults(status) {
            // Update step indicator
            updateStepIndicator(4);
            
            // Hide import progress
            document.getElementById('import-section').style.display = 'none';
            
            // Show results
            document.getElementById('results-section').style.display = 'block';
            
            // TODO: Parse and display import summary from status
            const summary = document.getElementById('import-summary');
            summary.innerHTML = `
                <p>Your cPanel backup has been successfully imported!</p>
                <p>Import completed at: ${status.completed_at}</p>
            `;
        }

        function showImportError(error) {
            alert('Import failed: ' + error);
            
            // Reset to selection form
            document.getElementById('import-section').style.display = 'none';
            document.getElementById('selection-section').style.display = 'block';
            updateStepIndicator(1);
        }

        function updateStepIndicator(step) {
            for (let i = 1; i <= 4; i++) {
                const stepElement = document.getElementById(`step${i}`);
                if (stepElement) {
                    if (i <= step) {
                        stepElement.className = stepElement.className.replace('bg-secondary', 'bg-primary');
                    } else {
                        stepElement.className = stepElement.className.replace('bg-primary', 'bg-secondary');
                    }
                }
            }
        }

        function startNewImport() {
            // Reset form and show selection section
            document.getElementById('backup_file_select').value = '';
            document.getElementById('selected-backup-info').style.display = 'none';
            document.getElementById('select-backup-btn').disabled = true;
            document.getElementById('results-section').style.display = 'none';
            document.getElementById('selection-section').style.display = 'block';
            
            updateStepIndicator(1);
            currentImportId = null;
            
            if (statusCheckInterval) {
                clearInterval(statusCheckInterval);
            }
            
            refreshBackupFiles();
        }

        // Utility functions
        function formatFileSize(bytes) {
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            if (bytes === 0) return '0 Bytes';
            const i = Math.floor(Math.log(bytes) / Math.log(1024));
            return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
        }

        function formatDate(timestamp) {
            return new Date(timestamp * 1000).toLocaleString();
        }
    </script>
</body>
</html>