<?php
/**
 * Permission Checker Functions
 * Provides functions to check permissions and handle unauthorized access
 */

require_once('/docker/whp/web/libs/permission_manager.php');
require_once('/docker/whp/web/libs/network.php');

/**
 * Check if user has permission to access the current page
 * If not, log the attempt and redirect to dashboard with error message
 */
function check_page_permission() {
    // Only check if AUTH_USER is defined (user is authenticated)
    if (!defined('AUTH_USER')) {
        return;
    }
    
    $user = AUTH_USER;
    $page = isset($_GET['page']) ? $_GET['page'] : 'dashboard';
    
    // Initialize permission manager and network info
    $PermManager = new permission_manager();
    $NetInfo = new network_info();
    
    // Check permission
    $has_permission = $PermManager->check_page_permission($user, $page);
    
    // Get client info for logging
    $ip_address = $NetInfo->get_client_ip();
    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    
    if (!$has_permission) {
        // Log the denied access attempt
        $reason = "User {$user} does not have permission to access {$page}";
        $PermManager->log_access_attempt($user, $page, $ip_address, $user_agent, false, $reason);
        
        // Set error message in session
        $denial_message = $PermManager->get_denial_message($page, $user);
        $_SESSION['permission_error'] = $denial_message;
        
        // Redirect to dashboard
        header('Location: index.php');
        exit();
    } else {
        // Log successful access attempt
        $PermManager->log_access_attempt($user, $page, $ip_address, $user_agent, true);
    }
}

/**
 * Check if user has permission to access an API
 * If not, return JSON error response
 */
function check_api_permission($api_name) {
    // Only check if AUTH_USER is defined (user is authenticated)
    if (!defined('AUTH_USER')) {
        http_response_code(401);
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Authentication required'
        ));
        exit();
    }
    
    $user = AUTH_USER;
    
    // Initialize permission manager and network info
    $PermManager = new permission_manager();
    $NetInfo = new network_info();
    
    // Check permission
    $has_permission = $PermManager->check_api_permission($user, $api_name);
    
    // Get client info for logging
    $ip_address = $NetInfo->get_client_ip();
    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    
    if (!$has_permission) {
        // Log the denied access attempt
        $reason = "User {$user} does not have permission to access API {$api_name}";
        $PermManager->log_access_attempt($user, $api_name, $ip_address, $user_agent, false, $reason);
        
        // Return JSON error
        http_response_code(403);
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Access denied. Insufficient privileges.'
        ));
        exit();
    } else {
        // Log successful access attempt
        $PermManager->log_access_attempt($user, $api_name, $ip_address, $user_agent, true);
    }
}

/**
 * Display permission error message if one exists in session
 */
function display_permission_error() {
    if (isset($_SESSION['permission_error'])) {
        $error_message = $_SESSION['permission_error'];
        unset($_SESSION['permission_error']); // Clear the message after displaying
        
        echo '<div class="row">';
        echo '<div class="col-sm-12">';
        echo '<div class="alert alert-danger alert-dismissible" role="alert">';
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
        echo '<i class="fa fa-exclamation-triangle"></i> ';
        echo htmlspecialchars($error_message);
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }
}

/**
 * Get access statistics for the current user
 */
function get_user_access_statistics($days = 30) {
    if (!defined('AUTH_USER')) {
        return null;
    }
    
    $PermManager = new permission_manager();
    return $PermManager->get_user_access_stats(AUTH_USER, $days);
}

/**
 * Get system-wide access statistics (root only)
 */
function get_system_access_statistics($days = 30) {
    if (!defined('AUTH_USER') || AUTH_USER !== 'root') {
        return null;
    }
    
    $PermManager = new permission_manager();
    return $PermManager->get_system_access_stats($days);
}

/**
 * Check if user has permission to access a specific domain
 * 
 * @param string $domain The domain name
 * @return bool True if user has permission, false otherwise
 */
function check_domain_permission($domain) {
    if (!defined('AUTH_USER')) {
        return false;
    }
    
    $PermManager = new permission_manager();
    return $PermManager->check_domain_ownership(AUTH_USER, $domain);
}

/**
 * Get domains owned by the current user
 * 
 * @return array Array of domain names owned by the user
 */
function get_user_domains() {
    if (!defined('AUTH_USER')) {
        return array();
    }
    
    $PermManager = new permission_manager();
    return $PermManager->get_user_domains(AUTH_USER);
}

/**
 * Check if user is root
 * 
 * @return bool True if user is root, false otherwise
 */
function is_root_user() {
    return defined('AUTH_USER') && AUTH_USER === 'root';
} 