#!/bin/bash
# WHP Update Script
# Updates existing WHP installation from release package

# Exit on any error
set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Get the directory where this script is located
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

# Check if this is being run from extracted package
if [ ! -f "$SCRIPT_DIR/VERSION" ]; then
    echo -e "${RED}Error: update.sh must be run from extracted WHP package directory${NC}"
    echo "Expected to find VERSION file in: $SCRIPT_DIR"
    exit 1
fi

# Check if WHP is installed
if [ ! -f /etc/whp-version ] && [ ! -f /root/whp/VERSION ]; then
    echo -e "${RED}Error: WHP not installed. Please run setup.sh for new installations.${NC}"
    exit 1
fi

# Get current version (check both locations for compatibility)
if [ -f /etc/whp-version ]; then
    CURRENT_VERSION=$(cat /etc/whp-version)
elif [ -f /root/whp/VERSION ]; then
    CURRENT_VERSION=$(cat /root/whp/VERSION)
else
    echo -e "${RED}Error: Cannot determine current WHP version${NC}"
    exit 1
fi

NEW_VERSION=$(cat "$SCRIPT_DIR/VERSION")

echo -e "${BLUE}=== WHP Update ===${NC}"
echo "Current version: $CURRENT_VERSION"
echo "New version: $NEW_VERSION"

# Check if already on latest version
if [ "$CURRENT_VERSION" = "$NEW_VERSION" ]; then
    echo -e "${GREEN}Already running version $NEW_VERSION${NC}"
    exit 0
fi

# Check migration compatibility
if [ -f "$SCRIPT_DIR/MIGRATION_HISTORY.json" ]; then
    OLDEST_SUPPORTED=$(grep '"oldest_supported"' "$SCRIPT_DIR/MIGRATION_HISTORY.json" | cut -d'"' -f4)
    
    # Simple version comparison (works for YYYY.MM.N format)
    if [[ "$CURRENT_VERSION" < "$OLDEST_SUPPORTED" ]]; then
        echo -e "${RED}Error: Your current version ($CURRENT_VERSION) is too old.${NC}"
        echo "This update package only supports versions $OLDEST_SUPPORTED and newer."
        echo "Please update to an intermediate version first."
        echo ""
        echo "Visit https://whp-releases.cloud-hosting.io for older releases."
        exit 1
    fi
fi

# Confirmation prompt
echo ""
echo -e "${YELLOW}This will update WHP from $CURRENT_VERSION to $NEW_VERSION${NC}"
read -p "Continue with update? (y/N) " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Update cancelled."
    exit 0
fi

# Backup current installation
echo ""
echo "Creating backup..."
if [ -x /root/whp/scripts/backup-whp.sh ]; then
    /root/whp/scripts/backup-whp.sh
else
    # Simple backup if backup script doesn't exist
    BACKUP_FILE="/root/whp-backup-$(date +%Y%m%d-%H%M%S).tar.gz"
    tar -czf "$BACKUP_FILE" -C / root/whp docker/whp/web 2>/dev/null || true
    echo "Backup created: $BACKUP_FILE"
fi

# Create WHP directory if needed
mkdir -p /root/whp

# First, copy the apply-migrations.sh script if it exists
if [ -f "$SCRIPT_DIR/scripts/apply-migrations.sh" ]; then
    echo "Updating migration scripts..."
    mkdir -p /root/whp/scripts
    cp "$SCRIPT_DIR/scripts/apply-migrations.sh" /root/whp/scripts/
    chmod +x /root/whp/scripts/apply-migrations.sh
fi

# Apply SQL migrations before file sync
if [ -d "$SCRIPT_DIR/sql/migrations" ]; then
    echo ""
    echo "Applying database migrations..."
    
    # Create a temporary script to apply migrations
    cat > /tmp/whp-apply-migrations.sh << 'MIGRATION_SCRIPT'
#!/bin/bash
CURRENT_VERSION="$1"
NEW_VERSION="$2"
MIGRATION_DIR="$3"

# Function to compare versions (YYYY.MM.N format)
version_compare() {
    # Convert version to comparable number: YYYYMMN
    local v1=$(echo "$1" | sed 's/\.//g')
    local v2=$(echo "$2" | sed 's/\.//g')
    
    if [ "$v1" -lt "$v2" ]; then
        echo "-1"
    elif [ "$v1" -gt "$v2" ]; then
        echo "1"
    else
        echo "0"
    fi
}

# Get list of versions between current and new
cd "$MIGRATION_DIR"
VERSIONS=($(ls -d */ 2>/dev/null | sed 's/\///' | sort -V))

echo "Found ${#VERSIONS[@]} migration versions"

# Apply each version's migrations in order
for VERSION in "${VERSIONS[@]}"; do
    COMPARE_CURRENT=$(version_compare "$VERSION" "$CURRENT_VERSION")
    COMPARE_NEW=$(version_compare "$VERSION" "$NEW_VERSION")
    
    if [ "$COMPARE_CURRENT" = "1" ] && [ "$COMPARE_NEW" -le "0" ]; then
        echo "Applying migrations for version $VERSION..."
        
        # Get all SQL files in this version directory
        for SQL_FILE in "$VERSION"/*.sql; do
            if [ -f "$SQL_FILE" ]; then
                echo "  - Running: $SQL_FILE"
                mysql < "$SQL_FILE"
                if [ $? -ne 0 ]; then
                    echo "Error applying migration: $SQL_FILE"
                    exit 1
                fi
            fi
        done
    fi
done

echo "All migrations applied successfully"
MIGRATION_SCRIPT

    chmod +x /tmp/whp-apply-migrations.sh
    /tmp/whp-apply-migrations.sh "$CURRENT_VERSION" "$NEW_VERSION" "$SCRIPT_DIR/sql/migrations"
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}Error applying migrations. Update aborted.${NC}"
        rm -f /tmp/whp-apply-migrations.sh
        exit 1
    fi
    
    rm -f /tmp/whp-apply-migrations.sh
fi

# Sync files (excluding sql/migrations since we already applied them)
echo ""
echo "Syncing files to /root/whp..."
rsync -av \
  --exclude='sql/migrations/' \
  --exclude='update.sh' \
  "$SCRIPT_DIR/" /root/whp/

# Sync web files
if [ -d "$SCRIPT_DIR/web-files" ]; then
    echo ""
    echo "Syncing web files..."
    mkdir -p /docker/whp/web
    rsync -av --delete "$SCRIPT_DIR/web-files/" /docker/whp/web/
    
    # Set correct permissions
    chown -R apache:apache /docker/whp/web/
    find /docker/whp/web -type d -exec chmod 755 {} \;
    find /docker/whp/web -type f -exec chmod 644 {} \;
fi

# Update version file in both locations
echo $NEW_VERSION > /etc/whp-version
echo $NEW_VERSION > /root/whp/VERSION

# Set executable permissions on scripts
chmod +x /root/whp/scripts/*.sh 2>/dev/null || true
chmod +x /root/whp/setup.sh 2>/dev/null || true
chmod +x /root/whp/update.sh 2>/dev/null || true

# Run any post-update scripts
if [ -f /root/whp/scripts/post-update.sh ]; then
    echo ""
    echo "Running post-update tasks..."
    /root/whp/scripts/post-update.sh "$CURRENT_VERSION" "$NEW_VERSION"
fi

# Services typically don't need restart for WHP updates
# If restart is needed, it should be done via post-update script

echo ""
echo -e "${GREEN}Update complete!${NC}"
echo "WHP has been updated from $CURRENT_VERSION to $NEW_VERSION"
echo ""
echo "Please verify that everything is working correctly."
echo "If you encounter any issues, check the logs:"
echo "  - Apache: /var/log/httpd/error_log"
echo "  - PHP: /var/log/php-fpm/www-error.log"