<?php
/**
 * Email Management Library for Web Hosting Panel
 * Handles email account CRUD operations, forwarding, and aliases
 */

require_once 'mailserver-client.php';

class emailmgmt {
    
    private $db;
    private $mail_client;
    
    public function __construct() {
        $this->db = $this->get_mysql_connection();
        
        // Initialize mail server client
        try {
            $this->mail_client = MailServerClient::createFromConfig();
        } catch (Exception $e) {
            // Log the error but don't fail the constructor
            error_log("Mail server client initialization failed: " . $e->getMessage());
            $this->mail_client = null;
        }
    }
    
    /**
     * Get MySQL connection
     */
    private function get_mysql_connection() {
        try {
            require_once('/docker/whp/web/libs/mysqlmgmt.php');
            $MySQLMgmt = new mysqlmgmt();
            $pdo = $MySQLMgmt->getMySQLConnection();
            
            if ($pdo) {
                $pdo->exec("USE whp");
                return $pdo;
            }
        } catch (Exception $e) {
            error_log("Email management database connection failed: " . $e->getMessage());
        }
        return null;
    }
    
    /**
     * Get all email accounts for a user (or all users if root)
     */
    public function getEmailAccounts($username = null) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            if ($username && $username !== 'root') {
                // Regular user - see their own email accounts (both active and inactive)
                $stmt = $this->db->prepare("
                    SELECT ea.*, d.domain_name as domain_display_name
                    FROM whp.email_accounts ea
                    LEFT JOIN whp.domains d ON ea.domain_name = d.domain_name
                    WHERE ea.username = ?
                    ORDER BY ea.active DESC, ea.email_address
                ");
                $stmt->execute([$username]);
            } else {
                // Root user - see all email accounts (both active and inactive)
                $stmt = $this->db->prepare("
                    SELECT ea.*, d.domain_name as domain_display_name
                    FROM whp.email_accounts ea
                    LEFT JOIN whp.domains d ON ea.domain_name = d.domain_name
                    ORDER BY ea.username, ea.active DESC, ea.email_address
                ");
                $stmt->execute();
            }
            
            $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return ['status' => '0', 'accounts' => $accounts];
            
        } catch (Exception $e) {
            error_log("Failed to get email accounts: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve email accounts'];
        }
    }
    
    /**
     * Get user's available domains for email creation
     */
    public function getUserDomains($username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            if ($username === 'root') {
                // Root can see all domains
                $stmt = $this->db->prepare("
                    SELECT domain_name, username, active
                    FROM whp.domains
                    WHERE active = 1
                    ORDER BY domain_name
                ");
                $stmt->execute();
            } else {
                // Regular users see only their own domains
                $stmt = $this->db->prepare("
                    SELECT domain_name, active
                    FROM whp.domains
                    WHERE username = ? AND active = 1
                    ORDER BY domain_name
                ");
                $stmt->execute([$username]);
            }
            
            $domains = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return ['status' => '0', 'domains' => $domains];
            
        } catch (Exception $e) {
            error_log("Failed to get user domains: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve domains'];
        }
    }
    
    /**
     * Get all domains from WHP database (for admin users)
     */
    public function getAllWhpDomains() {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            $stmt = $this->db->prepare("
                SELECT domain_name, username, active, created_at
                FROM whp.domains
                WHERE active = 1
                ORDER BY domain_name
            ");
            $stmt->execute();
            
            $domains = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return ['status' => '0', 'domains' => $domains];
            
        } catch (Exception $e) {
            error_log("Failed to get all WHP domains: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve domains'];
        }
    }
    
    /**
     * Check if user can create more email accounts
     */
    public function canCreateEmailAccount($username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Get user's email allowance
            $stmt = $this->db->prepare("
                SELECT max_email_accounts
                FROM whp.client_allowances
                WHERE username = ?
            ");
            $stmt->execute([$username]);
            $allowance = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$allowance) {
                return ['status' => '1', 'msg' => 'User allowance not found'];
            }
            
            // Count current email accounts
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count
                FROM whp.email_accounts
                WHERE username = ? AND active = 1
            ");
            $stmt->execute([$username]);
            $current = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $can_create = $current['count'] < $allowance['max_email_accounts'];
            $remaining = max(0, $allowance['max_email_accounts'] - $current['count']);
            
            return [
                'status' => '0',
                'can_create' => $can_create,
                'current_count' => $current['count'],
                'max_allowed' => $allowance['max_email_accounts'],
                'remaining' => $remaining
            ];
            
        } catch (Exception $e) {
            error_log("Failed to check email account creation: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to check email account limits'];
        }
    }
    
    /**
     * Check if user can create an email account with given quota without exceeding storage limit
     */
    public function checkEmailStorageQuota($username, $new_quota_mb) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Get user's email storage allowance
            $stmt = $this->db->prepare("
                SELECT max_email_storage_mb
                FROM whp.client_allowances
                WHERE username = ?
            ");
            $stmt->execute([$username]);
            $allowance = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$allowance) {
                return ['status' => '1', 'msg' => 'User allowance not found'];
            }
            
            // Get current email storage usage
            $stmt = $this->db->prepare("
                SELECT COALESCE(SUM(quota_mb), 0) as current_storage
                FROM whp.email_accounts
                WHERE username = ? AND active = 1
            ");
            $stmt->execute([$username]);
            $current = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $max_storage = $allowance['max_email_storage_mb'];
            $current_storage = $current['current_storage'];
            $new_total = $current_storage + $new_quota_mb;
            
            if ($new_total > $max_storage) {
                $available = max(0, $max_storage - $current_storage);
                return [
                    'status' => '1', 
                    'msg' => sprintf(
                        'Email storage quota exceeded. You are using %d MB of %d MB. This account requires %d MB, but only %d MB is available.',
                        $current_storage,
                        $max_storage,
                        $new_quota_mb,
                        $available
                    )
                ];
            }
            
            return [
                'status' => '0',
                'current_storage' => $current_storage,
                'max_storage' => $max_storage,
                'new_total' => $new_total,
                'remaining' => $max_storage - $new_total
            ];
            
        } catch (Exception $e) {
            error_log("Failed to check email storage quota: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to check email storage quota'];
        }
    }
    
    /**
     * Create a new email account
     */
    public function createEmailAccount($username, $email_address, $password, $domain_name, $quota_mb = 100) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Validate email format
            if (!filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
                return ['status' => '1', 'msg' => 'Invalid email address format'];
            }
            
            // Check if email already exists
            $stmt = $this->db->prepare("
                SELECT id FROM whp.email_accounts WHERE email_address = ?
            ");
            $stmt->execute([$email_address]);
            if ($stmt->fetch()) {
                return ['status' => '1', 'msg' => 'Email address already exists'];
            }
            
            // Check if user owns the domain
            $stmt = $this->db->prepare("
                SELECT id FROM whp.domains 
                WHERE domain_name = ? AND username = ? AND active = 1
            ");
            $stmt->execute([$domain_name, $username]);
            if (!$stmt->fetch()) {
                return ['status' => '1', 'msg' => 'Domain not found or not owned by user'];
            }
            
            // Check if user can create more email accounts
            $can_create = $this->canCreateEmailAccount($username);
            if ($can_create['status'] !== '0') {
                return $can_create;
            }
            if (!$can_create['can_create']) {
                return ['status' => '1', 'msg' => 'Email account limit reached'];
            }
            
            // Check email storage quota
            $storage_check = $this->checkEmailStorageQuota($username, $quota_mb);
            if ($storage_check['status'] !== '0') {
                return $storage_check;
            }
            
            // Hash password
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            // Create email account on mail server if client is available
            if ($this->mail_client) {
                try {
                    $username_part = explode('@', $email_address)[0];
                    $this->mail_client->createEmailAccount($email_address, $password, $username_part, $quota_mb);
                } catch (Exception $e) {
                    error_log("Failed to create email account on mail server: " . $e->getMessage());
                    return ['status' => '1', 'msg' => 'Failed to create email account on mail server: ' . $e->getMessage()];
                }
            }
            
            // Create email account in database
            $stmt = $this->db->prepare("
                INSERT INTO whp.email_accounts (email_address, username, domain_name, password_hash, quota_mb)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$email_address, $username, $domain_name, $password_hash, $quota_mb]);
            
            return ['status' => '0', 'msg' => 'Email account created successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to create email account: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to create email account'];
        }
    }
    
    /**
     * Update email account
     */
    public function updateEmailAccount($email_id, $username, $password = null, $quota_mb = null, $active = null) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Check if email account exists and user has permission
            $stmt = $this->db->prepare("
                SELECT id, username FROM whp.email_accounts WHERE id = ?
            ");
            $stmt->execute([$email_id]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$account) {
                return ['status' => '1', 'msg' => 'Email account not found'];
            }
            
            // Check permission (user can only edit their own accounts, root can edit all)
            if ($account['username'] !== $username && $username !== 'root') {
                return ['status' => '1', 'msg' => 'Permission denied'];
            }
            
            // Build update query
            $updates = [];
            $params = [];
            
            if ($password !== null) {
                $updates[] = "password_hash = ?";
                $params[] = password_hash($password, PASSWORD_DEFAULT);
            }
            
            if ($quota_mb !== null) {
                // Check storage quota for updates (excluding current account's quota)
                $stmt = $this->db->prepare("SELECT quota_mb FROM whp.email_accounts WHERE id = ?");
                $stmt->execute([$email_id]);
                $current_quota = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($current_quota) {
                    $quota_difference = $quota_mb - $current_quota['quota_mb'];
                    if ($quota_difference > 0) {
                        // Only check if increasing quota
                        $storage_check = $this->checkEmailStorageQuota($account['username'], $quota_difference);
                        if ($storage_check['status'] !== '0') {
                            return $storage_check;
                        }
                    }
                }
                
                $updates[] = "quota_mb = ?";
                $params[] = $quota_mb;
            }
            
            if ($active !== null) {
                $updates[] = "active = ?";
                $params[] = $active ? 1 : 0;
            }
            
            if (empty($updates)) {
                return ['status' => '1', 'msg' => 'No changes specified'];
            }
            
            // Get email address for mail server updates
            $stmt = $this->db->prepare("
                SELECT email_address FROM whp.email_accounts WHERE id = ?
            ");
            $stmt->execute([$email_id]);
            $email_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Update on mail server if client is available
            if ($this->mail_client && $email_data) {
                try {
                    $mail_updates = [];
                    
                    if ($password !== null) {
                        $mail_updates['passwordPlaintext'] = $password;
                    }
                    
                    if ($quota_mb !== null) {
                        // Update quota separately using the quota endpoint
                        $this->mail_client->updateEmailQuota($email_data['email_address'], $quota_mb);
                    }
                    
                    if (!empty($mail_updates)) {
                        $this->mail_client->patchEmailAccount($email_data['email_address'], $mail_updates);
                    }
                } catch (Exception $e) {
                    error_log("Failed to update email account on mail server: " . $e->getMessage());
                    // Continue with database update even if mail server fails
                }
            }
            
            $params[] = $email_id;
            $sql = "UPDATE whp.email_accounts SET " . implode(', ', $updates) . " WHERE id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            return ['status' => '0', 'msg' => 'Email account updated successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to update email account: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to update email account'];
        }
    }
    
    /**
     * Delete email account
     */
    public function deleteEmailAccount($email_id, $username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Check if email account exists and user has permission
            $stmt = $this->db->prepare("
                SELECT id, username FROM whp.email_accounts WHERE id = ?
            ");
            $stmt->execute([$email_id]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$account) {
                return ['status' => '1', 'msg' => 'Email account not found'];
            }
            
            // Check permission (user can only delete their own accounts, root can delete all)
            if ($account['username'] !== $username && $username !== 'root') {
                return ['status' => '1', 'msg' => 'Permission denied'];
            }
            
            // Get email address for mail server deletion
            $stmt = $this->db->prepare("
                SELECT email_address FROM whp.email_accounts WHERE id = ?
            ");
            $stmt->execute([$email_id]);
            $email_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete from mail server if client is available
            if ($this->mail_client && $email_data) {
                try {
                    $this->mail_client->deleteEmailAccount($email_data['email_address']);
                } catch (Exception $e) {
                    error_log("Failed to delete email account from mail server: " . $e->getMessage());
                    // Continue with database deletion even if mail server fails
                }
            }
            
            // Hard delete from database
            $stmt = $this->db->prepare("
                DELETE FROM whp.email_accounts WHERE id = ?
            ");
            $stmt->execute([$email_id]);
            
            return ['status' => '0', 'msg' => 'Email account deleted successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to delete email account: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to delete email account'];
        }
    }
    
    /**
     * Get email forwarders for a user
     */
    public function getEmailForwarders($username = null) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            if ($username && $username !== 'root') {
                $stmt = $this->db->prepare("
                    SELECT * FROM whp.email_forwarders 
                    WHERE username = ?
                    ORDER BY active DESC, source_email
                ");
                $stmt->execute([$username]);
            } else {
                $stmt = $this->db->prepare("
                    SELECT * FROM whp.email_forwarders 
                    ORDER BY username, active DESC, source_email
                ");
                $stmt->execute();
            }
            
            $forwarders = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return ['status' => '0', 'forwarders' => $forwarders];
            
        } catch (Exception $e) {
            error_log("Failed to get email forwarders: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve email forwarders'];
        }
    }
    
    /**
     * Create email forwarder
     */
    public function createEmailForwarder($username, $source_email, $destination_email) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Validate email formats
            if (!filter_var($source_email, FILTER_VALIDATE_EMAIL)) {
                return ['status' => '1', 'msg' => 'Invalid source email address format'];
            }
            
            if (!filter_var($destination_email, FILTER_VALIDATE_EMAIL)) {
                return ['status' => '1', 'msg' => 'Invalid destination email address format'];
            }
            
            // Check if forwarder already exists
            $stmt = $this->db->prepare("
                SELECT id FROM whp.email_forwarders 
                WHERE source_email = ? AND destination_email = ?
            ");
            $stmt->execute([$source_email, $destination_email]);
            if ($stmt->fetch()) {
                return ['status' => '1', 'msg' => 'Email forwarder already exists'];
            }
            
            // Verify that the source email domain belongs to the user
            $domain = explode('@', $source_email)[1];
            $stmt = $this->db->prepare("
                SELECT id FROM whp.domains 
                WHERE domain_name = ? AND username = ? AND active = 1
            ");
            $stmt->execute([$domain, $username]);
            if (!$stmt->fetch()) {
                return ['status' => '1', 'msg' => 'Source email domain not found or not owned by user'];
            }
            
            // Create forwarder on mail server if client is available
            if ($this->mail_client) {
                try {
                    $username_part = explode('@', $source_email)[0];
                    $this->mail_client->createEmailForwarder($source_email, $destination_email, $username_part);
                } catch (Exception $e) {
                    error_log("Failed to create email forwarder on mail server: " . $e->getMessage());
                    return ['status' => '1', 'msg' => 'Failed to create email forwarder on mail server: ' . $e->getMessage()];
                }
            }
            
            // Create forwarder in database
            $stmt = $this->db->prepare("
                INSERT INTO whp.email_forwarders (source_email, destination_email, username)
                VALUES (?, ?, ?)
            ");
            $stmt->execute([$source_email, $destination_email, $username]);
            
            return ['status' => '0', 'msg' => 'Email forwarder created successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to create email forwarder: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to create email forwarder'];
        }
    }
    
    /**
     * Delete email forwarder
     */
    public function deleteEmailForwarder($forwarder_id, $username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Check if forwarder exists and user has permission
            $stmt = $this->db->prepare("
                SELECT id, username, source_email FROM whp.email_forwarders WHERE id = ?
            ");
            $stmt->execute([$forwarder_id]);
            $forwarder = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$forwarder) {
                return ['status' => '1', 'msg' => 'Email forwarder not found'];
            }
            
            // Check permission
            if ($forwarder['username'] !== $username && $username !== 'root') {
                return ['status' => '1', 'msg' => 'Permission denied'];
            }
            
            // Delete from mail server if client is available
            if ($this->mail_client && $forwarder['source_email']) {
                try {
                    $this->mail_client->deleteEmailForwarder($forwarder['source_email']);
                } catch (Exception $e) {
                    error_log("Failed to delete email forwarder from mail server: " . $e->getMessage());
                    // Continue with database deletion even if mail server fails
                }
            }
            
            // Hard delete from database
            $stmt = $this->db->prepare("
                DELETE FROM whp.email_forwarders WHERE id = ?
            ");
            $stmt->execute([$forwarder_id]);
            
            return ['status' => '0', 'msg' => 'Email forwarder deleted successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to delete email forwarder: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to delete email forwarder'];
        }
    }
    
    /**
     * Get email aliases for a user
     */
    public function getEmailAliases($username = null) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            if ($username && $username !== 'root') {
                $stmt = $this->db->prepare("
                    SELECT * FROM whp.email_aliases 
                    WHERE username = ?
                    ORDER BY active DESC, alias_email
                ");
                $stmt->execute([$username]);
            } else {
                $stmt = $this->db->prepare("
                    SELECT * FROM whp.email_aliases 
                    ORDER BY username, active DESC, alias_email
                ");
                $stmt->execute();
            }
            
            $aliases = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return ['status' => '0', 'aliases' => $aliases];
            
        } catch (Exception $e) {
            error_log("Failed to get email aliases: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve email aliases'];
        }
    }
    
    /**
     * Create email alias
     */
    public function createEmailAlias($username, $alias_email, $target_email) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Validate email formats
            if (!filter_var($alias_email, FILTER_VALIDATE_EMAIL)) {
                return ['status' => '1', 'msg' => 'Invalid alias email address format'];
            }
            
            if (!filter_var($target_email, FILTER_VALIDATE_EMAIL)) {
                return ['status' => '1', 'msg' => 'Invalid target email address format'];
            }
            
            // Check if alias already exists
            $stmt = $this->db->prepare("
                SELECT id FROM whp.email_aliases WHERE alias_email = ?
            ");
            $stmt->execute([$alias_email]);
            if ($stmt->fetch()) {
                return ['status' => '1', 'msg' => 'Email alias already exists'];
            }
            
            // Create alias on mail server if client is available
            if ($this->mail_client) {
                try {
                    $this->mail_client->createEmailAlias($alias_email, $target_email);
                } catch (Exception $e) {
                    error_log("Failed to create email alias on mail server: " . $e->getMessage());
                    return ['status' => '1', 'msg' => 'Failed to create email alias on mail server: ' . $e->getMessage()];
                }
            }
            
            // Create alias in database
            $stmt = $this->db->prepare("
                INSERT INTO whp.email_aliases (alias_email, target_email, username)
                VALUES (?, ?, ?)
            ");
            $stmt->execute([$alias_email, $target_email, $username]);
            
            return ['status' => '0', 'msg' => 'Email alias created successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to create email alias: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to create email alias'];
        }
    }
    
    /**
     * Delete email alias
     */
    public function deleteEmailAlias($alias_id, $username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Check if alias exists and user has permission
            $stmt = $this->db->prepare("
                SELECT id, username FROM whp.email_aliases WHERE id = ?
            ");
            $stmt->execute([$alias_id]);
            $alias = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$alias) {
                return ['status' => '1', 'msg' => 'Email alias not found'];
            }
            
            // Check permission
            if ($alias['username'] !== $username && $username !== 'root') {
                return ['status' => '1', 'msg' => 'Permission denied'];
            }
            
            // Get alias email for mail server deletion
            $stmt = $this->db->prepare("
                SELECT alias_email FROM whp.email_aliases WHERE id = ?
            ");
            $stmt->execute([$alias_id]);
            $alias_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete from mail server if client is available
            if ($this->mail_client && $alias_data) {
                try {
                    $this->mail_client->deleteEmailAlias($alias_data['alias_email']);
                } catch (Exception $e) {
                    error_log("Failed to delete email alias from mail server: " . $e->getMessage());
                    // Continue with database deletion even if mail server fails
                }
            }
            
            // Hard delete from database
            $stmt = $this->db->prepare("
                DELETE FROM whp.email_aliases WHERE id = ?
            ");
            $stmt->execute([$alias_id]);
            
            return ['status' => '0', 'msg' => 'Email alias deleted successfully'];
            
        } catch (Exception $e) {
            error_log("Failed to delete email alias: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to delete email alias'];
        }
    }
    
    /**
     * Get email usage statistics for a user
     */
    public function getEmailUsageStats($username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            // Get user's email allowance
            $stmt = $this->db->prepare("
                SELECT max_email_accounts
                FROM whp.client_allowances
                WHERE username = ?
            ");
            $stmt->execute([$username]);
            $allowance = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$allowance) {
                return ['status' => '1', 'msg' => 'User allowance not found'];
            }
            
            // Count current email accounts
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count
                FROM whp.email_accounts
                WHERE username = ? AND active = 1
            ");
            $stmt->execute([$username]);
            $accounts = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Count forwarders
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count
                FROM whp.email_forwarders
                WHERE username = ? AND active = 1
            ");
            $stmt->execute([$username]);
            $forwarders = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Count aliases
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count
                FROM whp.email_aliases
                WHERE username = ? AND active = 1
            ");
            $stmt->execute([$username]);
            $aliases = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return [
                'status' => '0',
                'max_accounts' => $allowance['max_email_accounts'],
                'current_accounts' => $accounts['count'],
                'forwarders' => $forwarders['count'],
                'aliases' => $aliases['count'],
                'remaining' => max(0, $allowance['max_email_accounts'] - $accounts['count'])
            ];
            
        } catch (Exception $e) {
            error_log("Failed to get email usage stats: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to retrieve email usage statistics'];
        }
    }
    
    /**
     * Test mail server connection
     */
    public function testMailServerConnection() {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->testConnection();
            return $result;
        } catch (Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Create domain on mail server
     */
    public function createMailServerDomain($domain_name, $options = []) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->createDomain($domain_name, $options);
            return ['status' => '0', 'msg' => 'Domain created on mail server successfully'];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to create domain on mail server: ' . $e->getMessage()];
        }
    }
    
    /**
     * Delete domain from mail server
     */
    public function deleteMailServerDomain($domain_name) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->deleteDomain($domain_name);
            return ['status' => '0', 'msg' => 'Domain deleted from mail server successfully'];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to delete domain from mail server: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get domain information from mail server
     */
    public function getMailServerDomain($domain_name) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->getDomain($domain_name);
            return ['status' => '0', 'domain' => $result];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to get domain from mail server: ' . $e->getMessage()];
        }
    }
    
    /**
     * Update domain settings on mail server
     */
    public function updateMailServerDomain($domain_name, $updates = []) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->updateDomain($domain_name, $updates);
            return ['status' => '0', 'msg' => 'Domain updated on mail server successfully'];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to update domain on mail server: ' . $e->getMessage()];
        }
    }
    
    /**
     * Create DKIM records for a domain on mail server with retry logic
     */
    public function createDkimRecordsWithRetry($domain_name, $username = null, $max_retries = 3, $delay_seconds = 2) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        // Check domain ownership if username is provided
        if ($username && $username !== 'root') {
            $ownership_result = $this->checkDomainOwnership($domain_name, $username);
            if ($ownership_result['status'] !== '0') {
                return $ownership_result;
            }
        }
        
        // Try to create DKIM records with retries
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            try {
                // Add delay before first attempt and between retries
                if ($attempt > 1) {
                    sleep($delay_seconds);
                }
                
                $result = $this->mail_client->createDkimRecords($domain_name);
                
                if ($result && isset($result['selector']) && isset($result['public'])) {
                    return ['status' => '0', 'msg' => 'DKIM records created successfully', 'dkim' => $result];
                }
                
                // If we get here, the request succeeded but didn't return expected data
                return ['status' => '0', 'msg' => 'DKIM records created successfully', 'dkim' => $result];
                
            } catch (Exception $e) {
                $error_msg = $e->getMessage();
                
                // If this is the last attempt, return the error
                if ($attempt >= $max_retries) {
                    return ['status' => '1', 'msg' => "Failed to create DKIM records after $max_retries attempts: $error_msg"];
                }
                
                // Log the retry attempt
                error_log("DKIM creation attempt $attempt failed for domain $domain_name: $error_msg. Retrying in $delay_seconds seconds...");
            }
        }
        
        return ['status' => '1', 'msg' => 'Failed to create DKIM records after maximum retries'];
    }
    
    /**
     * Create DKIM records for a domain on mail server
     */
    public function createDkimRecords($domain_name, $username = null) {
        // Use the retry version by default
        return $this->createDkimRecordsWithRetry($domain_name, $username);
    }
    
    /**
     * Get DKIM records for a domain from mail server
     */
    public function getDkimRecords($domain_name, $username = null) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        // Check domain ownership if username is provided
        if ($username && $username !== 'root') {
            $ownership_result = $this->checkDomainOwnership($domain_name, $username);
            if ($ownership_result['status'] !== '0') {
                return $ownership_result;
            }
        }
        
        try {
            $result = $this->mail_client->getDkimRecords($domain_name);
            
            // Clean the public key for DNS record display
            if (isset($result['public'])) {
                $cleaned_public_key = str_replace([
                    '-----BEGIN PUBLIC KEY-----',
                    '-----END PUBLIC KEY-----',
                    "\n", "\r", " "
                ], '', $result['public']);
                
                // Add both raw and cleaned versions
                $result['public_cleaned'] = $cleaned_public_key;
                $result['dns_record'] = '"v=DKIM1; k=rsa; p=' . $cleaned_public_key . '"';
            }
            
            return ['status' => '0', 'dkim' => $result];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to get DKIM records: ' . $e->getMessage()];
        }
    }
    
    /**
     * Check if user owns a domain
     */
    private function checkDomainOwnership($domain_name, $username) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            $stmt = $this->db->prepare("SELECT username FROM whp.domains WHERE domain_name = ? AND active = 1");
            $stmt->execute([$domain_name]);
            $domain = $stmt->fetch();
            
            if (!$domain) {
                return ['status' => '1', 'msg' => 'Domain not found or inactive'];
            }
            
            if ($domain['username'] !== $username) {
                return ['status' => '1', 'msg' => 'You do not own this domain'];
            }
            
            return ['status' => '0', 'msg' => 'Domain ownership verified'];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to verify domain ownership: ' . $e->getMessage()];
                }
    }

    /**
     * Create DNS record for existing DKIM data
     * This method should be called from domains.php after DKIM records are created
     */
    public function createDkimDnsRecord($domain_name, $selector = null, $public_key = null, $username = null) {
        try {
            // If selector and public key not provided, get them from mail server
            if (!$selector || !$public_key) {
                $dkim_result = $this->getDkimRecords($domain_name, $username);
                if ($dkim_result['status'] !== '0') {
                    return $dkim_result;
                }
                
                if (!isset($dkim_result['dkim']['selector']) || !isset($dkim_result['dkim']['public'])) {
                    return ['status' => '1', 'msg' => 'DKIM records not found for domain'];
                }
                
                $selector = $dkim_result['dkim']['selector'];
                $public_key = $dkim_result['dkim']['public'];
            }
            
            // Load PowerDNS integration
            require_once('/docker/whp/web/libs/pdns-integration.php');
            
            // Initialize PowerDNS integration
            $pdns = new PowerDNSIntegration($this->db);
            
            // Create DKIM DNS record
            $record_name = $selector . '._domainkey';
            
            // Strip PEM headers and whitespace from public key
            $clean_public_key = str_replace([
                '-----BEGIN PUBLIC KEY-----',
                '-----END PUBLIC KEY-----',
                "\n", "\r", " "
            ], '', $public_key);
            
            $record_content = '"v=DKIM1; k=rsa; p=' . $clean_public_key . '"';
            
            // Log the attempt
            error_log("Attempting to add DKIM DNS record for $domain_name: $record_name TXT $record_content");
            
            $result = $pdns->addRecord($domain_name, $record_name, 'TXT', $record_content, 3600);
            
            if ($result) {
                error_log("Successfully added DKIM DNS record for $domain_name");
                return ['status' => '0', 'msg' => 'DKIM DNS record added successfully'];
            } else {
                error_log("Failed to add DKIM DNS record for $domain_name: PowerDNS returned false");
                return ['status' => '1', 'msg' => 'Failed to add DKIM DNS record to PowerDNS'];
            }
        } catch (Exception $e) {
            error_log("Exception adding DKIM DNS record for $domain_name: " . $e->getMessage());
            return ['status' => '1', 'msg' => 'Failed to add DKIM DNS record: ' . $e->getMessage()];
        }
    }
    
    /**
     * List all domains from mail server
     */
    public function listMailServerDomains() {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->listDomains();
            return ['status' => '0', 'domains' => $result];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to list domains from mail server: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get email account statistics from mail server
     */
    public function getEmailStats($email) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $result = $this->mail_client->getEmailStats($email);
            return ['status' => '0', 'stats' => $result];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to get email statistics: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get email statistics summary (last 30 days)
     */
    public function getEmailStatsSummary($email) {
        $stats_result = $this->getEmailStats($email);
        if ($stats_result['status'] !== '0') {
            return $stats_result;
        }
        
        $stats = $stats_result['stats'];
        $total_in = 0;
        $total_out = 0;
        $days_with_activity = 0;
        $last_activity = null;
        
        foreach ($stats as $date => $day_stats) {
            $in_count = $day_stats['in'] ?? 0;
            $out_count = $day_stats['out'] ?? 0;
            
            $total_in += $in_count;
            $total_out += $out_count;
            
            if ($in_count > 0 || $out_count > 0) {
                $days_with_activity++;
                $last_activity = $date;
            }
        }
        
        return [
            'status' => '0',
            'summary' => [
                'total_in' => $total_in,
                'total_out' => $total_out,
                'total_emails' => $total_in + $total_out,
                'days_with_activity' => $days_with_activity,
                'last_activity' => $last_activity,
                'avg_in_per_day' => count($stats) > 0 ? round($total_in / count($stats), 1) : 0,
                'avg_out_per_day' => count($stats) > 0 ? round($total_out / count($stats), 1) : 0
            ],
            'daily_stats' => $stats
        ];
    }
    
    /**
     * Get mail server statistics
     */
    public function getMailServerStats() {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        try {
            $stats = $this->mail_client->getStatistics();
            return ['status' => '0', 'stats' => $stats];
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Failed to get mail server statistics: ' . $e->getMessage()];
        }
    }
    
    /**
     * Clean up all email database entries for a domain when the domain is deleted
     * This should be called when a domain is being deleted to ensure no orphaned email records remain
     * 
     * @param string $domain_name The domain name to clean up
     * @return array Result array with status and cleanup information
     */
    public function cleanupDomainEmailEntries($domain_name) {
        if (!$this->db) {
            return ['status' => '1', 'msg' => 'Database connection failed'];
        }
        
        try {
            $cleanup_results = [
                'accounts_deleted' => 0,
                'forwarders_deleted' => 0,
                'aliases_deleted' => 0,
                'errors' => []
            ];
            
            // Delete email accounts for this domain
            $stmt = $this->db->prepare("DELETE FROM whp.email_accounts WHERE domain_name = ?");
            if ($stmt->execute([$domain_name])) {
                $cleanup_results['accounts_deleted'] = $stmt->rowCount();
            } else {
                $cleanup_results['errors'][] = 'Failed to delete email accounts';
            }
            
            // Delete email forwarders for this domain (source emails ending with @domain)
            $stmt = $this->db->prepare("DELETE FROM whp.email_forwarders WHERE source_email LIKE ?");
            if ($stmt->execute(['%@' . $domain_name])) {
                $cleanup_results['forwarders_deleted'] = $stmt->rowCount();
            } else {
                $cleanup_results['errors'][] = 'Failed to delete email forwarders';
            }
            
            // Delete email aliases for this domain (alias emails ending with @domain)
            $stmt = $this->db->prepare("DELETE FROM whp.email_aliases WHERE alias_email LIKE ?");
            if ($stmt->execute(['%@' . $domain_name])) {
                $cleanup_results['aliases_deleted'] = $stmt->rowCount();
            } else {
                $cleanup_results['errors'][] = 'Failed to delete email aliases';
            }
            
            // Log the cleanup results
            $total_deleted = $cleanup_results['accounts_deleted'] + $cleanup_results['forwarders_deleted'] + $cleanup_results['aliases_deleted'];
            if ($total_deleted > 0) {
                error_log("Email cleanup for domain {$domain_name}: {$cleanup_results['accounts_deleted']} accounts, {$cleanup_results['forwarders_deleted']} forwarders, {$cleanup_results['aliases_deleted']} aliases deleted");
            }
            
            if (empty($cleanup_results['errors'])) {
                return [
                    'status' => '0',
                    'msg' => "Email database cleanup completed for domain {$domain_name}",
                    'results' => $cleanup_results
                ];
            } else {
                return [
                    'status' => '1',
                    'msg' => 'Email database cleanup completed with errors: ' . implode(', ', $cleanup_results['errors']),
                    'results' => $cleanup_results
                ];
            }
            
        } catch (PDOException $e) {
            error_log("Email cleanup error for domain {$domain_name}: " . $e->getMessage());
            return [
                'status' => '1',
                'msg' => 'Email database cleanup failed: ' . $e->getMessage(),
                'results' => ['errors' => [$e->getMessage()]]
            ];
        }
    }

    /**
     * Manually retry DKIM creation for an existing domain
     * This is useful for domains that were created before the timing fix
     */
    public function retryDkimCreation($domain_name, $username = null, $max_retries = 5, $delay_seconds = 3) {
        if (!$this->mail_client) {
            return ['status' => '1', 'msg' => 'Mail server client not available'];
        }
        
        // Check domain ownership if username is provided
        if ($username && $username !== 'root') {
            $ownership_result = $this->checkDomainOwnership($domain_name, $username);
            if ($ownership_result['status'] !== '0') {
                return $ownership_result;
            }
        }
        
        // First, check if domain exists on mail server
        try {
            $domain_check = $this->mail_client->getDomain($domain_name);
            if (!$domain_check) {
                return ['status' => '1', 'msg' => 'Domain not found on mail server'];
            }
        } catch (Exception $e) {
            return ['status' => '1', 'msg' => 'Domain not found on mail server: ' . $e->getMessage()];
        }
        
        // Try to create DKIM records with extended retry logic
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            try {
                // Add delay before first attempt and between retries
                if ($attempt > 1) {
                    sleep($delay_seconds);
                }
                
                $result = $this->mail_client->createDkimRecords($domain_name);
                
                if ($result && isset($result['selector']) && isset($result['public'])) {
                    // Success! Now try to create the DNS record
                    $dns_result = $this->createDkimDnsRecord($domain_name, $result['selector'], $result['public'], $username);
                    if ($dns_result['status'] === '0') {
                        return ['status' => '0', 'msg' => 'DKIM records created and DNS record added successfully', 'dkim' => $result];
                    } else {
                        return ['status' => '0', 'msg' => 'DKIM records created but DNS record failed: ' . $dns_result['msg'], 'dkim' => $result];
                    }
                }
                
                // If we get here, the request succeeded but didn't return expected data
                return ['status' => '0', 'msg' => 'DKIM records created successfully', 'dkim' => $result];
                
            } catch (Exception $e) {
                $error_msg = $e->getMessage();
                
                // If this is the last attempt, return the error
                if ($attempt >= $max_retries) {
                    return ['status' => '1', 'msg' => "Failed to create DKIM records after $max_retries attempts: $error_msg"];
                }
                
                // Log the retry attempt
                error_log("DKIM retry attempt $attempt failed for domain $domain_name: $error_msg. Retrying in $delay_seconds seconds...");
            }
        }
        
        return ['status' => '1', 'msg' => 'Failed to create DKIM records after maximum retries'];
    }
}
?> 