<?php
/**
 * Site Creation API Endpoint
 * 
 * Handles AJAX requests for asynchronous site creation and status checking
 */

// Permission checking is handled by the permission system in auto-prepend.php
header('Content-Type: application/json');

require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('/docker/whp/web/libs/ContainerQueue.php');
require_once('/docker/whp/web/libs/docker-api.php');
require_once('/docker/whp/web/libs/pdns-integration.php');

// Verify CSRF token for security
if (!validate_csrf_token()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Request validation failed']);
    exit;
}

$is_root = (defined('AUTH_USER') && AUTH_USER === 'root');

// Initialize database connection
$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();

if (!$db) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database connection failed']);
    exit;
}

$containerQueue = new ContainerQueue($db);
$PowerDNS = new PowerDNSIntegration($db);

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'create_site':
        try {
            // Validate required fields
            if (empty($_POST['site_name']) || empty($_POST['primary_domain_id']) || empty($_POST['container_type_id'])) {
                echo json_encode(['success' => false, 'error' => 'Missing required fields']);
                exit;
            }

            $site_name = trim($_POST['site_name']);
            
            // Validate site name to prevent XSS and ensure valid format
            if (!preg_match('/^[a-zA-Z0-9\s\-_.]{1,50}$/', $site_name)) {
                echo json_encode(['success' => false, 'error' => 'Invalid site name format. Use only letters, numbers, spaces, hyphens, underscores, and periods (max 50 characters)']);
                exit;
            }
            
            // Additional sanitization for storage
            $site_name = htmlspecialchars($site_name, ENT_QUOTES, 'UTF-8');
            $primary_fqdn = trim($_POST['primary_domain_id']);
            
            // Validate FQDN format to prevent injection attacks
            if (!preg_match('/^[a-zA-Z0-9][a-zA-Z0-9.-]{0,254}$/', $primary_fqdn) || strlen($primary_fqdn) > 255) {
                echo json_encode(['success' => false, 'error' => 'Invalid domain format']);
                exit;
            }
            $container_type_id = intval($_POST['container_type_id']);
            $container_count = intval($_POST['container_count'] ?? 1);
            $cpu_per_container = floatval($_POST['cpu_per_container'] ?? 0.25);
            $memory_per_container = intval($_POST['memory_per_container'] ?? 256);
            
            // Validate container count
            if ($container_count < 1 || $container_count > 10) {
                echo json_encode(['success' => false, 'error' => 'Container count must be between 1 and 10']);
                exit;
            }
            
            // Get domain info and check ownership
            $primary_domain_row = null;
            $all_domains = $db->query("SELECT id, domain_name, username FROM whp.domains WHERE active=1")->fetchAll(PDO::FETCH_ASSOC);
            foreach ($all_domains as $row) {
                if ($primary_fqdn === $row['domain_name'] || (str_ends_with($primary_fqdn, '.' . $row['domain_name']))) {
                    $primary_domain_row = $row;
                    break;
                }
            }
            if (!$primary_domain_row) {
                echo json_encode(['success' => false, 'error' => 'Primary domain/subdomain not found or not owned by you']);
                exit;
            }
            if (!$is_root && $primary_domain_row['username'] !== AUTH_USER) {
                echo json_encode(['success' => false, 'error' => 'You can only create sites with your own domains/subdomains']);
                exit;
            }
            
            // Validate FQDN is a valid A or CNAME for this domain
            $valid_fqdn = false;
            $fqdn_records = $PowerDNS->getRecords($primary_domain_row['domain_name']);
            foreach ($fqdn_records as $rec) {
                if (strcasecmp($rec['name'], $primary_fqdn) === 0 && in_array(strtoupper($rec['type']), ['A','CNAME'])) {
                    $valid_fqdn = true;
                    break;
                }
            }
            if (!$valid_fqdn) {
                echo json_encode(['success' => false, 'error' => 'Selected primary FQDN is not a valid A or CNAME record']);
                exit;
            }
            
            // Get container type info
            $stmt = $db->prepare("SELECT * FROM whp.container_types WHERE id = ? AND active = 1");
            $stmt->execute([$container_type_id]);
            $container_type = $stmt->fetch();
            
            if (!$container_type) {
                echo json_encode(['success' => false, 'error' => 'Invalid or inactive container type']);
                exit;
            }
            
            // Validate resource requirements
            if ($cpu_per_container < $container_type['min_cpu']) {
                echo json_encode(['success' => false, 'error' => 'CPU per container must be at least ' . $container_type['min_cpu'] . ' cores']);
                exit;
            }
            
            if ($memory_per_container < $container_type['min_memory']) {
                echo json_encode(['success' => false, 'error' => 'Memory per container must be at least ' . $container_type['min_memory'] . ' MB']);
                exit;
            }

            // Begin transaction for site creation
            $db->beginTransaction();
            
            // Create site record
            $stmt = $db->prepare("
                INSERT INTO whp.sites (site_name, primary_domain_id, container_type_id, container_count, cpu_per_container, memory_per_container, username, ssl_enabled, auto_recreate_on_update, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $ssl_enabled = isset($_POST['enable_ssl']) && $_POST['enable_ssl'] == '1' ? 1 : 0;
            $auto_recreate = isset($_POST['auto_recreate_containers']) && $_POST['auto_recreate_containers'] == '1' ? 1 : 0;
            
            $stmt->execute([
                $site_name, 
                $primary_domain_row['id'], 
                $container_type_id, 
                $container_count, 
                $cpu_per_container, 
                $memory_per_container, 
                $primary_domain_row['username'], 
                $ssl_enabled, 
                $auto_recreate
            ]);
            
            $site_id = $db->lastInsertId();
            
            // Add primary domain to site_domains
            $stmt = $db->prepare("
                INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) 
                VALUES (?, ?, ?, 1)
            ");
            $stmt->execute([$site_id, $primary_domain_row['id'], $primary_fqdn]);
            
            // Handle additional domains
            $additional_domains_added = [];
            if (!empty($_POST['additional_domains'])) {
                $additional_domains = array_filter(array_map('trim', explode("\n", $_POST['additional_domains'])));
                
                foreach ($additional_domains as $fqdn) {
                    if (empty($fqdn)) continue;
                    
                    // Find parent domain for this FQDN
                    $parent_row = null;
                    foreach ($all_domains as $row) {
                        if ($fqdn === $row['domain_name'] || (str_ends_with($fqdn, '.' . $row['domain_name']))) {
                            $parent_row = $row;
                            break;
                        }
                    }
                    
                    if ($parent_row && ($is_root || $parent_row['username'] === AUTH_USER)) {
                        $stmt = $db->prepare("
                            INSERT INTO whp.site_domains (site_id, domain_id, fqdn, is_primary) 
                            VALUES (?, ?, ?, 0)
                        ");
                        $stmt->execute([$site_id, $parent_row['id'], $fqdn]);
                        $additional_domains_added[] = $fqdn;
                    }
                }
            }
            
            // Prepare configuration for queue
            $config = [
                'site_id' => $site_id,
                'site_name' => $site_name,
                'primary_fqdn' => $primary_fqdn,
                'primary_domain_row' => $primary_domain_row,
                'container_type_id' => $container_type_id,
                'container_type' => $container_type,
                'container_count' => $container_count,
                'cpu_per_container' => $cpu_per_container,
                'memory_per_container' => $memory_per_container,
                'enable_ssl' => $ssl_enabled,
                'additional_domains' => $additional_domains_added
            ];
            
            // Add to deployment queue
            $queue_id = $containerQueue->queueDeployment($site_id, $config);
            
            if (!$queue_id) {
                $db->rollBack();
                echo json_encode(['success' => false, 'error' => 'Failed to queue container deployment']);
                exit;
            }
            
            $db->commit();
            
            echo json_encode([
                'success' => true, 
                'site_id' => $site_id,
                'queue_id' => $queue_id,
                'message' => 'Site created successfully and queued for container deployment'
            ]);
            
        } catch (PDOException $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            error_log("Site creation database error: " . $e->getMessage());
            echo json_encode(['success' => false, 'error' => 'Database operation failed. Please try again later.']);
        } catch (Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            error_log("Site creation error: " . $e->getMessage());
            echo json_encode(['success' => false, 'error' => 'An error occurred. Please try again later.']);
        }
        break;
        
    case 'check_status':
        if (empty($_POST['site_id'])) {
            echo json_encode(['success' => false, 'error' => 'Site ID required']);
            exit;
        }
        
        $site_id = intval($_POST['site_id']);
        
        // Check site ownership
        $stmt = $db->prepare("SELECT username FROM whp.sites WHERE id = ?");
        $stmt->execute([$site_id]);
        $site = $stmt->fetch();
        
        if (!$site) {
            echo json_encode(['success' => false, 'error' => 'Site not found']);
            exit;
        }
        
        if (!$is_root && $site['username'] !== AUTH_USER) {
            echo json_encode(['success' => false, 'error' => 'Access denied']);
            exit;
        }
        
        $status = $containerQueue->getDeploymentStatus($site_id);
        
        if (!$status) {
            echo json_encode(['success' => false, 'error' => 'No deployment status found']);
            exit;
        }
        
        $response = [
            'success' => true,
            'status' => $status['status'],
            'created_at' => $status['created_at'],
            'processed_at' => $status['processed_at'],
            'error_message' => $status['error_message']
        ];
        
        // If completed, check if containers are actually running
        if ($status['status'] === 'completed') {
            $containers_running = $containerQueue->checkSiteContainersRunning($site_id);
            $response['containers_running'] = $containers_running;
        }
        
        echo json_encode($response);
        break;
        
    default:
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid action']);
        break;
}
?>