#!/usr/bin/env php
<?php
/**
 * Container Deployment Queue Processor
 * 
 * Processes queued container deployments in the background
 * Runs as a daemon or via cron to handle asynchronous site creation
 */

// Set proper working directory
chdir('/root/whp');

// Include required libraries
require_once('/docker/whp/web/libs/mysqlmgmt.php');
require_once('/docker/whp/web/libs/ContainerQueue.php');
require_once('/docker/whp/web/libs/docker-api.php');
require_once('/docker/whp/web/libs/haproxy_manager.php');
require_once('/docker/whp/web/libs/ssl_manager.php');

// Initialize logging
function log_message($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[{$timestamp}] [{$level}] {$message}\n";
    
    // Log to console if running interactively
    if (posix_isatty(STDOUT)) {
        echo $log_entry;
    }
    
    // Always log to file
    file_put_contents('/var/log/container-queue-processor.log', $log_entry, FILE_APPEND | LOCK_EX);
}

// Handle script termination gracefully
$running = true;
pcntl_signal(SIGTERM, function() use (&$running) {
    global $running;
    $running = false;
    log_message("Received SIGTERM, shutting down gracefully...");
});
pcntl_signal(SIGINT, function() use (&$running) {
    global $running;
    $running = false;
    log_message("Received SIGINT, shutting down gracefully...");
});

log_message("Container Queue Processor starting...");

// Initialize database connection
$MySQLMgmt = new mysqlmgmt();
$db = $MySQLMgmt->getMySQLConnection();

if (!$db) {
    log_message("Database connection failed", 'ERROR');
    exit(1);
}

$containerQueue = new ContainerQueue($db);
$DockerAPI = new docker_api();

// Main processing loop
while ($running) {
    pcntl_signal_dispatch(); // Handle signals
    
    try {
        // Get next pending deployment
        $deployment = $containerQueue->getNextPendingDeployment();
        
        if (!$deployment) {
            // No pending deployments, sleep and continue
            sleep(10);
            continue;
        }
        
        log_message("Processing deployment ID {$deployment['id']} for site ID {$deployment['site_id']}");
        
        // Already marked as processing in getNextPendingDeployment() to prevent race conditions
        
        // Decode configuration
        $config = json_decode($deployment['config'], true);
        if (!$config) {
            log_message("Invalid configuration for deployment ID {$deployment['id']}", 'ERROR');
            $containerQueue->markAsFailed($deployment['id'], 'Invalid configuration JSON');
            continue;
        }
        
        // Process the deployment
        $success = processDeployment($deployment['id'], $config, $DockerAPI, $containerQueue, $db);
        
        if ($success) {
            log_message("Successfully completed deployment ID {$deployment['id']}");
            $containerQueue->markAsCompleted($deployment['id']);
        } else {
            log_message("Failed to complete deployment ID {$deployment['id']}", 'ERROR');
            // markAsFailed is called within processDeployment with specific error message
        }
        
    } catch (Exception $e) {
        log_message("Unexpected error in main loop: " . $e->getMessage(), 'ERROR');
        sleep(30); // Wait longer on unexpected errors
    }
}

log_message("Container Queue Processor shutting down");

/**
 * Process a single deployment
 */
function processDeployment($queue_id, $config, $DockerAPI, $containerQueue, $db) {
    try {
        $site_id = $config['site_id'];
        $container_errors = [];
        
        log_message("Creating {$config['container_count']} container(s) for site: {$config['site_name']}");
        
        // Create and start Docker containers for the site
        $container_creation_success = true;
        
        // Get container type details for Docker creation
        $stmt = $db->prepare("
            SELECT ct.base_image, ct.startup_env, ct.mount_options, ct.listen_port 
            FROM whp.container_types ct 
            WHERE ct.id = ?
        ");
        $stmt->execute([$config['container_type_id']]);
        $container_type_info = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$container_type_info) {
            $containerQueue->markAsFailed($queue_id, 'Container type not found');
            return false;
        }
        
        // Create containers
        for ($i = 1; $i <= $config['container_count']; $i++) {
            // Sanitize container name to prevent command injection
            $container_name = preg_replace('/[^a-zA-Z0-9.-]/', '', $config['primary_fqdn']) . '-' . sprintf('%02d', $i);
            
            // Ensure it meets Docker naming requirements
            if (!preg_match('/^[a-zA-Z0-9][a-zA-Z0-9_.-]*$/', $container_name)) {
                throw new Exception("Invalid container name generated from domain: " . $config['primary_fqdn']);
            }
            
            $result = $DockerAPI->create_container([
                'name' => $container_name,
                'image' => $container_type_info['base_image'],
                'cpu' => $config['cpu_per_container'],
                'memory' => $config['memory_per_container'] . 'M',
                'user' => $config['primary_domain_row']['username'],
                'startup_env' => $container_type_info['startup_env'],
                'mount_options' => $container_type_info['mount_options'],
                'listen_port' => $container_type_info['listen_port']
            ]);
            
            // Log only essential information, not sensitive config
            log_message("Docker API result for container $i: success=" . ($result['success'] ? 'true' : 'false') . 
                       ", container_id=" . ($result['container_id'] ?? 'none'));
            
            if ($result['success']) {
                // Save container info to database
                $stmt = $db->prepare("
                    INSERT INTO whp.site_containers (site_id, container_id, container_number, status) 
                    VALUES (?, ?, ?, 'running')
                ");
                $stmt->execute([$site_id, $result['container_id'], $i]);
                
                log_message("Created container: {$container_name} (ID: {$result['container_id']})");
            } else {
                $container_creation_success = false;
                $error_msg = "Failed to create container $i: " . $result['error'];
                $container_errors[] = $error_msg;
                log_message($error_msg, 'ERROR');
            }
        }
        
        // Configure HAProxy if containers were created successfully
        if ($container_creation_success) {
            log_message("Configuring HAProxy for site ID: {$site_id}");
            
            $haproxy_manager = new haproxy_manager();
            $haproxy_result = $haproxy_manager->configureSite($site_id, [
                'username' => $config['primary_domain_row']['username'],
                'primary_domain' => $config['primary_domain_row']['domain_name'],
                'container_type_id' => $config['container_type_id']
            ]);
            
            if (!$haproxy_result['success']) {
                $container_creation_success = false;
                $error_msg = "HAProxy configuration failed: " . $haproxy_result['error'];
                $container_errors[] = $error_msg;
                log_message($error_msg, 'ERROR');
            } else {
                log_message("HAProxy configured successfully for site ID: {$site_id}");
            }
        }
        
        // Enable SSL if requested and containers/HAProxy succeeded
        if ($config['enable_ssl'] && $container_creation_success) {
            log_message("Enabling SSL for site ID: {$site_id}");
            
            $ssl_manager = new ssl_manager();
            $ssl_result = $ssl_manager->enable_ssl($site_id, 'letsencrypt');
            
            if (!$ssl_result['success']) {
                // SSL failures are not considered fatal - site is still usable
                log_message("SSL configuration failed: " . $ssl_result['error'], 'WARNING');
            } else {
                log_message("SSL enabled successfully for site ID: {$site_id}");
            }
        }
        
        if ($container_creation_success) {
            log_message("Deployment completed successfully for site: {$config['site_name']}");
            return true;
        } else {
            $error_message = "Container deployment failed: " . implode(', ', $container_errors);
            $containerQueue->markAsFailed($queue_id, $error_message);
            return false;
        }
        
    } catch (Exception $e) {
        $error_message = "Deployment exception: " . $e->getMessage();
        log_message($error_message, 'ERROR');
        $containerQueue->markAsFailed($queue_id, $error_message);
        return false;
    }
}
?>