-- Container Deployment Queue System
-- Implements asynchronous container deployment to prevent "site not found" errors
-- Addresses timing issues between site creation and container provisioning

USE whp;

CREATE TABLE IF NOT EXISTS container_deployment_queue (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    config JSON NOT NULL,
    status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
    error_message TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    processed_at TIMESTAMP NULL,
    retry_count INT DEFAULT 0,
    max_retries INT DEFAULT 3,
    INDEX idx_status (status),
    INDEX idx_site_id (site_id),
    INDEX idx_created_at (created_at),
    FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add queue_id column to sites table to track deployment status
ALTER TABLE sites ADD COLUMN IF NOT EXISTS queue_id INT NULL;

-- Add foreign key constraint (with check to prevent duplicate)
SET @constraint_exists = (
    SELECT COUNT(*) 
    FROM information_schema.TABLE_CONSTRAINTS 
    WHERE CONSTRAINT_SCHEMA = 'whp' 
    AND TABLE_NAME = 'sites' 
    AND CONSTRAINT_NAME = 'fk_sites_queue'
);

SET @sql = IF(@constraint_exists = 0, 
    'ALTER TABLE sites ADD CONSTRAINT fk_sites_queue FOREIGN KEY (queue_id) REFERENCES container_deployment_queue(id) ON DELETE SET NULL',
    'SELECT "Constraint fk_sites_queue already exists"'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;