<?php
/**
 * Container Deployment Queue Manager
 * 
 * Handles asynchronous container deployment to prevent timing issues
 * between site creation and container provisioning.
 */

class ContainerQueue {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Add a site to the deployment queue
     */
    public function queueDeployment($site_id, $config) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO container_deployment_queue (site_id, config, status) 
                VALUES (?, ?, 'pending')
            ");
            $stmt->execute([$site_id, json_encode($config)]);
            
            $queue_id = $this->pdo->lastInsertId();
            
            // Update sites table with queue_id
            $update_stmt = $this->pdo->prepare("UPDATE sites SET queue_id = ? WHERE id = ?");
            $update_stmt->execute([$queue_id, $site_id]);
            
            return $queue_id;
        } catch (Exception $e) {
            error_log("ContainerQueue::queueDeployment() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get deployment status for a site
     */
    public function getDeploymentStatus($site_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT q.*, s.domain_name 
                FROM container_deployment_queue q
                JOIN sites s ON q.site_id = s.id
                WHERE q.site_id = ? 
                ORDER BY q.created_at DESC 
                LIMIT 1
            ");
            $stmt->execute([$site_id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("ContainerQueue::getDeploymentStatus() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get next pending deployment from queue
     */
    public function getNextPendingDeployment() {
        try {
            $stmt = $this->pdo->prepare("
                SELECT * FROM container_deployment_queue 
                WHERE status = 'pending' 
                ORDER BY created_at ASC 
                LIMIT 1
            ");
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("ContainerQueue::getNextPendingDeployment() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Mark deployment as processing
     */
    public function markAsProcessing($queue_id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE container_deployment_queue 
                SET status = 'processing', processed_at = NOW() 
                WHERE id = ?
            ");
            return $stmt->execute([$queue_id]);
        } catch (Exception $e) {
            error_log("ContainerQueue::markAsProcessing() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Mark deployment as completed
     */
    public function markAsCompleted($queue_id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE container_deployment_queue 
                SET status = 'completed', processed_at = NOW() 
                WHERE id = ?
            ");
            return $stmt->execute([$queue_id]);
        } catch (Exception $e) {
            error_log("ContainerQueue::markAsCompleted() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Mark deployment as failed
     */
    public function markAsFailed($queue_id, $error_message = null) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE container_deployment_queue 
                SET status = 'failed', error_message = ?, processed_at = NOW(),
                    retry_count = retry_count + 1
                WHERE id = ?
            ");
            return $stmt->execute([$error_message, $queue_id]);
        } catch (Exception $e) {
            error_log("ContainerQueue::markAsFailed() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Retry a failed deployment if within retry limits
     */
    public function retryDeployment($queue_id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE container_deployment_queue 
                SET status = 'pending', error_message = NULL, processed_at = NULL
                WHERE id = ? AND retry_count < max_retries
            ");
            return $stmt->execute([$queue_id]);
        } catch (Exception $e) {
            error_log("ContainerQueue::retryDeployment() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get deployment statistics
     */
    public function getQueueStats() {
        try {
            $stmt = $this->pdo->prepare("
                SELECT 
                    status,
                    COUNT(*) as count
                FROM container_deployment_queue 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                GROUP BY status
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("ContainerQueue::getQueueStats() Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Clean up old completed/failed deployments
     */
    public function cleanupOldDeployments($days = 7) {
        try {
            $stmt = $this->pdo->prepare("
                DELETE FROM container_deployment_queue 
                WHERE status IN ('completed', 'failed') 
                AND created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            return $stmt->execute([$days]);
        } catch (Exception $e) {
            error_log("ContainerQueue::cleanupOldDeployments() Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if site has containers running (for status checking)
     */
    public function checkSiteContainersRunning($site_id) {
        try {
            // Get site info
            $stmt = $this->pdo->prepare("SELECT domain_name, user FROM sites WHERE id = ?");
            $stmt->execute([$site_id]);
            $site = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$site) {
                return false;
            }
            
            // Check if containers are running
            $container_name = $site['domain_name'] . '-01';
            $output = [];
            $return_var = 0;
            exec("docker ps --filter name={$container_name} --format '{{.Names}}' 2>/dev/null", $output, $return_var);
            
            return !empty($output) && in_array($container_name, $output);
        } catch (Exception $e) {
            error_log("ContainerQueue::checkSiteContainersRunning() Error: " . $e->getMessage());
            return false;
        }
    }
}
?>